# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core import common

from elisa.core.media_uri import MediaUri
from elisa.plugins.base.models.media import PlayableModel

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.poblesec.actions import PlayAllAction, ShuffleAction
from elisa.plugins.poblesec.actions import AddToFavoritesAction, AddToPlaylistAction
from elisa.plugins.poblesec.actions import ViewSlideshowAction

from elisa.plugins.favorites.models import FavoritesItem
from elisa.plugins.database.models import Image

from twisted.internet import defer

import datetime
import os.path


class ArtistPlayAllAction(PlayAllAction):

    def get_items(self, item):
        dfr = item.tracks.find()
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(lambda items: sorted(items, key=lambda a: a.file_path.lower()))
        return dfr

class AlbumPlayAllAction(PlayAllAction):

    def get_items(self, item):
        # hack to skip the dummy "All Tracks" album
        if hasattr(item, 'artist'):
            return defer.succeed([])

        dfr = item.tracks.find()
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(lambda items: sorted(items, key=lambda a: a.file_path.lower()))
        return dfr


class TrackPlayAllAction(PlayAllAction):

    def get_items(self, item):
        return defer.succeed([item])

    def run(self):
        track = self.controller.track
        if track is not None:
            playable_model = PlayableModel()
            playable_model.uri = MediaUri('file://' + track.file_path)
            playable_model.title = track.title
            # Enqueue the following tracks
            tracks = self.controller.tracks
            following_tracks = [playable_model]
            index = tracks.index(track)
            for ftrack in tracks[index+1:]:
                playable_model = PlayableModel()
                playable_model.uri = MediaUri('file://' + ftrack.file_path)
                playable_model.title = ftrack.title
                following_tracks.append(playable_model)
            self.play(following_tracks)
            dfr = defer.succeed(True)
        else:
            dfr = super(TrackPlayAllAction, self).run()

        return dfr


class ArtistShuffleAction(ArtistPlayAllAction, ShuffleAction):
    pass

class AlbumShuffleAction(AlbumPlayAllAction, ShuffleAction):
    pass

class TrackShuffleAction(TrackPlayAllAction, ShuffleAction):
    pass


class AlbumAddToFavoritesAction(AddToFavoritesAction):

    def run(self):
        if self.toggled:
            if self.controller.artist:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.controller.artist.name,
                                                    FavoritesItem.foreign_class == u'Artist')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
        else:
            favorite = FavoritesItem()
            favorite.added_time = datetime.datetime.now()

            # expecting an AlbumsController
            if self.controller.artist:
                favorite.foreign_id = self.controller.artist.name
                favorite.foreign_class = u'Artist'
                favorite.title = self.controller.artist.name
                self.store(favorite)

class TrackAddToFavoritesAction(AddToFavoritesAction):

    def run(self):
        if self.toggled:
            if self.controller.album:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.controller.album.name,
                                                    FavoritesItem.foreign_class == u'MusicAlbum')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
            elif self.controller.track:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.controller.track.file_path,
                                                    FavoritesItem.foreign_class == u'MusicTrack')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
        else:
            favorite = FavoritesItem()
            favorite.added_time = datetime.datetime.now()

            # expecting a TracksController
            if self.controller.album:
                favorite.foreign_id = self.controller.album.name
                favorite.foreign_class = u'MusicAlbum'
                favorite.title = self.controller.album.name
                self.store(favorite)
            elif self.controller.track:
                favorite.foreign_id = self.controller.track.file_path
                favorite.foreign_class = u'MusicTrack'
                favorite.title = self.controller.track.title
                self.store(favorite)

class PhotoAddToFavoritesAction(AddToFavoritesAction):

    def run(self):
        if self.toggled:
            if self.controller.album:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.controller.album.name,
                                                    FavoritesItem.foreign_class == u'PhotoAlbum')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
            elif self.controller.photo:
                dfr = common.application.store.find(FavoritesItem,
                                                    FavoritesItem.foreign_id == self.controller.photo.file_path,
                                                    FavoritesItem.foreign_class == u'Image')
                dfr.addCallback(lambda rs: rs.one())
                dfr.addCallback(self.remove)
        else:
            favorite = FavoritesItem()
            favorite.added_time = datetime.datetime.now()

            # expecting a TracksController
            if self.controller.album:
                favorite.foreign_id = self.controller.album.name
                favorite.foreign_class = u'PhotoAlbum'
                favorite.title = self.controller.album.name
                self.store(favorite)
            elif self.controller.photo:
                favorite.foreign_id = self.controller.photo.file_path
                favorite.foreign_class = u'Image'
                # FIXME: find a better title
                favorite.title = os.path.basename(self.controller.photo.file_path)
                self.store(favorite)


class ArtistAddToPlaylistAction(AddToPlaylistAction):

    def run(self):
        raise NotImplementedError()

class AlbumAddToPlaylistAction(AddToPlaylistAction):

    def run(self):
        raise NotImplementedError()

class TrackAddToPlaylistAction(AddToPlaylistAction):

    def run(self):
        raise NotImplementedError()


class ViewPhotoAlbumSlideshowAction(ViewSlideshowAction):

    def order_photos(self, rs):
        # copied and pasted the ordering from database.PhotosController
        rs.order_by(Image.shot_time, Image.file_path)
        return rs.all()

    def get_items(self, item):
        dfr = common.application.store.find(Image, Image.album_name == item.name)
        dfr.addCallback(self.order_photos)
        return dfr

