/*
 * Display Toggle for Linux
 * Copyright (C) 2008  Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * @file DisplayToggle.c
 *
 * Implementation of DisplayToggle
 */

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "DisplayToggle.h"
#include "DbgPrint.h"

const int DisplayToggle::DISPLAY_NONE = 0;
const int DisplayToggle::DISPLAY_INTERNAL = 1;
const int DisplayToggle::DISPLAY_EXTERNAL = 2;
const int DisplayToggle::DISPLAY_DUAL = 3;

const char *DisplayToggle::VGA = "VGA";
const char *DisplayToggle::LVDS = "LVDS";

const char *DisplayToggle::XRANDR_VGA =
	"xrandr --output LVDS --off --output VGA --auto";
const char *DisplayToggle::XRANDR_LVDS =
	"xrandr --output VGA --off --output LVDS --auto";
const char *DisplayToggle::XRANDR_DUAL =
	"xrandr --output LVDS --auto --output VGA --auto";

/**
 * An empty function for Xlib after function.
 */
int X11AfterFunction(Display *)
{
	return 0;
}

/**
 * Error handler to prevent application crash from X11 BadValue error
 */
int X11ErrorHandler(Display *, XErrorEvent *evn)
{
	DbgPrint("FnKeyMon had X11 BadValue Error!!!\n");
	DbgPrint("serial: %ld, error_code: %d, request_code: %d, minor_code: %d\n",
		evn->serial, evn->error_code, evn->request_code, evn->minor_code);
	return 0;
}

/**
 * Constructor of DisplayToggle.
 */
DisplayToggle::DisplayToggle() : m_pDisplay(NULL), m_pSR(NULL),
	m_pLvdsInfo(NULL), m_pVgaInfo(NULL)
{
	m_pDisplay = XOpenDisplay(NULL);
	assert(NULL != m_pDisplay);

	XSynchronize(m_pDisplay, True);

	/* to prevent the X11 BadValue error to crash the application */
	XSetAfterFunction(m_pDisplay, X11AfterFunction);
	XSetErrorHandler(X11ErrorHandler);

	m_RootWindow = DefaultRootWindow(m_pDisplay);
}

/**
 * Destructor of DisplayToggle.
 */
DisplayToggle::~DisplayToggle()
{
	if (NULL != m_pDisplay) {
		XCloseDisplay(m_pDisplay);
	}
}

/**
 * Get current display status.
 *
 * @return	the current display status:
 * 			DISPLAY_NONE if no display;
 * 			DISPLAY_INTERNAL if internal LCD display only;
 * 			DISPLAY_EXTERNAL if external VGA display only;
 * 			DISPLAY_DUAL if both
 */
int DisplayToggle::GetDisplayStatus()
{
	int status = DISPLAY_NONE;
	
	RenewInfo();

	if (m_pLvdsInfo->crtc) {
		status |= DISPLAY_INTERNAL; /* change to INTERNAL */
	}

	if (m_pVgaInfo->crtc) {
		status |= DISPLAY_EXTERNAL; /* change to EXERNAL or DUAL */
	}

	return status;
}

/**
 * Set new display status.
 *
 * @param desired	new display status
 * @param current	current display status
 * @return	the new display status:
 * 			DISPLAY_NONE if no display;
 * 			DISPLAY_INTERNAL if internal LCD display only;
 * 			DISPLAY_EXTERNAL if external VGA display only;
 * 			DISPLAY_DUAL if both
 */
int DisplayToggle::SetDisplayStatus(int desired, int current)
{
	if (current == desired) {
		return current; /* no need to change */
	}

	switch (desired) {
	case DISPLAY_INTERNAL:
		if (RR_Connected == m_pLvdsInfo->connection) {
			system(XRANDR_LVDS);
		}
		break;
			
	case DISPLAY_EXTERNAL:
		if (RR_Connected == m_pVgaInfo->connection) {
			system(XRANDR_VGA);
		}
		break;

	case DISPLAY_DUAL:
		system(XRANDR_DUAL);
		break;

	default:
		DbgPrint("error display status\n");
		assert(0);
	}

	return GetDisplayStatus();
}

/**
 * toggle between display status. The order is Internal > External >
 * Dual > Internal ...
 *
 * @return	the new display status:
 * 			DISPLAY_NONE if no display;
 * 			DISPLAY_INTERNAL if internal LCD display only;
 * 			DISPLAY_EXTERNAL if external VGA display only;
 * 			DISPLAY_DUAL if both
 */
int DisplayToggle::ToggleDisplay()
{
	int current;

	current = GetDisplayStatus();

	switch (current) {
	case DISPLAY_INTERNAL:
		return SetDisplayStatus(DISPLAY_EXTERNAL, current);

	case DISPLAY_EXTERNAL:
		return SetDisplayStatus(DISPLAY_DUAL, current);

	case DISPLAY_DUAL:
		return SetDisplayStatus(DISPLAY_INTERNAL, current);

	default:
		DbgPrint("error display status\n");
		assert(0);
		return current;
	}
}

/**
 * Reset output infomation.
 */
void DisplayToggle::RenewInfo()
{
	XRROutputInfo *outputInfo;
	int i;

	m_pLvdsInfo = m_pVgaInfo = NULL;

	m_pSR = XRRGetScreenResources(m_pDisplay, m_RootWindow);

	for (i = 0; i < m_pSR->noutput; ++i) {
		outputInfo = XRRGetOutputInfo(m_pDisplay, m_pSR, m_pSR->outputs[i]);
		if (0 == strncmp(LVDS, outputInfo->name, strlen(LVDS))) {
			m_pLvdsInfo = outputInfo;
		} else if (0 == strncmp(VGA, outputInfo->name, strlen(VGA))) {
			m_pVgaInfo = outputInfo;
		}
	}

	assert(m_pVgaInfo && m_pLvdsInfo);
}
