/****************************************************************************

 * Copyright (C) 2008  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License, version 2.1, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

File Name:          PMPluginApi.cpp

Description:

    This file implements class PMPlugin.

Environment (opt):

    OS: Ubuntu
    SE: Code::Block

Notes (opt):

  =====================================================================

  Revision   Revision History               Author     Date

  =====================================================================

  0.1        Create                         Deng Jing 2008-8-15

  =====================================================================

****************************************************************************/

#include <iostream>
#include <fstream>
#include "PMPluginAPi.h"
#include "PowerSchemeMgmt.h"
#include "PMBatteryMgmt.h"
#include "./inc/PMPlugin.h"
#include "PMDriverMgmt.h"

using namespace std;

#define LINK_GLINK 0x0000000a
#define ETHTOOL_GSET 0x00000001 /* Get settings. */
#define ETHTOOL_SSET 0x00000002 /* Set settings, privileged. */
#define BUFFERSIZE 1024

/* The forced speed, 10Mb, 100Mb, gigabit, 10GbE. */
#define SPEED_10 10
#define SPEED_100 100
#define SPEED_1000 1000
#define SPEED_10000 10000

/* Duplex, half or full. */
#define DUPLEX_HALF 0x00
#define DUPLEX_FULL 0x01

/* Enable or disable autonegotiation.  If this is set to enable,
* the forced link modes above are completely ignored.
*/
#define AUTONEG_DISABLE 0x00
#define AUTONEG_ENABLE 0x01

struct link_speed
{
    __uint32_t cmd;
    __uint32_t supported; /* Features this interface supports */
    __uint32_t advertising; /* Features this interface advertises */
    __uint16_t speed; /* The forced speed, 10Mb, 100Mb, gigabit */
    __uint8_t duplex; /* Duplex, half or full */
    __uint8_t port; /* Which connector port */
    __uint8_t phy_address;
    __uint8_t transceiver; /* Which tranceiver to use */
    __uint8_t autoneg; /* Enable or disable autonegotiation */
    __uint32_t maxtxpkt; /* Tx pkts before generating tx int */
    __uint32_t maxrxpkt; /* Rx pkts before generating rx int */
    __uint32_t reserved[4];
};

const char* COMMON_CONF_PATH = "/etc/ipower-management/config-file-bak/CommonSettings.xml";
const char* COMMON_SETTINGS = "globalsettings";
const char* COMMON_ITEMS = "items";
const char* COMMON_LCD = "LCDBrightness";
const char* COMMON_LCDMAX = "maxlevel";

//#define PM_DEBUG
extern FILE* pm_log;

PMPlugin::PMPlugin()
{
    m_pFuncs = NULL;
    GetLCDMaxBrightness();
}

PMPlugin::~PMPlugin()
{
    if (m_pFuncs != NULL)
    {
        m_pFuncs = NULL;
    }

//    fclose(pm1_log);
}

void PMPlugin::InitializeDCS_Funcs(DCS_Funcs* pFuncs)
{
    if (NULL != pFuncs)
    {
        m_pFuncs = pFuncs;
    }
}

/** Gets the LCD maximum brightness. */
DCSAPI DCS_Return_Code PMPlugin::PP_GetLCDMaxBrightness(int* bright)
{
    if (NULL == bright)
    {
        return DCS_FAIL_OPERATION;
    }

    *bright = m_MaxLCDBrightness;

    return DCS_SUCCESS;
}

DCSAPI DCS_Return_Code PMPlugin::GetLCDMaxBrightness()
{
    m_MaxLCDBrightness = 7;

    TiXmlDocument xmlDoc(COMMON_CONF_PATH);
    if ( !xmlDoc.LoadFile() )
    {
        return DCS_FAIL_OPERATION;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return DCS_FAIL_OPERATION;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_ITEMS);
	if (NULL == pNode)
	{
	    return DCS_FAIL_OPERATION;
	}
	xmlFirstSubElement = pNode->ToElement();

    pNode = xmlFirstSubElement->FirstChild(COMMON_LCD);
    if (NULL == pNode)
    {
        return DCS_FAIL_OPERATION;
    }
    xmlSubElement = pNode->ToElement();

    xmlSubElement->Attribute(COMMON_LCDMAX, &m_MaxLCDBrightness);

    return DCS_SUCCESS;
}

/** Gets the LCD brightness, bright is the percentage (0-100). */
DCSAPI DCS_Return_Code PMPlugin::PP_GetLCDBrightness(int* bright)
{
	*bright = PMDriverMgmt::Getinstance()->GetBrightness();
	if (*bright < 0)
	{
	    return DCS_REQUEST_DENIED;
	}

    return DCS_SUCCESS;
}

/** Sets the LCD brightness, bright is the percentage (0-100). */
DCSAPI DCS_Return_Code PMPlugin::PP_SetLCDBrightness(int bright)
{

    if (bright < 0 || m_MaxLCDBrightness < bright)
    {
        return DCS_FAIL_OPERATION;
    }

    if (PMDriverMgmt::Getinstance()->SetBrightness(bright) < 0)
    {
        return DCS_REQUEST_DENIED;
    }

    if (NULL != m_pFuncs)
    {
        m_pFuncs->dispatchEvent(DCS_POWER_MANAGER, PM_LCD_BRIGHTNESS_CHANGED, &bright, sizeof(int));
    }

    return DCS_SUCCESS;
}

/** Gets wireless status. */
DCSAPI DCS_Return_Code PMPlugin::PP_GetWirelessStatus(BOOL* pEnabled)
{
    *pEnabled = PMDriverMgmt::Getinstance()->GetWireless();
    if (*pEnabled < 0)
    {
        return DCS_REQUEST_DENIED;
    }

    return DCS_SUCCESS;
}

/** Turns on/off wireless. */
DCSAPI DCS_Return_Code PMPlugin::PP_SetWirelessStatus(BOOL enable)
{
     if ((enable != 0) && (enable != 1))
    {
        return DCS_FAIL_OPERATION;
    }

    BOOL oldstate = 0;
    DCS_Return_Code iRet  = PP_GetWirelessStatus(&oldstate);

    if (DCS_REQUEST_DENIED == iRet)
    {
        if (PMDriverMgmt::Getinstance()->SetWireless(enable) < 0)
        {
            return DCS_REQUEST_DENIED;
        }
    }
    else
    {
        if (oldstate == enable)
        {
            return DCS_SUCCESS;
        }

        if (PMDriverMgmt::Getinstance()->SetWireless(enable) < 0)
        {
            return DCS_REQUEST_DENIED;
        }
    }

    if (NULL != m_pFuncs)
    {
        m_pFuncs->dispatchEvent(DCS_POWER_MANAGER, PM_WLAN_CHANGED, &enable, sizeof(BOOL));
    }

    return DCS_SUCCESS;
}

/** Gets LAN status. */
DCSAPI DCS_Return_Code PMPlugin::PP_GetLANStatus(BOOL* pEnabled)
{

    *pEnabled = PMDriverMgmt::Getinstance()->GetLan();
    if (*pEnabled < 0)
    {
            return DCS_REQUEST_DENIED;
    }


    return DCS_SUCCESS;
}

/** Turns on/off LAN. */
DCSAPI DCS_Return_Code PMPlugin::PP_SetLANStatus(BOOL enable)
{
    if ((enable != 0) && (enable != 1))
    {
        return DCS_FAIL_OPERATION;
    }

	// add by dengjing on 2009/02/24
	if (enable == 0)
    {
        return DCS_NOT_SUPPORTED;
    }
    
    BOOL oldstate = 0;
    DCS_Return_Code iRet  = PP_GetLANStatus(&oldstate);

    if (DCS_REQUEST_DENIED == iRet)
    {
        if (PMDriverMgmt::Getinstance()->SetLan(enable) < 0)
        {
            return DCS_REQUEST_DENIED;
        }
    }
    else
    {
        if (oldstate == enable)
        {
            return DCS_SUCCESS;
        }
        if (PMDriverMgmt::Getinstance()->SetLan(enable) < 0)
        {
            return DCS_REQUEST_DENIED;
        }
    }

    if (NULL != m_pFuncs)
    {
        m_pFuncs->dispatchEvent(DCS_POWER_MANAGER, PM_LAN_CHANGED, &enable, sizeof(BOOL));
    }

    return DCS_SUCCESS;
}

/** Sets LAN speed. */
DCSAPI DCS_Return_Code PMPlugin::PP_SetLANSpeed(DCS_Lan_Speed speed)
{
    return DCS_NOT_SUPPORTED;
// the bug is defered: 100M->10M->100M , lan is disconnected. temporary solution is to defer
    int fd= socket(AF_INET, SOCK_DGRAM, 0);
    if (fd < 0)
    {
        perror("Cannot get control socket");
        return DCS_FAIL_OPERATION;
    }

    struct ifreq* pIfr = GetIfreq(fd);
    if(NULL == pIfr)
    {
        return DCS_FAIL_OPERATION;
    }

//    fprintf(pm1_log, "set speed of lan interface is: %s \n", pIfr->ifr_name);
//    fflush(pm1_log);

//    int autoneg_wanted = AUTONEG_ENABLE;

    struct link_speed linkSpeed;
    memset(&linkSpeed, 0, sizeof(linkSpeed));

	int duplexWanted = 0;
    switch (speed)
    {
        case DCS_LAN_SPEED_10M_HALF_DUPLEX:
            linkSpeed.speed = SPEED_10;
            duplexWanted = DUPLEX_HALF;
            break;
        case DCS_LAN_SPEED_10M_FULL_DUPLEX:
            linkSpeed.speed = SPEED_10;
            duplexWanted = DUPLEX_FULL;
            break;
        case DCS_LAN_SPEED_100M_HALF_DUPLEX:
            linkSpeed.speed = SPEED_100;
            duplexWanted = DUPLEX_HALF;
            break;
        case DCS_LAN_SPEED_100M_FULL_DUPLEX:
            linkSpeed.speed = SPEED_100;
            duplexWanted = DUPLEX_FULL;
            break;
        default:
            break;
    }

    // Change everything the user specified.
    //LinkSpeed.autoneg = autoneg_wanted;
	linkSpeed.duplex = duplexWanted;
    // Try to perform the update.
    linkSpeed.cmd = ETHTOOL_SSET;
    pIfr->ifr_data = (caddr_t)&linkSpeed;
    if (ioctl(fd, SIOCETHTOOL, pIfr) < 0)
    {
        perror("Cannot set new settings");
        return DCS_REQUEST_DENIED;
    }

    return DCS_SUCCESS;
}

/** Gets LAN speed. */
DCSAPI DCS_Return_Code PMPlugin::PP_GetLANSpeed(DCS_Lan_Speed* pSpeed)
{
    int socketfd = socket(AF_INET, SOCK_DGRAM, 0);
    if (socketfd < 0)
    {
        perror("Cannot get control socket");
        return DCS_FAIL_OPERATION;
    }

    struct ifreq* pIfr = GetIfreq(socketfd);
    if(NULL == pIfr)
    {
        return DCS_FAIL_OPERATION;
    }

//    fprintf(pm1_log, "get speed of lan interface is: %s \n", pIfr->ifr_name);
//    fflush(pm1_log);

   struct link_speed linkSpeed;
   memset(&linkSpeed, 0, sizeof(linkSpeed));

    linkSpeed.cmd = ETHTOOL_GSET;
    pIfr->ifr_data = (caddr_t)&linkSpeed;

    if (ioctl(socketfd, SIOCETHTOOL, pIfr) < 0)
    {
        perror("Cannot get current device settings");
        return DCS_REQUEST_DENIED;
    }

    int speed = linkSpeed.speed;
    int duplex = linkSpeed.duplex;

    * pSpeed = DCS_LAN_SPEED_100M_FULL_DUPLEX;

    if ( (SPEED_10 == speed) && (DUPLEX_HALF == duplex) )
    {
        * pSpeed = DCS_LAN_SPEED_10M_HALF_DUPLEX;
    }

    if ( (SPEED_10 == speed) && (DUPLEX_FULL == duplex) )
    {
        * pSpeed = DCS_LAN_SPEED_10M_FULL_DUPLEX;
    }

    if ( (SPEED_100 == speed) && (DUPLEX_HALF == duplex) )
    {
        * pSpeed = DCS_LAN_SPEED_100M_HALF_DUPLEX;
    }

    if ( (SPEED_100 == speed) && (DUPLEX_FULL == duplex) )
    {
        * pSpeed = DCS_LAN_SPEED_100M_FULL_DUPLEX;
    }

    return DCS_SUCCESS;
}

/**
 * Get the battery detail information by index.
 *
 * @param battery information
 * @param batteryIndex the index of the battery
 */
DCSAPI DCS_Return_Code PMPlugin::PP_GetBatteryInfo(PDCS_Battery_Info battery, int batteryIndex)
{

     if (NULL == battery)
    {
        return DCS_FAIL_OPERATION;
    }

    Battery_Detailed detailed;
    Battery_General general;

    int res = GetBatteriesInformation(&general);

    int res2 = GetBatteryInfoByDetail(batteryIndex, &detailed);


    if (detailed.present == 1 || res < 0 || res2 < 0)
    {
        battery->BatteryStatus = -1;
        battery->RemainPercent = -1;
        battery->RemainTime = -1;
        battery->DesignedCapacity = -1;
        battery->Voltage = -1;
        strncpy((char *)battery->Chemistry, "NULL", 4);
//        battery->DefaultAlert1 = -1;
//        battery->DefaultAlert2 = -1;
//        battery->Capabilities = -1;
        battery->FullChargedCapacity = -1;

        return DCS_FAIL_OPERATION;
    }

    battery->BatteryStatus = detailed.chargingState;
    battery->RemainPercent = general.remPerc;
    if (GetACAdapterStatus() == 2)
    {
        battery->RemainTime = general.remMins;
    }
    else
    {
        battery->RemainTime = general.remChargingMins;
    }
    battery->DesignedCapacity = detailed.designCapacity;
    battery->FullChargedCapacity = detailed.fullCapacity;
    strncpy((char *)battery->Chemistry, (char *)detailed.batteryType, 4);
//    battery->DefaultAlert1 = detailed.designCapacityWarning;
//    battery->DefaultAlert2 = detailed.designCapacityLow;
    battery->Voltage = detailed.presentVoltage;
//    battery->Capabilities = detailed.remainingCapacity;

    return DCS_SUCCESS;
}

/**
 * Get battery count.
 *
 * @param count of batteries
 */
DCSAPI DCS_Return_Code PMPlugin::PP_GetBatteryCount(ULONG * batCount)
{
    *batCount = GetDevicesNum(ACPI_BAT_DIR);

    return DCS_SUCCESS;
}

/**
 * Get the active power scheme.
 *
 * @param schemeId the ID of the active power scheme
 */
DCSAPI DCS_Return_Code PMPlugin::PP_GetActivePwrScheme(int *pSchemeId)
{
    PowerSchemeMgmt schemeObject(m_UserHomeDir);
    schemeObject.GetActiveSchemeId(*pSchemeId);

    return DCS_SUCCESS;
}

/**
 * Delete the power scheme.
 *
 * @param schemeId the ID of the power scheme to delete
 */
DCSAPI DCS_Return_Code PMPlugin::PP_DeletePwrScheme(int schemeId)
{
    PowerSchemeMgmt schemeObject(m_UserHomeDir);

    return schemeObject.DeletePowerScheme(schemeId);
}

/**
 * Applies a power scheme.
 *
 * @param schemeId	the ID of the power scheme to set active
 */
DCSAPI DCS_Return_Code PMPlugin::PP_SetActivePwrScheme(int schemeId)
{
//    fprintf(pm1_log, "Set active scheme in pmplugin 3 \n");
//    fflush(pm1_log);

    PowerSchemeMgmt schemeObject(m_UserHomeDir);

    return schemeObject.SetActivePowerScheme(schemeId);
}

/**
 * Adds a new power scheme.
 *
 * @param pName		the power scheme name
 * @param pDesc		the description of power scheme
 * @param copyFrom	the index of the power scheme copied
 * @param schemeId	pointer to the newly added power scheme ID
 */
DCSAPI DCS_Return_Code PMPlugin::PP_AddPwrScheme(char *pName, char *pDesc, int copyFrom, int *schemeId)
{
    *schemeId = (time(0)- 946656000); // Number 946656000 is from the date 2000/01/01/00:00;

    PowerSchemeMgmt schemeObject(m_UserHomeDir);
    return schemeObject.AddPowerScheme(pName, pDesc, copyFrom, *schemeId);
}

/**
 * Sets a variable value in a power scheme.
 *
 * @param schemeId	the power scheme ID
 * @param isAC      \c TRUE if AC; \c FALSE if DC
 * @param pVarName	the variable name
 * @param value		the variable value to be set
 */
DCSAPI DCS_Return_Code PMPlugin::PP_SetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int value)
{
     PowerSchemeMgmt schemeObject(m_UserHomeDir);

    return schemeObject.SetPowerVariable(schemeId, isAC, pVarName, value);
}

/**
 * Gets the value of the variable.
 *
 * @param schemeId	the power scheme ID
 * @param isAC      \c TRUE if AC; \c FALSE if DC
 * @param pVarName	the variable name
 * @param pValue	the pointer to the value got
 */
DCSAPI DCS_Return_Code PMPlugin::PP_GetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int *pValue)
{
    PowerSchemeMgmt schemeObject(m_UserHomeDir);

    return schemeObject.GetPowerVariable(schemeId, isAC, pVarName, pValue);
}

/**
 * Gets the name and description of the power scheme.
 *
 * @param schemeId	  the power scheme ID
 * @param schemeName  the power scheme name
 * @param schemeDesc  the power scheme description
 */
DCSAPI DCS_Return_Code PMPlugin::PP_GetPwrSchemeNameDesc(int schemeId, string &schemeName, string &schemeDesc)
{
    PowerSchemeMgmt schemeObject(m_UserHomeDir);

    if (DCS_FAIL_OPERATION == schemeObject.GetPowerSchemeNameDesc(schemeId, schemeName, schemeDesc))
    {
        return DCS_FAIL_OPERATION;
    }

    return DCS_SUCCESS;
}

ifreq* PMPlugin::GetIfreq(int socketfd)
{
    char buf[BUFFERSIZE] = {0};
    int number;
    struct ifreq* ifr;
    struct ifconf ifc;
    struct iwreq wrq;

    memset(&ifc, 0, sizeof(ifc));
    memset(&wrq, 0, sizeof(wrq));

    ifc.ifc_len = BUFFERSIZE;
    ifc.ifc_buf = buf;
    ioctl(socketfd, SIOCGIFCONF, &ifc);

    number = ifc.ifc_len / sizeof(struct ifreq);
    ifr = ifc.ifc_req;


    for (int j = 0; j < number; j++)
    {
        ioctl(socketfd, SIOCGIFFLAGS, ifr);
        string name = ifr->ifr_name;
        if (name.find(":") != string::npos || (0 == name.compare("wmaster0")))
        {
            continue;
        }

        if(((ifr->ifr_flags & IFF_LOOPBACK) == 0) && (ifr->ifr_flags & IFF_UP))
        {
            memcpy(wrq.ifr_name, name.c_str(), IFNAMSIZ);
            if (0 == ioctl(socketfd, SIOCGIWNAME, &wrq))
            {
                continue;
            }
            else
            {
                break;
            }
        }

        ifr++;
    }

     return ifr;
}

DCSAPI DCS_Return_Code PMPlugin::SetUserHomeDir(char * dirPath)
{
    m_UserHomeDir = dirPath;
    return DCS_SUCCESS;
}

DCSAPI DCS_Return_Code PMPlugin::PP_PostDownInterfaceReq(int networkType)
{
    if (NULL != m_pFuncs)
    {
        m_pFuncs->dispatchEvent(DCS_POWER_MANAGER, PM_DOWN_INTERFACE_REQ, &networkType, sizeof(int));
    }
}
