/*
 * Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
#include "ProfileEditor.h"
#include <stdlib.h>
#include <wx/filename.h>
#include <wx/log.h>
ProfileEditor::ProfileEditor(void)
{
	srand((unsigned int)time(NULL));
}

ProfileEditor::~ProfileEditor(void)
{
}

bool ProfileEditor::ReadProfile(ProfileList_t& profile, const wxString& fileName)
{
	if ( fileName.IsEmpty() || !wxFileName::FileExists(fileName))
    {
        return false;
    }
    wxLogNull log;
	ProfileList_t tp;
	wxFileName::SplitPath(fileName, NULL, NULL, &tp.sProfileName, NULL, NULL);
    wxXmlDocument doc;
    if ( !doc.Load(fileName) )
    {
        return false;
    }

	wxXmlNode* pRoot = doc.GetRoot();
	if ( pRoot->GetName() != _T("profile") )
		return false;
	if ( !pRoot->GetPropVal(_T("permission"), &tp.sPermission) )
		return false;
	if ( !pRoot->GetPropVal(_T("priority"), &tp.sPriority) )
		return false;
	if ( !pRoot->GetPropVal(_T("icon"), &tp.sProfileIcon) )
		return false;
	if ( !pRoot->GetPropVal(_T("show"), &tp.sShow) )
		return false;
    wxString strCompatible = _T("true");
	pRoot->GetPropVal(_T("compatible"), &strCompatible);

	bool result = true;
	wxXmlNode* pChild = pRoot->GetChildren();
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("GenernalSettings") )
		{
			result &= GetProfileGeneralSettings(tp.stGenernalSet, pChild);
		}
		else if ( pChild->GetName() == _T("WirelessSettings") )
		{
			result &= GetProfileWirelessSettings(tp.stWirelessSet, pChild);
		}
		else if ( pChild->GetName() == _T("OtherSettings") )
		{
			result &= GetProfileOtherSettings(tp.stOtherSet, pChild);
		}
		pChild = pChild->GetNext();
	}
	if ( result )
	{
		profile = tp;
		DecryptProfile(profile, strCompatible == _T("true"));
	}
	return profile.sProfileName.Length() <= 32;
}

bool ProfileEditor::WriteProfile(ProfileList_t& profile, const wxString& fileName, int priority)
{
	if ( priority > 0 )
	{
		profile.sPriority = wxString::Format(_T("%d"), priority);
	}
	wxLogNull log;
	ProfileList_t tp = profile;

	EncryptProfile(tp);

	wxXmlNode* pRoot = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("profile"));
	pRoot->AddProperty(_T("permission"), tp.sPermission);
	pRoot->AddProperty(_T("priority"), tp.sPriority);
	pRoot->AddProperty(_T("icon"), tp.sProfileIcon);
	pRoot->AddProperty(_T("show"), tp.sShow);
	pRoot->AddProperty(_T("compatible"), _T("false"));

	wxXmlNode* pNode = NULL;
	pNode = CreateNode(tp.stGenernalSet);
	if ( pNode == NULL )
	{
		delete pRoot;
		return false;
	}
	pRoot->AddChild(pNode);

	pNode = CreateNode(tp.stWirelessSet);
	if ( pNode == NULL )
	{
		delete pRoot;
		return false;
	}
	pRoot->AddChild(pNode);

	pNode = CreateNode(tp.stOtherSet);
	if ( pNode == NULL )
	{
		delete pRoot;
		return false;
	}
	pRoot->AddChild(pNode);

	wxXmlDocument doc;
	doc.SetVersion(_T("1.0"));
	doc.SetFileEncoding(_T("UTF-8"));
	doc.SetRoot(pRoot);
	return doc.Save(fileName, 2);
}

bool ProfileEditor::DeleteProfile(const wxString& fileName)
{
	return ::wxRemoveFile(fileName);
}

wxXmlNode* ProfileEditor::CreateNode(const GenernalSettings_t& st)
{
	wxXmlNode* pNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("GenernalSettings"));
	wxXmlNode* pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("ConnectType"));
	pSub->AddProperty(_T("value"), st.ConnecType);
	return pNode;
}

wxXmlNode* ProfileEditor::CreateNode(const WirelessSettings_t& st)
{
	wxXmlNode* pNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("WirelessSettings"));

	wxXmlNode* pSub = NULL;
	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("SSID"));
	pSub->AddProperty(_T("value"), st.SSID);

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("WirlessType"));
	pSub->AddProperty(_T("value"), st.WirelessType);

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("WlanSecurity"));
	pSub->AddProperty(_T("value"), st.WlanSecurity);

	wxXmlNode* p = NULL;
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("WEPSettings"));
	p->AddProperty(_T("value"), st.stWEPSecurity.value);
	p->AddProperty(_T("KeyIndex"), st.stWEPSecurity.keyIndex);

	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("WPASettings"));
	p->AddProperty(_T("value"), st.stWPASecurity.value);

	return pNode;
}

wxXmlNode* ProfileEditor::CreateNode(const OtherSettings_t& st)
{
	wxXmlNode* pNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("OtherSettings"));
	wxXmlNode* pSub = NULL;
	wxXmlNode* p = NULL;

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("NetworkSecurity"));
	pSub->AddProperty(_T("value"), st.NetworkSecurity);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("ForbidShareFileAndPrint"));
	p->AddProperty(_T("value"), st.stNetSecurity.sForbidShareFileAndPrint);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("ForbidShareInternet"));
	p->AddProperty(_T("value"), st.stNetSecurity.sForbidShareInternet);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("StartWindowFirewall"));
	p->AddProperty(_T("value"), st.stNetSecurity.sStartWindowFirewall);

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("AutoRunAppFlag"));
	pSub->AddProperty(_T("value"), st.AutoRunAppFlag);
	for ( vector<wxString>::size_type i = 0; i < st.AutoRunApp.size(); i++ )
	{
		p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("AutoRun"));
		p->AddProperty(_T("value"), st.AutoRunApp[i]);
	}

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("IPDNS"));
	pSub->AddProperty(_T("value"), st.IPDNSFlag);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("IPvalue"));
	p->AddProperty(_T("value"), st.stIPDNSSet.IPvalue);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("SubnetAddress"));
	p->AddProperty(_T("value"), st.stIPDNSSet.SubnetAddress);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("NetGateway"));
	p->AddProperty(_T("value"), st.stIPDNSSet.NetGateway);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("DNSvalue"));
	p->AddProperty(_T("value"), st.stIPDNSSet.DNSvalue);

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("HomepageFlag"));
	pSub->AddProperty(_T("value"), st.IsHomepage);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("Homepage"));
	p->AddProperty(_T("value"), st.Homepage);

	pSub = new wxXmlNode(pNode, wxXML_ELEMENT_NODE, _T("ProxyFlag"));
	pSub->AddProperty(_T("value"), st.ProxyFlag);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("Proxy"));
	p->AddProperty(_T("value"), st.stProxySet.Proxy);
	p = new wxXmlNode(pSub, wxXML_ELEMENT_NODE, _T("Port"));
	p->AddProperty(_T("value"), st.stProxySet.Port);

	return pNode;
}

bool ProfileEditor::GetProfileGeneralSettings(GenernalSettings_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("GenernalSettings") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("ConnectType") )
		{
			result = pChild->GetPropVal(_T("value"), &st.ConnecType);
			break;
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileWirelessSettings(WirelessSettings_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("WirelessSettings") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("SSID") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.SSID);
		}
		else if ( pChild->GetName() == _T("WirlessType") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.WirelessType);
		}
		else if ( pChild->GetName() == _T("WlanSecurity") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.WlanSecurity);
			wxXmlNode* pn = pChild->GetChildren();
			bool b = (pn != NULL);
			while ( pn != NULL )
			{
				if ( pn->GetName() == _T("WEPSettings") )
				{
					b &= pn->GetPropVal(_T("value"), &st.stWEPSecurity.value);
					b &= pn->GetPropVal(_T("KeyIndex"), &st.stWEPSecurity.keyIndex);
				}
				else if ( pn->GetName() == _T("WPASettings") )
				{
					b &= pn->GetPropVal(_T("value"), &st.stWPASecurity.value);
				}
				pn = pn->GetNext();
			}
			result &= b;
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileOtherSettings(OtherSettings_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("OtherSettings") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("NetworkSecurity") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.NetworkSecurity);
			result &= GetProfileNetworkSecurity(st.stNetSecurity, pChild);
		}
		else if ( pChild->GetName() == _T("AutoRunAppFlag") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.AutoRunAppFlag);
			result &= GetProfileAutoRunApp(st.AutoRunApp, pChild);
		}
		else if ( pChild->GetName() == _T("IPDNS") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.IPDNSFlag);
			result &= GetProfileIPDnsSettings(st.stIPDNSSet, pChild);
		}
		else if ( pChild->GetName() == _T("HomepageFlag") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.IsHomepage);
			wxXmlNode* pn = pChild->GetChildren();
			bool b = (pn != NULL);
			while ( pn != NULL )
			{
				if ( pn->GetName() == _T("Homepage") )
				{
					b &= pn->GetPropVal(_T("value"), &st.Homepage);
				}
				pn = pn->GetNext();
			}
			result &= b;
		}
		else if ( pChild->GetName() == _T("ProxyFlag") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.ProxyFlag);
			result &= GetProfileProxySettings(st.stProxySet, pChild);
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileNetworkSecurity(NetworkSecuritySet_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("NetworkSecurity") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("ForbidShareFileAndPrint") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.sForbidShareFileAndPrint);
		}
		else if ( pChild->GetName() == _T("ForbidShareInternet") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.sForbidShareInternet);
		}
		else if ( pChild->GetName() == _T("StartWindowFirewall") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.sStartWindowFirewall);
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileAutoRunApp(vector<wxString>& arr, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("AutoRunAppFlag") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = true;
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("AutoRun") )
		{
			wxString strAppName;
			result &= pChild->GetPropVal(_T("value"), &strAppName);
			arr.push_back(strAppName);
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileIPDnsSettings(IPDNSSet_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("IPDNS") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("IPvalue") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.IPvalue);
		}
		else if ( pChild->GetName() == _T("SubnetAddress") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.SubnetAddress);
		}
		else if ( pChild->GetName() == _T("NetGateway") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.NetGateway);
		}
		else if ( pChild->GetName() == _T("DNSvalue") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.DNSvalue);
		}
		pChild = pChild->GetNext();
	}
	return result;
}

bool ProfileEditor::GetProfileProxySettings(ProxySet_t& st, const wxXmlNode* pNode)
{
	if ( pNode == NULL || pNode->GetName() != _T("ProxyFlag") )
		return false;

	wxXmlNode* pChild = pNode->GetChildren();
	bool result = (pChild != NULL);
	while ( pChild != NULL )
	{
		if ( pChild->GetName() == _T("Proxy") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.Proxy);
		}
		else if ( pChild->GetName() == _T("Port") )
		{
			result &= pChild->GetPropVal(_T("value"), &st.Port);
		}
		pChild = pChild->GetNext();
	}
	return result;
}

wxString ProfileEditor::EncryptString(const wxString& str)
{
	size_t len = str.Length() + 1;
	unsigned short* buffer = new unsigned short[len];
	memset(buffer, 0, sizeof(unsigned short) * len);
	unsigned short mask = (unsigned short)rand();
	buffer[0] = mask;
	for ( size_t i = 0; i < str.Length(); i++ )
	{
		unsigned short c = (unsigned short)str[i];
		buffer[i + 1] = EncodeWChar(c, mask);
	}
	wxString result;
	for ( size_t i = 0; i < len; i++ )
	{
		result += wxString::Format(_T("%04X"), buffer[i]);
	}
	delete[] buffer;
	return result;
}

wxString ProfileEditor::DecryptString(const wxString& str, bool compatible)
{
    if ( compatible )
    {
        if ( str.Length() % 2 != 0 )
        {
            return wxEmptyString;
        }

        size_t len = str.Length() / 2 + 1;
        unsigned char* buffer = new unsigned char[len];
        memset(buffer, 0, sizeof(char) * len);

        for ( size_t i = 0; i < len; i++ )
        {
            unsigned char c = DecodeChar(str, 2 * i);
            buffer[i] = EncodeChar(c);
        }
        wxString result = wxString::FromUTF8((char*)buffer);
        delete[] buffer;
        return result;
    }
    else
    {
        if ( str.Length() % 4 != 0 )
        {
            return wxEmptyString;
        }

        size_t len = str.Length() / 4;
        wxChar* buffer = new wxChar[len];
        memset(buffer, 0, sizeof(wxChar) * len);
        unsigned short mask = DecodeWChar(str, 0);
        for ( size_t i = 1; i < len; i++ )
        {
            unsigned short c = DecodeWChar(str, 4 * i);
            buffer[i - 1] = (wxChar)EncodeWChar(c, mask);
        }
        wxString result(buffer);
        delete[] buffer;
        return result;
    }
}

void ProfileEditor::EncryptProfile(ProfileList_t& profile)
{
	profile.sPermission = EncryptString(profile.sPermission);
	profile.sPriority = EncryptString(profile.sPriority);
	profile.sProfileIcon = EncryptString(profile.sProfileIcon);
	profile.sShow = EncryptString(profile.sShow);

	profile.stGenernalSet.ConnecType = EncryptString(profile.stGenernalSet.ConnecType);
	profile.stWirelessSet.SSID = EncryptString(profile.stWirelessSet.SSID);
	profile.stWirelessSet.stWEPSecurity.keyIndex = EncryptString(profile.stWirelessSet.stWEPSecurity.keyIndex);
	profile.stWirelessSet.stWEPSecurity.value = EncryptString(profile.stWirelessSet.stWEPSecurity.value);
	profile.stWirelessSet.stWPASecurity.value = EncryptString(profile.stWirelessSet.stWPASecurity.value);
	profile.stWirelessSet.WirelessType = EncryptString(profile.stWirelessSet.WirelessType);
	profile.stWirelessSet.WlanSecurity = EncryptString(profile.stWirelessSet.WlanSecurity);

	profile.stOtherSet.NetworkSecurity = EncryptString(profile.stOtherSet.NetworkSecurity);
	profile.stOtherSet.stNetSecurity.sForbidShareFileAndPrint = EncryptString(profile.stOtherSet.stNetSecurity.sForbidShareFileAndPrint);
	profile.stOtherSet.stNetSecurity.sForbidShareInternet = EncryptString(profile.stOtherSet.stNetSecurity.sForbidShareInternet);
	profile.stOtherSet.stNetSecurity.sStartWindowFirewall = EncryptString(profile.stOtherSet.stNetSecurity.sStartWindowFirewall);

	profile.stOtherSet.AutoRunAppFlag = EncryptString(profile.stOtherSet.AutoRunAppFlag);
	for ( vector<wxString>::size_type i = 0; i < profile.stOtherSet.AutoRunApp.size(); i++ )
	{
		profile.stOtherSet.AutoRunApp[i] = EncryptString(profile.stOtherSet.AutoRunApp[i]);
	}

	profile.stOtherSet.IPDNSFlag = EncryptString(profile.stOtherSet.IPDNSFlag);
	profile.stOtherSet.stIPDNSSet.IPvalue = EncryptString(profile.stOtherSet.stIPDNSSet.IPvalue);
	profile.stOtherSet.stIPDNSSet.SubnetAddress = EncryptString(profile.stOtherSet.stIPDNSSet.SubnetAddress);
	profile.stOtherSet.stIPDNSSet.NetGateway = EncryptString(profile.stOtherSet.stIPDNSSet.NetGateway);
	profile.stOtherSet.stIPDNSSet.DNSvalue = EncryptString(profile.stOtherSet.stIPDNSSet.DNSvalue);

	profile.stOtherSet.IsHomepage = EncryptString(profile.stOtherSet.IsHomepage);
	profile.stOtherSet.Homepage = EncryptString(profile.stOtherSet.Homepage);
	profile.stOtherSet.ProxyFlag = EncryptString(profile.stOtherSet.ProxyFlag);
	profile.stOtherSet.stProxySet.Proxy = EncryptString(profile.stOtherSet.stProxySet.Proxy);
	profile.stOtherSet.stProxySet.Port = EncryptString(profile.stOtherSet.stProxySet.Port);
}

void ProfileEditor::DecryptProfile(ProfileList_t& profile, bool compatible)
{
	profile.sPermission = DecryptString(profile.sPermission, compatible);
	profile.sPriority = DecryptString(profile.sPriority, compatible);
	profile.sProfileIcon = DecryptString(profile.sProfileIcon, compatible);
	profile.sShow = DecryptString(profile.sShow, compatible);

	profile.stGenernalSet.ConnecType = DecryptString(profile.stGenernalSet.ConnecType, compatible);
	profile.stWirelessSet.SSID = DecryptString(profile.stWirelessSet.SSID, compatible);
	profile.stWirelessSet.stWEPSecurity.keyIndex = DecryptString(profile.stWirelessSet.stWEPSecurity.keyIndex, compatible);
	profile.stWirelessSet.stWEPSecurity.value = DecryptString(profile.stWirelessSet.stWEPSecurity.value, compatible);
	profile.stWirelessSet.stWPASecurity.value = DecryptString(profile.stWirelessSet.stWPASecurity.value, compatible);
	profile.stWirelessSet.WirelessType = DecryptString(profile.stWirelessSet.WirelessType, compatible);
	profile.stWirelessSet.WlanSecurity = DecryptString(profile.stWirelessSet.WlanSecurity, compatible);

	profile.stOtherSet.NetworkSecurity = DecryptString(profile.stOtherSet.NetworkSecurity, compatible);
	profile.stOtherSet.stNetSecurity.sForbidShareFileAndPrint = DecryptString(profile.stOtherSet.stNetSecurity.sForbidShareFileAndPrint, compatible);
	profile.stOtherSet.stNetSecurity.sForbidShareInternet = DecryptString(profile.stOtherSet.stNetSecurity.sForbidShareInternet, compatible);
	profile.stOtherSet.stNetSecurity.sStartWindowFirewall = DecryptString(profile.stOtherSet.stNetSecurity.sStartWindowFirewall, compatible);

	profile.stOtherSet.AutoRunAppFlag = DecryptString(profile.stOtherSet.AutoRunAppFlag, compatible);
	for ( vector<wxString>::size_type i = 0; i < profile.stOtherSet.AutoRunApp.size(); i++ )
	{
		profile.stOtherSet.AutoRunApp[i] = DecryptString(profile.stOtherSet.AutoRunApp[i], compatible);
	}

	profile.stOtherSet.IPDNSFlag = DecryptString(profile.stOtherSet.IPDNSFlag, compatible);
	profile.stOtherSet.stIPDNSSet.IPvalue = DecryptString(profile.stOtherSet.stIPDNSSet.IPvalue, compatible);
	profile.stOtherSet.stIPDNSSet.SubnetAddress = DecryptString(profile.stOtherSet.stIPDNSSet.SubnetAddress, compatible);
	profile.stOtherSet.stIPDNSSet.NetGateway = DecryptString(profile.stOtherSet.stIPDNSSet.NetGateway, compatible);
	profile.stOtherSet.stIPDNSSet.DNSvalue = DecryptString(profile.stOtherSet.stIPDNSSet.DNSvalue, compatible);

	profile.stOtherSet.IsHomepage = DecryptString(profile.stOtherSet.IsHomepage, compatible);
	profile.stOtherSet.Homepage = DecryptString(profile.stOtherSet.Homepage, compatible);
	profile.stOtherSet.ProxyFlag = DecryptString(profile.stOtherSet.ProxyFlag, compatible);
	profile.stOtherSet.stProxySet.Proxy = DecryptString(profile.stOtherSet.stProxySet.Proxy, compatible);
	profile.stOtherSet.stProxySet.Port = DecryptString(profile.stOtherSet.stProxySet.Port, compatible);
}

unsigned short ProfileEditor::EncodeWChar(const unsigned short number, const unsigned short mask)
{
	return number ^ mask;
}

unsigned char ProfileEditor::EncodeChar(const unsigned char c)
{
    unsigned int n = ((c >> 4) & 0x000F) | ((c << 4) & 0x00F0);
    return (unsigned char)n;
}

unsigned short ProfileEditor::DecodeWChar(const wxString& str, size_t index)
{
	if ( index > str.Length() - 4 )
	{
		return 0;
	}
	wxString strNumber = str.Mid(index, 4);
	unsigned long n = 0;
	strNumber.ToULong(&n, 16);
	return (unsigned short)n;
}

unsigned char ProfileEditor::DecodeChar(const wxString& str, size_t index)
{
	if ( index > str.Length() - 2 )
	{
		return 0;
	}
	wxString strNumber = str.Mid(index, 2);
	unsigned long n = 0;
	strNumber.ToULong(&n, 16);
	return (unsigned char)n;
}
