/*
 * Copyright (C) 2004-2008 Geometer Plus <contact@geometerplus.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include <algorithm>

#include "FBTextView.h"
#include "FBReader.h"
#include "FBReaderActions.h"
#include "../bookmodel/FBTextKind.h"

FBTextView::FBTextView(FBReader &reader, shared_ptr<ZLPaintContext> context) : FBView(reader, context) {
	myScrollingInProgress = false;
	myScrollingSpeed = 0;
}

bool FBTextView::_onStylusPress(int x, int y) {
	myPressedX = x;
	myPressedY = y;

	if (!isSelectionEnabled()) {
		myLatestUpdateY = y;
		myScrollingInProgress = true;
	}

	return false;
}

bool FBTextView::onStylusRelease(int x, int y) {
	myScrollingInProgress = false;

	if (FBView::onStylusRelease(x, y)) {
		return true;
	}

	if (myScrollingSpeed != 0) {
		myScrollingSpeed = 0;
		return true;
	}

	const int deltaX = x - myPressedX;
	if ((deltaX > 5) || (deltaX < -5)) {
		return false;
	}
	const int deltaY = y - myPressedY;
	if ((deltaY > 5) || (deltaY < -5)) {
		return false;
	}

	const ZLTextElementArea *area = elementByCoordinates(x, y);
	if (area != 0) {
		std::string id;
		bool isExternal;
		if (getHyperlinkId(*area, id, isExternal)) {
			fbreader().tryShowFootnoteView(id, isExternal);
			return true;
		}
		
		if (fbreader().isDictionarySupported() &&
				fbreader().EnableSingleClickDictionaryOption.value()) {
			const std::string txt = word(*area);
			if (!txt.empty()) {
				fbreader().openInDictionary(txt);
				return true;
			}
		}
	}

	if (!isSelectionEnabled()) {
		myLatestUpdateY = y;	
		myLatestUpdateTime = ZLTime();
		return true;
	}

	return false;
}

bool FBTextView::onStylusMove(int x, int y) {
	if (isSelectionEnabled()) {
		const ZLTextElementArea *area = elementByCoordinates(x, y);
		std::string id;
		bool isExternal;
		fbreader().setHyperlinkCursor((area != 0) && getHyperlinkId(*area, id, isExternal));
	} else {
		fbreader().setHyperlinkCursor(true);
	}
	return true;
}

bool FBTextView::onStylusMovePressed(int x, int y) {
	if (!isSelectionEnabled()) {
		if ((y <= 0) || (y >= context().height() - 1)) {
			return true;
		}
		if (!myScrollingInProgress) {
			myLatestUpdateY = y;
			myScrollingInProgress = true;
			return true;
		}
		int timeDiff = myLatestUpdateTime.millisecondsTo(ZLTime());
		int yDiff = y - myLatestUpdateY;
		if (timeDiff >= 100) {
			myScrollingSpeed = 0;
		} else if ((timeDiff > 0) && (abs(yDiff) >= 25)) {
			int fingerSpeed = 1000 * yDiff / timeDiff;
			if (abs(fingerSpeed) > 5000) {
				myScrollingSpeed += fingerSpeed / 1000;
				myScrollingInProgress = false;
				myLatestUpdateY = y;
			}
		}
		bool doRepaint = false;
		if (myScrollingSpeed > 0) {
			scrollPage(false, SCROLL_LINES, myScrollingSpeed--);
			doRepaint = true;
		} else if (myScrollingSpeed < 0) {
			scrollPage(true, SCROLL_LINES, -(myScrollingSpeed++));
			doRepaint = true;
		} else /*if (myScrollingSpeed == 0)*/ {
			int step = std::min(
				lineHeight(0),
				(size_t)(context().stringHeight() + context().descent() + 5)
			);
			if (step == 0) {
				step = 20;
			}
			if (y > myLatestUpdateY + step) {
				scrollPage(false, SCROLL_LINES, 1);
				myLatestUpdateY += step;
				doRepaint = true;
			} else if (y < myLatestUpdateY - step) {
				scrollPage(true, SCROLL_LINES, 1);
				myLatestUpdateY -= step;
				doRepaint = true;
			}
		}
		if (doRepaint) {
			myLatestUpdateTime = ZLTime();
			fbreader().refreshWindow();
		}
		return true;
	}
	return FBView::onStylusMovePressed(x, y);
}

bool FBTextView::getHyperlinkId(const ZLTextElementArea &area, std::string &id, bool &isExternal) const {
	if ((area.Kind != ZLTextElement::WORD_ELEMENT) &&
			(area.Kind != ZLTextElement::IMAGE_ELEMENT)) {
		return false;
	}
	ZLTextWordCursor cursor = startCursor();
	cursor.moveToParagraph(area.ParagraphIndex);
	cursor.moveToParagraphStart();
	ZLTextKind hyperlinkKind = REGULAR;
	for (int i = 0; i < area.ElementIndex; ++i) {
		const ZLTextElement &element = cursor.element();
		if (element.kind() == ZLTextElement::CONTROL_ELEMENT) {
			const ZLTextControlEntry &control = ((const ZLTextControlElement&)element).entry();
			if (control.isHyperlink()) {
				hyperlinkKind = control.kind();
				id = ((const ZLTextHyperlinkControlEntry&)control).label();
			} else if (!control.isStart() && (control.kind() == hyperlinkKind)) {
				hyperlinkKind = REGULAR;
			}
		}
		cursor.nextWord();
	}

	isExternal = hyperlinkKind == EXTERNAL_HYPERLINK;
	return hyperlinkKind != REGULAR;
}

bool FBTextView::isSelectionEnabled() const {
	return
		!fbreader().isFullscreen() &&
		(ZLStringOption(ZLCategoryKey::LOOK_AND_FEEL, "ToggleButtonGroup", "finger", "").value() == ActionCode::SELECT_TEXT);
}
