/****************************************************************************

 * Copyright (c) 2008-2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

File Name:          BatteryInfoMgmtProxy.cpp

Description:

    This file implements class BatteryInfoMgmtProxy.

Environment (opt):

    OS: Ubuntu
    SE: Code::Block

Notes (opt):

  =====================================================================

  Revision   Revision History               Author     Date

  =====================================================================

  0.1        Create                         Deng Jing 2008-7-20

  =====================================================================

****************************************************************************/

#include "include/BatteryInfoMgmtProxy.h"
#include "include/PMUtility.h"

#define ACPI_DIR "/proc/acpi"
#define ACPI_ACADAPTER_DIR "/proc/acpi/ac_adapter"


const unsigned int MAX_FILE_PATH_LENGTH = 512;
const unsigned int MAX_LINE_LENGTH = 1024;
const int MODULE_UNAVIA = -11;
const int DEVICE_UNAVIA = -12;
const int PMUNKOWN = -1;


enum tagEISACPIEnum
    {
        NOT_SUP = -1,
        APM_S,
        ACPI_S
    };
enum tagAC_STATEEum
    {
        AC_UNKNOWN,
        AC_ON,
        AC_OFF
    };

/**************************************************************************************************
* Function Name : PM_GetBatteriesInfo                                                             *
* Description  : Get battery info from throught DCS                                               *
* Date   : 08/07/20                                                                               *
* Parameter     :  batteryInfo  batteryIndex                                                      *
* Return Code  :   bool                                                                           *
* Author   :                                                                                      *
**************************************************************************************************/
bool BatteryInfoMgmtProxy::PM_GetBatteriesInfo(PDCS_Battery_Info batteryInfo, int& batteryIndex)
{

    unsigned long batteryCount = 0;
    DCS_GetBatteryCount(&batteryCount);

    if (batteryCount <= 0)
    {
        batteryIndex = -1;
        batteryInfo->BatteryStatus = -1;
        batteryInfo->RemainPercent = -1;
        batteryInfo->RemainTime = -1;
        batteryInfo->DesignedCapacity = -1;
        batteryInfo->Voltage = -1;
        strncpy((char *)batteryInfo->Chemistry, "NULL", 4);
        batteryInfo->FullChargedCapacity = -1;

        return false;
    }
    else
    {

        for (unsigned long i = 0; i < batteryCount; i++)
        {

            int retValue = DCS_GetBatteryInfo(batteryInfo, i);
            if ((retValue != DCS_FAIL_OPERATION) &&
                ((1 == batteryInfo->BatteryStatus) || (2 == batteryInfo->BatteryStatus)))
            {
                batteryIndex = i;
                return true;
            }
            memset(batteryInfo, 0, sizeof(DCS_Battery_Info));
        }

        for (unsigned long i = 0; i < batteryCount; i++)
        {
            int retValue = DCS_GetBatteryInfo(batteryInfo, i);
            if (retValue != DCS_FAIL_OPERATION)
            {
                batteryIndex = i;
                return true;
            }
        }


        batteryIndex = 0;
        return false;

    }


    int retValue = DCS_GetBatteryInfo(batteryInfo, 0);

    if (retValue == DCS_FAIL_OPERATION)
    {
        return false;
    }
    else
    {
        return true;
    }

}


/**************************************************************************************************
* Function Name : PM_GetBatteryInfo                                                               *
* Description  : Get battery info from throught DCS  with battery index                           *
* Date   : 08/07/20                                                                               *
* Parameter     :  batteryInfo  batteryIndex                                                      *
* Return Code  :   bool                                                                           *
* Author   :                                                                                      *
**************************************************************************************************/
bool BatteryInfoMgmtProxy::PM_GetBatteryInfo(PDCS_Battery_Info batteryInfo, int batteryIndex)
{
    int res = DCS_GetBatteryInfo(batteryInfo, batteryIndex);

    if (DCS_FAIL_OPERATION == res)
    {
        return false;
    }

    return true;

}


/**************************************************************************************************
* Function Name : PM_GetBatteriesCount                                                            *
* Description  : Get battery count from throught DCS                                              *
* Date   : 08/07/20                                                                               *
* Parameter     :  count                                                                          *
* Return Code  :   int                                                                            *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::PM_GetBatteriesCount(unsigned long& count)
{

    DCS_GetBatteryCount(&count);

    return 0;
}

/**************************************************************************************************
* Function Name : PM_GetACAdapterStatus                                                           *
* Description  : Check the AC adapter is plugined                                                 *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::PM_GetACAdapterStatus()
{
    if (GetACAdapterStatusOri() <= AC_ON)
    {
        return AC_ON;
    }
    else
    {
        return AC_OFF;
    }
}

static int g_isACPI = NOT_SUP;


/**************************************************************************************************
* Function Name : CheckACPIAVAI                                                                   *
* Description  : Check the ACPI is available                                                      *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::CheckACPIAVAI()
{
	if (g_isACPI != NOT_SUP)
	{
		return g_isACPI;
	}


	if (access(ACPI_DIR, R_OK))
	{
		g_isACPI = NOT_SUP;
	}
	else
	{
	    g_isACPI = ACPI_S;
	}

	return g_isACPI;

}

/**************************************************************************************************
* Function Name : CheckACPIDirExit                                                                *
* Description  : Check the ACPI directory is available                                            *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::CheckACPIDirExit(const char* dir)
{
    int iCount;
	DIR *acpi_dir;
	struct dirent *dirent;


	acpi_dir = opendir(dir);
	if (!acpi_dir)
	{
		return MODULE_UNAVIA;
	}

	for (iCount = 0; iCount < 3; iCount++)
	{
		dirent = readdir(acpi_dir);
		if (dirent <= 0)
		{
			closedir(acpi_dir);
			return DEVICE_UNAVIA;
		}
		if ('.' == dirent->d_name[0])
		{
			continue;
		}
	}

	closedir(acpi_dir);

	if (3 == iCount)
	{
		return 1;
	}
	else
	{
		return DEVICE_UNAVIA;
	}
}

/**************************************************************************************************
* Function Name : GetValueByColon                                                                 *
* Description  : parse the /proc/acpi/battery/ files and get values                               *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::GetValueByColon(FILE *bat_file, char *value, const size_t value_size,
                        char *key,  const size_t key_size)
{
    char *pRet = NULL;
	char line[MAX_LINE_LENGTH + 1] = "";
	unsigned int sLength = 0;
	unsigned int index = 0;
	unsigned int index_end = 0;
	unsigned int index_mid = 0;

	pRet = fgets(line, sizeof(line), bat_file);
	if (!pRet)
	{
		strcpy(value, "unknown");
		return -1;
	}

	sLength = strlen(line);
    /*skip space*/
	for (index = 0;
        (index < MAX_LINE_LENGTH) && (line[index] != ':') && (line[index] != '\0') && (line[index] != '\n');
        index++)
        {
        }


    if (index >= key_size)
	{
		strcpy(value, "unknown");
		return -1;
	}

	if ((index >= MAX_LINE_LENGTH) || (line[index] != ':'))
	{
		strcpy(value, "unknown");
		return -1;
	}


	if (isspace(line[0]) || index < 2)
	{
		return -1;
	}

	/*skip  all spaces */
	for (index_end = index - 1; (index_end >= 0) && isspace(line[index_end]); index_end--)
	{
	}

	index_end++;
	memcpy (key, line, index_end);

	*(key + index_end) = '\0';
	index++;

	/*skip all spaces*/
	for (index_mid = index;
        (index_mid < sLength) && (index_mid < MAX_LINE_LENGTH) && isspace(line[index_mid]);
        index_mid++)
        {
        }

	for (index = index_mid;
        (index < sLength) && (index < MAX_LINE_LENGTH) && (*(line + index) != '\n') && (*(line + index) != '\0');
        index++)
        {
        }

	/*skip spaces*/
	for (index_end = sLength-1; (index_end >= index) && isspace(line[index_end]) ; index_end--)
	{
	}

	index_end++;
	if (index_end - index_mid >= value_size)
	{
		strcpy(value, "unknown");
		return -1;
	}

	memcpy (value, line + index_mid, index_end - index_mid);

	*(value + (index_end - index_mid)) = '\0';
	return 1;
}

/**************************************************************************************************
* Function Name : GetDir                                                                          *
* Description  : scan directory and get the file path                                             *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::GetDir(const int number, char *file_dest, const size_t file_size, const char *dir)
{
    int dir_count = 0;
	DIR *dir_dest;
	struct dirent *dirent;

	dir_dest = opendir(dir);

	if (!dir_dest)
	{
		return -1;
	}

	while ((dirent = readdir(dir_dest)))
	{
		if ('.' == dirent->d_name[0])
		{
			continue;
		}
        if (number == dir_count)
        {
			break;
        }

		dir_count++;
	}

	if (!dirent || !file_dest || !dirent->d_name)
	{
		closedir(dir_dest);
		return -1;
	}

	strcpy(file_dest, dirent->d_name);
	closedir(dir_dest);
	return 1;
}


/**************************************************************************************************
* Function Name : GetACAdapterStatusOri                                                           *
* Description  : get ac adapter is is plugined                                                    *
* Date   : 08/07/20                                                                               *
* Parameter     :                                                                                 *
* Return Code  :                                                                                  *
* Author   :                                                                                      *
**************************************************************************************************/
int BatteryInfoMgmtProxy::GetACAdapterStatusOri()
{
    int iRet = 0;
	int testRet = 0;

	FILE *acFile = NULL;

	char ac_state_file[MAX_FILE_PATH_LENGTH + 1] = "";
 	char ac_file[MAX_FILE_PATH_LENGTH + 1] = "";

    char value[MAX_LINE_LENGTH + 1] = "";
 	char temp_str[MAX_LINE_LENGTH + 1] = "";


	testRet = CheckACPIAVAI();

	if (ACPI_S == testRet)
	{
		iRet = CheckACPIDirExit(ACPI_ACADAPTER_DIR);

		if (iRet < 0)
		{
            return iRet;
		}

		if (GetDir(0, ac_file, sizeof (ac_file), ACPI_ACADAPTER_DIR) < 0)
		{
			return DEVICE_UNAVIA;
		}

		snprintf(ac_state_file, MAX_FILE_PATH_LENGTH, "%s/%s/state", ACPI_ACADAPTER_DIR, ac_file);
		acFile = fopen(ac_state_file, "r");
		if (!acFile)
		{
			return AC_UNKNOWN;
		}

		if (GetValueByColon(acFile, value, sizeof (value), temp_str, sizeof (temp_str)) < 0)
		{
			iRet = AC_UNKNOWN;
		}
		else
		{
		    if (!strcmp(value, "off-line"))
			{
				iRet = AC_OFF;
			}
			else if (!strcmp(value, "on-line"))
			{
				iRet = AC_ON;
			}

			else
			{
				iRet = AC_UNKNOWN;
			}
		}

		fclose(acFile);
	}
	else
	{
		return AC_UNKNOWN;
	}
	return iRet;
}
