/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "launcher-iconview.h"

#include <string.h>
#include <tidy/tidy.h>

#include "launcher-app.h"
#include "launcher-defines.h"
#include "launcher-icon.h"
#include "launcher-util.h"

#include "tidy-texture-frame.h"

G_DEFINE_TYPE (LauncherIconview, launcher_iconview, TIDY_TYPE_VIEWPORT);

#define LAUNCHER_ICONVIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ICONVIEW, \
  LauncherIconviewPrivate))

struct _LauncherIconviewPrivate
{
  ClutterActor *scroll;
};


static gboolean
on_scroll_event (ClutterActor       *actor,
                 ClutterScrollEvent *event,
                 TidyAdjustment     *adjust)
{
#define VALUE 50.0
  gdouble diff = VALUE;
  gdouble lower, upper;

  if (event->direction == CLUTTER_SCROLL_UP)
    diff = -VALUE;

  diff += tidy_adjustment_get_value (adjust)+diff;

  tidy_adjustment_get_values (adjust, NULL, &lower, &upper, NULL, NULL, NULL);

  diff = CLAMP (diff, lower, upper);

  tidy_adjustment_set_value (adjust, diff);

  return TRUE;
}                  

void           
launcher_iconview_set_category (LauncherIconview     *view,
                                LauncherMenuCategory *cat)
{
  LauncherIconviewPrivate *priv;
  ClutterActor *scroll, *tex, *frame;
  ClutterActor *stage = clutter_stage_get_default ();
  TidyAdjustment *adjustment;
  gdouble lower, upper;
  GList *l;
  gint xoffset, yoffset, i = 0, rows = 0;
  gint app_cols = 0, app_rows = 0, app_count = 0;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  launcher_iconview_get_grid (&app_cols, &app_rows);
  app_count = app_cols *app_rows;
  
  xoffset =0;
  yoffset = ICONVIEW_PADDING;

  for (l = launcher_menu_category_get_applications (cat); l; l = l->next)
  {
    LauncherMenuApplication *app = l->data;
    ClutterActor *actor;
    
    actor = launcher_icon_new ();
    launcher_icon_set_application (LAUNCHER_ICON (actor), app);
    clutter_container_add_actor (CLUTTER_CONTAINER (view), actor);
    clutter_actor_set_position (actor, xoffset, yoffset);
    clutter_actor_show (actor);
        
    if (i == (app_cols -1))
    {
      i = 0;
      xoffset = 0;
      yoffset += ICON_HEIGHT () + ICONVIEW_PADDING;
      rows++;
    }
    else
    {
      xoffset += ICON_WIDTH() + ICONVIEW_PADDING;
      i++;
    }

  }

  clutter_actor_set_height (CLUTTER_ACTOR (view), (rows + 1) * ICON_HEIGHT());
  clutter_actor_set_clip (CLUTTER_ACTOR (view), 0, 0,
                          ICONVIEW_WIDTH(), ICONVIEW_HEIGHT());

  /* Add the scroll bar */
  if (g_list_length (launcher_menu_category_get_applications (cat)) > app_count)
  {
    tidy_scrollable_get_adjustments (TIDY_SCROLLABLE (view), NULL, &adjustment);
    tidy_adjustment_get_values (adjustment, NULL, &lower, &upper, 
                                NULL, NULL, NULL);

    tex = launcher_util_texture_new_from_file (PKGDATADIR"/scrollbar.svg");
    clutter_container_add_actor (CLUTTER_CONTAINER (view),
                                 tex);
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 5, 5, 5, 5);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment); 
    scroll = priv->scroll = tidy_scroll_bar_new_with_handle (adjustment, frame);
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 5, 5, 5, 5);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
    clutter_actor_set_reactive (CLUTTER_ACTOR (scroll), TRUE);
    g_signal_connect (scroll, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);     
    clutter_actor_set_opacity (frame, 100);
    tidy_frame_set_texture (TIDY_FRAME (scroll), frame);

    g_object_set (adjustment, 
      "upper", ((rows+2)*(ICON_HEIGHT()+ICONVIEW_PADDING)) + ICONVIEW_PADDING,
                  NULL);
    clutter_container_add_actor (CLUTTER_CONTAINER (stage), scroll);
    clutter_actor_set_size (scroll, ICONVIEW_HEIGHT()*0.97, 14);
    clutter_actor_set_anchor_point_from_gravity (scroll, 
                                                 CLUTTER_GRAVITY_CENTER);
    clutter_actor_set_position (scroll, 
                                clutter_actor_get_x (CLUTTER_ACTOR (view)) +
                                CAW(CLUTTER_ACTOR (view)) + ICONVIEW_PADDING+4,
                                (CSH()/2)+(PANEL_HEIGHT/2)+2);  
  
    clutter_actor_set_rotation (scroll, CLUTTER_Z_AXIS, 90, 0, 0, 0);
    clutter_actor_show (scroll);

    clutter_actor_set_reactive (CLUTTER_ACTOR (view), TRUE);
    g_signal_connect (view, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
  }

}

void   
launcher_iconview_get_grid     (gint *cols, gint *rows)
{
  gint sheight= CSH();
  static gint grid_cols = 4;
  static gint grid_rows = 4;

  g_return_if_fail (cols);
  g_return_if_fail (rows);

  if (grid_rows)
  {
  }
  if (sheight < 600)
  {
    grid_rows = 3;
  }
  else if (sheight < 800)
  {
    grid_rows = 4;
  }
  else
  {
    grid_rows = 5;
  }

  *cols = grid_cols;
  *rows = grid_rows;
}

/* GObject stuff */
static void
launcher_iconview_finalize (GObject *object)
{
  if (LAUNCHER_IS_ICONVIEW (object) 
        && CLUTTER_IS_ACTOR (LAUNCHER_ICONVIEW (object)->priv->scroll))
  {
    clutter_actor_destroy (LAUNCHER_ICONVIEW (object)->priv->scroll);
  }

  G_OBJECT_CLASS (launcher_iconview_parent_class)->finalize (object);
}

static void
launcher_iconview_class_init (LauncherIconviewClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_iconview_finalize;

  g_type_class_add_private (obj_class, sizeof (LauncherIconviewPrivate));
}

      
static void
launcher_iconview_init (LauncherIconview *iconview)
{
  LauncherIconviewPrivate *priv;
  priv = iconview->priv = LAUNCHER_ICONVIEW_GET_PRIVATE (iconview);
}

ClutterActor *
launcher_iconview_new (void)

{
  ClutterActor *iconview = NULL;

  iconview = g_object_new (LAUNCHER_TYPE_ICONVIEW, 
                           "sync-adjustments", FALSE,  
                           NULL);

  return iconview;
}
