/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */


#include <glib.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <clutter/clutter.h>
#include <clutter/clutter-x11.h>

#include "launcher-app.h"
#include "launcher-defines.h"

#define SET_DESKTOP_BACKGROUND_KEY "gconftool-2 --set /desktop/gnome/background/picture_options --type=string stretched"

/* Forwards */
static void set_window_hints (ClutterStage *stage);

static gboolean windowed = FALSE;
static gint     width    = 1024;
static gint     height   = 600;

static GOptionEntry entries[] =
{
  {
    "windowed",
    'w', 0,
    G_OPTION_ARG_NONE,
    &windowed,
    "Launch in windowed mode (for testing - 1024x600)",
    NULL
  },
  {
    "width",
    'd', 0,
    G_OPTION_ARG_INT,
    &width,
    "width",
    NULL
  },
  {
    "height",
    'h', 0,
    G_OPTION_ARG_INT,
    &height,
    "height",
    NULL
  },   {
    NULL
  }
};

gint
main (gint argc, gchar *argv[])
{
  ClutterActor *stage, *app;
  ClutterColor black = { 0x00, 0x00, 0x00, 0xff };
  GError *error = NULL;
  gchar *local_apps;

  g_thread_init (NULL);
  g_set_application_name ("Ubuntu Launcher");

  gtk_init (&argc, &argv);
  clutter_init_with_args (&argc, &argv,
                          " - Ubuntu Launcher", entries,
                          NULL,
                          &error);
  if (error)
  {
      g_print ("Unable to run Ubuntu Launcher: %s", error->message);
      g_error_free (error);
      return EXIT_FAILURE;
  }

  /* FIXME: Some tests that shouldn't be in here */
  local_apps = g_build_filename (g_get_home_dir (), 
                                 ".local", "share", "applications", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
  {
    /* As the local application dir doesn't exist, we assume this is first
     * run
     */
    g_mkdir_with_parents (local_apps, 0700);
    g_debug ("Created local applications directory at %s", local_apps);

    /* A nasty little hack for now */
    g_spawn_command_line_async (SET_DESKTOP_BACKGROUND_KEY, NULL);
  }
  g_free (local_apps);
  
  local_apps = g_build_filename (g_get_home_dir (), 
                                 ".config", "ume-launcher", "icons", NULL);
  if (!g_file_test (local_apps, G_FILE_TEST_EXISTS))
  {
    g_mkdir_with_parents (local_apps, 0700);
    g_debug ("Created local icons directory at %s", local_apps);
  }
  g_free (local_apps);

  /* This stuff does belong here */
  stage = clutter_stage_get_default ();
  clutter_actor_set_size (stage, width, height);

  if (!windowed)
  {
    clutter_stage_fullscreen (CLUTTER_STAGE (stage));
    set_window_hints (CLUTTER_STAGE (stage));
  }
  clutter_stage_set_color (CLUTTER_STAGE (stage), &black);

  /* Init() the main application */
  app = launcher_app_get_default ();
 
  clutter_actor_show (stage);
  clutter_main ();

  return EXIT_SUCCESS;
}


/*
 * Apply the 'desktop' window type to the clutter-stage window. Also, set the
 * 'panel' hints, to stop the windows from maximising the entire way.
 */
static void
set_window_hints (ClutterStage *stage)
{
  GdkDisplay *display = gdk_display_get_default (); 
  Display *xdisplay;
  Window stage_win;
  Atom atom;

  xdisplay = GDK_DISPLAY_XDISPLAY (display);
  stage_win = clutter_x11_get_stage_window (stage);

  /* 
   * Make the clutter window a 'desktop' window, i.e maximised, but behind 
   * everything else
   */
  atom = gdk_x11_get_xatom_by_name_for_display (display,
                                                "_NET_WM_WINDOW_TYPE_DESKTOP");
  XChangeProperty (xdisplay, stage_win,
                   gdk_x11_get_xatom_by_name_for_display (display,
                                                "_NET_WM_WINDOW_TYPE"),
                   XA_ATOM , 32, PropModeReplace, 
                   (guchar *)&atom, 1);
}
