/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "launcher-iconview.h"

#include <string.h>
#include <tidy/tidy.h>

#include "clutter-focus-group.h"

#include "launcher-app.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-icon.h"
#include "launcher-util.h"

#include "tidy-texture-frame.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherIconview,
                         launcher_iconview, 
                         TIDY_TYPE_VIEWPORT,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));


#define LAUNCHER_ICONVIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ICONVIEW, \
  LauncherIconviewPrivate))

struct _LauncherIconviewPrivate
{
  GList *icons;

  ClutterActor *scroll;
  ClutterActor *focused;

  TidyAdjustment *adjust;
};


static gboolean
on_scroll_event (ClutterActor       *actor,
                 ClutterScrollEvent *event,
                 TidyAdjustment     *adjust)
{
#define VALUE 50.0
  gdouble diff = VALUE;
  gdouble lower, upper;

  if (event->direction == CLUTTER_SCROLL_UP)
    diff = -VALUE;

  diff += tidy_adjustment_get_value (adjust)+diff;

  tidy_adjustment_get_values (adjust, NULL, &lower, &upper, NULL, NULL, NULL);

  diff = CLAMP (diff, lower, upper);

  tidy_adjustment_set_value (adjust, diff);

  return TRUE;
}                  

void           
launcher_iconview_set_category (LauncherIconview     *view,
                                LauncherMenuCategory *cat)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *scroll, *tex, *frame;
  ClutterActor *stage = clutter_stage_get_default ();
  TidyAdjustment *adjustment;
  GList *l;
  gint xoffset, yoffset, i = 0, rows = 0;
  gint view_width = cfg->iconview_width;
  gint view_height = cfg->iconview_height;
  gint view_padding = cfg->iconview_padding;
  gint icon_width = cfg->icon_width;
  gint icon_height = cfg->icon_height;
  gint app_cols = cfg->iconview_cols;
  gint app_rows = cfg->iconview_rows;
  gint app_count = 0;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  app_count = app_cols *app_rows;

  xoffset = -view_padding *2;
  yoffset = view_padding;

  for (l = launcher_menu_category_get_applications (cat); l; l = l->next)
  {
    LauncherMenuApplication *app = l->data;
    ClutterActor *actor;
    
    actor = launcher_icon_new ();
    launcher_icon_set_application (LAUNCHER_ICON (actor), app);
    clutter_container_add_actor (CLUTTER_CONTAINER (view), actor);
    clutter_actor_set_position (actor, xoffset, yoffset);
    clutter_actor_show (actor);

    priv->icons = g_list_append (priv->icons, actor);
        
    if (i == (app_cols -1))
    {
      i = 0;
      xoffset = - view_padding;
      yoffset += icon_height + view_padding;
      rows++;
    }
    else
    {
      xoffset += icon_width + view_padding;
      i++;
    }

  }
  clutter_actor_set_height (CLUTTER_ACTOR (view), (rows + 1) * icon_height);
  clutter_actor_set_clip (CLUTTER_ACTOR (view), 0, 1,
                          view_width,
                          view_height -2);

  /* Add the scroll bar */
  if (g_list_length (launcher_menu_category_get_applications (cat)) > app_count)
  {
    tidy_scrollable_get_adjustments (TIDY_SCROLLABLE (view), NULL, &adjustment);
    priv->adjust = adjustment;
    tidy_adjustment_set_values (adjustment, 
                                0,
                                0, (rows) * (icon_height + view_padding),
                                icon_height + view_padding,
                                app_cols * icon_width,
                                app_rows * icon_height);

    tex = launcher_util_texture_new_from_file (PKGDATADIR"/scrollbar.svg");
    clutter_container_add_actor (CLUTTER_CONTAINER (view),
                                 tex);
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 5, 5, 5, 5);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment); 
    scroll = priv->scroll = tidy_scroll_bar_new_with_handle (adjustment, frame);
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 5, 5, 5, 5);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
    clutter_actor_set_reactive (CLUTTER_ACTOR (scroll), TRUE);
    g_signal_connect (scroll, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);     
    clutter_actor_set_opacity (frame, 100);
    tidy_frame_set_texture (TIDY_FRAME (scroll), frame);

    clutter_container_add_actor (CLUTTER_CONTAINER (stage), scroll);
    clutter_actor_set_size (scroll, view_height*0.97, 10);
    clutter_actor_set_anchor_point_from_gravity (scroll, 
                                                 CLUTTER_GRAVITY_CENTER);
    clutter_actor_set_position (scroll, 
                                clutter_actor_get_x (CLUTTER_ACTOR (view)) +
                                view_width - 10/2,
                                (CSH()/2)+(PANEL_HEIGHT));  
  
    clutter_actor_set_rotation (scroll, CLUTTER_Z_AXIS, 90, 0, 0, 0);
    clutter_actor_show (scroll);

    clutter_actor_set_reactive (CLUTTER_ACTOR (view), TRUE);
    g_signal_connect (view, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
  }

}

void   
launcher_iconview_get_grid     (gint *cols, gint *rows)
{
  gint sheight= CSH();
  static gint grid_cols = 4;
  static gint grid_rows = 4;

  g_return_if_fail (cols);
  g_return_if_fail (rows);

  if (grid_rows)
  {
  }
  if (sheight < 600)
  {
    grid_rows = 3;
    grid_cols = 3;
  }
  else if (sheight < 800)
  {
    grid_rows = 4;
  }
  else
  {
    grid_rows = 5;
  }

  *cols = grid_cols;
  *rows = grid_rows;
}

/*
 * FOCUS STUFF
 */
static void
launcher_iconview_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;

  if (has_focus)
  {
    priv->focused = priv->icons->data;
    launcher_icon_set_focus (LAUNCHER_ICON (priv->focused), TRUE);
  }
  else
  {
    launcher_icon_set_focus (LAUNCHER_ICON (priv->focused), FALSE);
  }
}

static gboolean
launcher_iconview_direction_event (ClutterFocusGroup     *group,
                                 ClutterFocusDirection  dir)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  LauncherIcon *new;
  GList *children, *c;
  gint current;
  gint next;

  /* Move to the next or previous category, 
   *  if category is first or last, return FALSE
   *  page up and page down takes you to top and bottom
   */
  children = priv->icons;
  current = next = g_list_index (children, priv->focused);

  switch (dir)
  {
    case CLUTTER_DIRECTION_LEFT:
      next -= 1;
      break;
    case CLUTTER_DIRECTION_RIGHT:
      next += 1;
      break;
    case CLUTTER_DIRECTION_UP:
      next -= cfg->iconview_rows;
      break;
    case CLUTTER_DIRECTION_DOWN:
      next += cfg->iconview_rows;
      break;
    case CLUTTER_DIRECTION_PAGE_UP:
      next = 0;
      break;
    case CLUTTER_DIRECTION_PAGE_DOWN:
      next = g_list_length (children) -1;
      break;
    default:
      break;
  }
  if (next < 0) next = 0; 
  next = CLAMP (next, 0, g_list_length (children)-1);
  new = g_list_nth_data (children, next);
  priv->focused = CLUTTER_ACTOR (new);
  launcher_icon_set_focus (new, TRUE);

  for (c = children; c; c= c->next)
  {
    LauncherIcon *icon = c->data;

    if (icon != new)
      launcher_icon_set_focus (icon, FALSE);
  }

  if (TIDY_IS_ADJUSTMENT (priv->adjust))
  {
    gint y, pos, page;

    y = tidy_adjustment_get_value (priv->adjust);
    pos = cfg->icon_height * (next/cfg->iconview_rows);
    page = cfg->iconview_rows * cfg->icon_height;

    if (pos >= (y + page))
    {
      tidy_adjustment_set_value (priv->adjust, pos);
    }
    else if (pos < y)
    {
      tidy_adjustment_set_value (priv->adjust, pos);
    }
  }

  return TRUE;
}

static gboolean
launcher_iconview_action_event (ClutterFocusGroup *group)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;

  launcher_icon_launch (LAUNCHER_ICON (priv->focused));

  return TRUE;
}

static gboolean
launcher_iconview_key_event (ClutterFocusGroup *group, const gchar *string)
{
  g_debug ("String event: %s", string);

  return TRUE;
}                        

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus       = launcher_iconview_set_focus;
  iface->direction_event = launcher_iconview_direction_event;
  iface->key_event       = launcher_iconview_key_event;
  iface->action_event    = launcher_iconview_action_event;
}

/*
 * /FOCUS STUFF
 */
 
/* GObject stuff */
static void
launcher_iconview_finalize (GObject *object)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (object)->priv;

  if (priv->scroll)
    clutter_actor_destroy (LAUNCHER_ICONVIEW (object)->priv->scroll);
  g_list_free (priv->icons);
  
  G_OBJECT_CLASS (launcher_iconview_parent_class)->finalize (object);
}

static void
launcher_iconview_class_init (LauncherIconviewClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_iconview_finalize;

  g_type_class_add_private (obj_class, sizeof (LauncherIconviewPrivate));
}

      
static void
launcher_iconview_init (LauncherIconview *iconview)
{
  LauncherIconviewPrivate *priv;
  priv = iconview->priv = LAUNCHER_ICONVIEW_GET_PRIVATE (iconview);
}

ClutterActor *
launcher_iconview_new (void)

{
  ClutterActor *iconview = NULL;

  iconview = g_object_new (LAUNCHER_TYPE_ICONVIEW, 
                           "sync-adjustments", FALSE,  
                           NULL);

  return iconview;
}
