/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "launcher-catbar.h"

#include <string.h>

#include "clutter-drag-server.h"
#include "clutter-drag-dest.h"
#include "clutter-focus-group.h"

#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-util.h"
#include "launcher-shortcut.h"
#include "launcher-sidebar.h"
#include "launcher-wm.h"

#include "tidy-texture-frame.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherCatbar, 
                         launcher_catbar, 
                         CLUTTER_TYPE_GROUP,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));

#define LAUNCHER_CATBAR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_CATBAR, \
  LauncherCatbarPrivate))

struct _LauncherCatbarPrivate
{
  LauncherMenuCategory *fav;
  LauncherMenuCategory *active;

  ClutterTimeline *time;
  ClutterEffectTemplate *temp;

  ClutterActor *favourite;
  ClutterActor *trash;
};

enum
{
  CAT_SELECTED,

  LAST_SIGNAL
};
static guint _catbar_signals[LAST_SIGNAL] = { 0 };

static gboolean
on_category_clicked (LauncherShortcut *hit,
                     LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;
  LauncherMenuCategory *cat = g_object_get_data (G_OBJECT (hit), "category");
  GList *cats, *c;

  priv = bar->priv;

  if (cat == priv->active)
    return TRUE;

  cats = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (c = cats; c; c = c->next)
  {
    LauncherShortcut *cut = c->data;

    if (LAUNCHER_IS_SHORTCUT (cut) && cut != hit)
      launcher_shortcut_set_active (cut, FALSE);
  }
  g_list_free (cats);

  bar->priv->active = cat;

  launcher_sidebar_show_trash (LAUNCHER_SIDEBAR(launcher_sidebar_get_default()),
                               priv->active == priv->fav ? TRUE:FALSE);

  g_signal_emit (bar, _catbar_signals[CAT_SELECTED], 0, cat);
  
  return TRUE;
}

static void
on_drag_started (ClutterDragServer *server, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;
  GList *actors, *a;
  gint i = 0;

  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;

  actors = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (a = actors; a; a = a->next)
  {
    ClutterActor *actor = a->data;

    if (i && actor != priv->trash)
      clutter_effect_fade (priv->temp, actor, 50, NULL, NULL);

    i++;
  }
  g_list_free (actors);

}

static void
on_drag_finished (ClutterDragServer *server, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;
  GList *actors, *a;
  gint i = 0;

  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;

  actors = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (a = actors; a; a = a->next)
  {
    ClutterActor *actor = a->data;

    if (i && actor != priv->trash)
      clutter_effect_fade (priv->temp, actor, 255, NULL, NULL);

    i++;
  }
  g_list_free (actors);

}

/* GObject stuff */
static void
launcher_catbar_class_init (LauncherCatbarClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

	_catbar_signals[CAT_SELECTED] =
		g_signal_new ("category-selected",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherCatbarClass, category_selected),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER, 
			      G_TYPE_NONE,1, G_TYPE_POINTER);

  g_type_class_add_private (obj_class, sizeof (LauncherCatbarPrivate));
}

static ClutterActor *
_append_category (LauncherCatbar       *bar, 
                  LauncherMenuCategory *category,
                  gint                  width,
                  gint                  height,
                  gint                  offset)
{
  ClutterActor *cat;

  cat = launcher_shortcut_new (launcher_menu_category_get_icon_name (category),
                               launcher_menu_category_get_name (category),
                               FALSE);
  clutter_container_add_actor (CLUTTER_CONTAINER (bar), cat);
  clutter_actor_set_position (cat, 0, offset);
  clutter_actor_show (cat);

  g_object_set_data (G_OBJECT (cat), "category", category);
  g_signal_connect (cat, "clicked", 
                    G_CALLBACK (on_category_clicked), bar);

  return cat;
}

static gboolean
on_drag_drop (ClutterDragDest *dest, gpointer data)
{
  LauncherMenuApplication *app = data;

  if (!app)
    return FALSE;

  launcher_util_create_favorite (app);
  clutter_drag_dest_finish (dest, TRUE);

  return TRUE;
}

static void
on_desktop_shown (LauncherWm *wm, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;
}

static void
on_menu_changed (LauncherMenu *menu, LauncherCatbar *bar)
{
  LauncherCatbarPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  LauncherMenuCategory *old_cat = NULL;
  gint bar_width = cfg->bar_width;
  gint bar_height = cfg->shortcut_height;
	GList *l, *childs, *c;
  gint i = 0;
  
  g_return_if_fail (LAUNCHER_IS_CATBAR (bar));
  priv = bar->priv;

  childs = clutter_container_get_children (CLUTTER_CONTAINER (bar));
  for (c = childs; c; c = c->next)
  {
    clutter_actor_destroy (c->data);
  }
  g_list_free (childs);


  for (l = launcher_menu_get_categories (menu); l; l = l->next)
  {
    LauncherMenuCategory *cat = l->data;
    ClutterActor *shortcut;

    shortcut = _append_category (bar, cat, 
                                 bar_width, bar_height,
                                 (bar_height * i));
    if (cat == priv->active)
    {
      old_cat = cat;
      launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), TRUE);
    }

    if (strcmp (launcher_menu_category_get_icon_name (cat), FAVCATNAME)==0)
    {
      clutter_drag_dest_enable (CLUTTER_DRAG_DEST (shortcut));
      g_signal_connect (shortcut, "drop", 
                      G_CALLBACK (on_drag_drop), NULL);      
      priv->fav = cat;
      priv->favourite = shortcut;
    }
    i++;
  }

  if (!old_cat)
  {
    priv->active = priv->fav;
  launcher_sidebar_show_trash (LAUNCHER_SIDEBAR(launcher_sidebar_get_default()),
                               TRUE);
  }

  g_signal_emit (bar, _catbar_signals[CAT_SELECTED], 0, bar->priv->active);
}

/*
 * FOCUS STUFF
 */
static void
launcher_catbar_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  g_debug ("%s\n", has_focus ? "had focus" : "lost focus");
}

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus = launcher_catbar_set_focus;
}
   
static void
launcher_catbar_init (LauncherCatbar *catbar)
{
  LauncherCatbarPrivate *priv;
  LauncherMenu *menu = launcher_menu_get_default ();
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterDragServer *server = clutter_drag_server_get_default ();
  gint bar_width = cfg->bar_width;
  gint bar_height = cfg->shortcut_height;
	GList *l;
  gint i = 0;

  priv = catbar->priv = LAUNCHER_CATBAR_GET_PRIVATE (catbar);

  for (l = launcher_menu_get_categories (menu); l; l = l->next)
  {
    LauncherMenuCategory *cat = l->data;
    ClutterActor *shortcut;
    const char *icon_name;

    shortcut = _append_category (catbar, cat, 
                                 bar_width, bar_height,
                                 (bar_height * i));
    icon_name = launcher_menu_category_get_icon_name (cat);
    if (icon_name && strcmp (icon_name, FAVCATNAME) == 0)
    {
      clutter_drag_dest_enable (CLUTTER_DRAG_DEST (shortcut));
      launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), TRUE);
      g_signal_connect (shortcut, "drop", 
                      G_CALLBACK (on_drag_drop), NULL);      
      priv->fav = priv->active = cat;
      priv->favourite = shortcut;
    }
    i++;
  }

  launcher_sidebar_show_trash (LAUNCHER_SIDEBAR(launcher_sidebar_get_default()),
                               TRUE);


 /* Hook up drag server signals */
  g_signal_connect (server, "drag-started",
                    G_CALLBACK (on_drag_started), catbar);
  g_signal_connect (server, "drag-finished",
                    G_CALLBACK (on_drag_finished), catbar);

  /* Hook up menu changes */
  g_signal_connect (menu, "menu-changed",
                    G_CALLBACK (on_menu_changed), catbar);

  /* Reset when desktop is shown */
  g_signal_connect (launcher_wm_get_default(), "hide-windec",
                    G_CALLBACK (on_desktop_shown), catbar);

  priv->time = clutter_timeline_new_for_duration (MID_TIME);
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);
}

ClutterActor *
launcher_catbar_get_default (void)

{
  static ClutterActor *catbar = NULL;

  if (!catbar)
    catbar = g_object_new (LAUNCHER_TYPE_CATBAR, 
                       NULL);

  return catbar;
}
