/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "launcher-sidebar.h"

#include <gio/gio.h>
#include <glib/gi18n.h>

#include "clutter-focus-group.h"

#include "launcher-config.h"
#include "launcher-defines.h"
#include "clutter-drag-dest.h"
#include "clutter-drag-server.h"
#include "launcher-icon.h"
#include "launcher-shortcut.h"
#include "launcher-notify.h"
#include "launcher-util.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherSidebar, 
                         launcher_sidebar, 
                         CLUTTER_TYPE_GROUP,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));


#define LAUNCHER_SIDEBAR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_SIDEBAR, \
  LauncherSidebarPrivate))

struct _LauncherSidebarPrivate
{
  GVolumeMonitor *monitor;

  ClutterActor *bg;
  ClutterActor *group;

  GList *shortcuts;
  GList *links;

  ClutterActor *trash;
  ClutterActor *focused;


  ClutterTimeline *timeline;
  ClutterEffectTemplate *temp;
};


static void
ensure_layout (LauncherSidebar *sidebar)
{
  LauncherSidebarPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GList *s;
  gint bar_height = cfg->shortcut_height;
  gint offset = 0;

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (sidebar));
  priv = sidebar->priv;

  for (s = priv->shortcuts; s; s = s->next)
  {
    ClutterActor *actor = s->data;

    clutter_actor_set_position (actor, 0, offset);

    offset += bar_height;
  }
}

static void
on_volume_mounted (GVolume          *volume,
                   GAsyncResult     *res,
                   LauncherShortcut *shortcut)
{
  gboolean success;
  GMount *mount;
  GFile *root;
  gchar *path, *command;
  
  g_return_if_fail (G_IS_VOLUME (volume));

  success = g_volume_mount_finish (volume, res, NULL);
  
  if (!success)
    return;

  launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), FALSE);

  mount = g_volume_get_mount (volume);
  if (!G_IS_MOUNT (mount))
  {
    return;
  }

  root = g_mount_get_root (mount);
  path = g_file_get_path (root);
  command = g_strdup_printf ("nautilus --no-desktop --browser %s", path);

  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    command, NULL);

  g_free (command);
  g_free (path);
}

static void
on_volume_clicked (ClutterActor *shortcut, GVolume *volume)
{
  GMount *mount;
  GFile *root;
  gchar *path, *command;
  GdkPixbuf *temp;
  
  g_return_if_fail (G_IS_VOLUME (volume));

  launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), FALSE);

  mount = g_volume_get_mount (volume);
  if (!G_IS_MOUNT (mount))
  {
    if (!g_volume_can_mount (volume))
      return;

    g_volume_mount (volume, 0, NULL, NULL, 
                    (GAsyncReadyCallback)on_volume_mounted, shortcut);

    return;
  }

  root = g_mount_get_root (mount);
  path = g_file_get_path (root);
  command = g_strdup_printf ("nautilus --no-desktop --browser %s", path);

  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    command, NULL);

  /* Notify the user */
  temp = launcher_shortcut_get_icon (LAUNCHER_SHORTCUT (shortcut));
  launcher_notify_popup (launcher_notify_get_default (),
                  launcher_shortcut_get_label (LAUNCHER_SHORTCUT (shortcut)),
                         temp, 
                         -1);

  g_object_unref (temp);
  g_free (command);
  g_free (path);
}

static void
on_eject_clicked (ClutterActor *shortcut, GVolume *volume)
{
  
  g_return_if_fail (G_IS_VOLUME (volume));

  g_volume_eject (volume, 0, NULL, NULL, NULL);
}

static void
on_link_clicked (ClutterActor *shortcut, gpointer null)
{
  const gchar *command;
  GdkPixbuf *temp;

  launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), FALSE);

  command = clutter_actor_get_name (shortcut);

  if (command)
    gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                      command, NULL);

  /* Notify the user */
  temp = launcher_shortcut_get_icon (LAUNCHER_SHORTCUT (shortcut));
  launcher_notify_popup (launcher_notify_get_default (),
                  launcher_shortcut_get_label (LAUNCHER_SHORTCUT (shortcut)),
                         temp, 
                         -1);

  g_object_unref (temp);
}

static void
on_trash_clicked (ClutterActor *shortcut, gpointer null)
{
  launcher_shortcut_set_active (LAUNCHER_SHORTCUT (shortcut), FALSE);
}

/* Taken from nautilus::panel-util.c, with some modifications */
static gchar *
get_icon_name (GVolume *volume)
{
  GIcon *icon;
  const gchar * const *names;
  GtkIconTheme *icon_theme;
  int i;

  icon = g_volume_get_icon (volume);

  if (!G_IS_THEMED_ICON (icon))
    return g_strdup ("gnome-fs-folder");

  names = g_themed_icon_get_names (G_THEMED_ICON (icon));
  icon_theme = gtk_icon_theme_get_default ();

  for (i = 0; names[i] != NULL; i++)
  {
    if (gtk_icon_theme_has_icon (icon_theme, names[i]))
          return g_strdup (names[i]);
  }

  return NULL;
}

static void
on_volume_added (GVolumeMonitor *monitor,
                 GVolume *volume, 
                 LauncherSidebar *sidebar)
{ 
  LauncherSidebarPrivate *priv;
  GMount *mount;
  ClutterActor *shortcut;
  gchar *icon_name = NULL;

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (sidebar));
  priv = sidebar->priv;

  g_debug ("Volume Added: %s\n", g_volume_get_name (volume));

  mount = g_volume_get_mount (volume);
  if (!G_IS_MOUNT (mount))
  {
    g_debug ("Volume has not been mounted yet, ignoring");
    return;
  }

  icon_name = get_icon_name (volume);
  shortcut = launcher_shortcut_new (icon_name, 
                                    g_volume_get_name (volume),
                                    g_volume_can_eject (volume));
  
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
  g_object_set_data (G_OBJECT (shortcut), "volume", volume);
  clutter_actor_show (shortcut);

  g_signal_connect (shortcut, "clicked", 
                    G_CALLBACK (on_volume_clicked), volume);
  g_signal_connect (shortcut, "eject", 
                    G_CALLBACK (on_eject_clicked), volume);
  priv->shortcuts = g_list_append (priv->shortcuts, shortcut);

  ensure_layout (sidebar);

  g_free (icon_name);
}

static void
on_volume_removed (GVolumeMonitor *monitor,
                   GVolume *volume,
                   LauncherSidebar *sidebar)
{ 
  LauncherSidebarPrivate *priv;
  GList *s;

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (sidebar));
  priv = sidebar->priv;

  g_debug ("Volume Removed: %s\n", g_volume_get_name (volume));

  for (s = priv->shortcuts; s; s= s->next)
  {
    ClutterActor *shortcut = s->data;
    GVolume *vol = g_object_get_data (G_OBJECT (shortcut), "volume");

    if (vol == volume)
    {
      priv->shortcuts = g_list_remove (priv->shortcuts, shortcut);
      clutter_actor_destroy (shortcut);
        break;
    }                                 
  }

  ensure_layout (sidebar);
}

static gboolean
on_trashed (ClutterDragDest *dest, gpointer data)
{
  LauncherMenuApplication *app = data;

  if (!app)
    return FALSE;

  launcher_util_remove_favorite (app);
  clutter_drag_dest_finish (dest, TRUE);

  return TRUE;
}


void
launcher_sidebar_show_trash (LauncherSidebar *sidebar,
                             gboolean         show)
{
  LauncherSidebarPrivate *priv;
  ClutterActor *shortcut;
  
  g_return_if_fail (LAUNCHER_IS_SIDEBAR (sidebar));
  priv = sidebar->priv;

  if (show)
  {
    if (CLUTTER_IS_ACTOR (priv->trash))
    {
      priv->shortcuts = g_list_remove (priv->shortcuts, priv->trash);
      clutter_actor_destroy (priv->trash);
    }

    shortcut = priv->trash = launcher_shortcut_new ("gnome-fs-trash-empty", 
                                                    _("Remove favorite"),
                                                    FALSE);
  
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
    clutter_actor_show (shortcut);
    clutter_drag_dest_enable (CLUTTER_DRAG_DEST (priv->trash));

    g_signal_connect (priv->trash, "drop", 
                      G_CALLBACK (on_trashed), NULL);
 
    g_signal_connect (shortcut, "clicked", 
                      G_CALLBACK (on_trash_clicked), NULL);
    priv->shortcuts = g_list_append (priv->shortcuts, shortcut);
  }
  else
  {
    priv->shortcuts = g_list_remove (priv->shortcuts, priv->trash);
    clutter_actor_destroy (priv->trash);
  }
  
  ensure_layout (sidebar);
}

static void
launcher_sidebar_show (ClutterActor *actor)
{
  LauncherSidebarPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_SIDEBAR (actor));
  priv = LAUNCHER_SIDEBAR (actor)->priv;

  CLUTTER_ACTOR_CLASS (launcher_sidebar_parent_class)->show (actor);
}

static void
launcher_sidebar_hide (ClutterActor *actor)
{
  LauncherSidebarPrivate *priv;
  gint width = SIDEBAR_WIDTH ();

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (actor));
  priv = LAUNCHER_SIDEBAR (actor)->priv;

  clutter_effect_fade (priv->temp, actor, 0, 
      (ClutterEffectCompleteFunc)
        CLUTTER_ACTOR_CLASS (launcher_sidebar_parent_class)->hide, actor);
 
  clutter_effect_move (priv->temp, priv->group, width, 0, NULL, NULL);
}

/*
 * FOCUS STUFF
 */
static void
launcher_sidebar_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  LauncherSidebarPrivate *priv = LAUNCHER_SIDEBAR (group)->priv;

  if (has_focus)
  {
    priv->focused = priv->shortcuts->data;
    launcher_shortcut_set_active (LAUNCHER_SHORTCUT (priv->focused), TRUE);
  }
  else
    launcher_shortcut_set_active (LAUNCHER_SHORTCUT (priv->focused), FALSE);
}

static gboolean
launcher_sidebar_direction_event (ClutterFocusGroup     *group,
                                 ClutterFocusDirection  dir)
{
  LauncherSidebarPrivate *priv = LAUNCHER_SIDEBAR (group)->priv;
  LauncherShortcut *new;
  GList *children, *c;
  gint current;
  gint next;

  switch (dir)
  {
    case CLUTTER_DIRECTION_RIGHT:
    case CLUTTER_DIRECTION_LEFT:
      return TRUE;
    default:
      ;
  }
  /* Move to the next or previous category, 
   *  if category is first or last, return FALSE
   *  page up and page down takes you to top and bottom
   */
  children = g_list_copy (priv->shortcuts);
  children = g_list_concat (children, g_list_copy (priv->links));
  current = next = g_list_index (children, priv->focused);

  switch (dir)
  {
    case CLUTTER_DIRECTION_UP:
      next -= 1;
      break;
    case CLUTTER_DIRECTION_DOWN:
      next += 1;
      break;
    case CLUTTER_DIRECTION_PAGE_UP:
      next = 0;
      break;
    case CLUTTER_DIRECTION_PAGE_DOWN:
      next = g_list_length (children) -1;
      break;
    default:
      break;
  }
  if (next < 0) next = 0;
  next = CLAMP (next, 0, g_list_length (children)-1);
  new = g_list_nth_data (children, next);
  priv->focused = CLUTTER_ACTOR (new);
  launcher_shortcut_set_active (new, TRUE);

  for (c = children; c; c= c->next)
  {
    LauncherShortcut *shortcut = c->data;

    if (!LAUNCHER_IS_SHORTCUT (shortcut))
      continue;

    if (shortcut != new)
      launcher_shortcut_set_active (shortcut, FALSE);
  }

  g_list_free (children);

  return TRUE;
}

static gboolean
launcher_sidebar_action_event (ClutterFocusGroup *group)
{
  LauncherSidebarPrivate *priv = LAUNCHER_SIDEBAR (group)->priv;

  launcher_shortcut_clicked (LAUNCHER_SHORTCUT (priv->focused));

  return TRUE;
}

static gboolean
launcher_sidebar_key_event (ClutterFocusGroup *group, const gchar *string)
{
  g_debug ("String event: %s", string);

  return TRUE;
}                        

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus       = launcher_sidebar_set_focus;
  iface->direction_event = launcher_sidebar_direction_event;
  iface->key_event       = launcher_sidebar_key_event;
  iface->action_event    = launcher_sidebar_action_event;
}

/*
 * /FOCUS STUFF
 */
 

/* GObject stuff */
static void
launcher_sidebar_finalize (GObject *object)
{
  LauncherSidebarPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (object));
  priv = LAUNCHER_SIDEBAR (object)->priv;

  g_object_unref (priv->monitor);

  g_list_free (priv->shortcuts);
  g_list_free (priv->links);

  G_OBJECT_CLASS (launcher_sidebar_parent_class)->finalize (object);
}

static void
launcher_sidebar_class_init (LauncherSidebarClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass   *act_class = CLUTTER_ACTOR_CLASS (klass);

  obj_class->finalize = launcher_sidebar_finalize;

  act_class->show = launcher_sidebar_show;
  act_class->hide = launcher_sidebar_hide;

  g_type_class_add_private (obj_class, sizeof (LauncherSidebarPrivate));
}

static ClutterActor * 
add_shortcut (LauncherSidebar *sidebar, 
              const gchar *icon_name, 
              const gchar *name,
              const gchar *command)
{
  LauncherSidebarPrivate *priv = sidebar->priv;
  ClutterActor *shortcut;
   
  shortcut = launcher_shortcut_new (icon_name, name, FALSE);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
  clutter_actor_set_name (shortcut, command);
  clutter_actor_show (shortcut);
  priv->shortcuts = g_list_append (priv->shortcuts, shortcut);
  g_signal_connect (shortcut, "clicked", G_CALLBACK (on_link_clicked), NULL);
  
  return shortcut;
}
      
static void
load_volumes (LauncherSidebar *sidebar)
{
  LauncherSidebarPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *shortcut;
  GList *volumes, *vol;
  gint bar_height = cfg->shortcut_height;
  gchar *home;

  g_return_if_fail (LAUNCHER_IS_SIDEBAR (sidebar));
  priv = sidebar->priv;

  home = g_strdup_printf ("nautilus --no-desktop --browser %s", 
                          g_get_home_dir ());
  shortcut = add_shortcut (sidebar, "gnome-fs-home", _("Home"), home);
  g_free (home);
  shortcut = add_shortcut (sidebar, "gnome-fs-network", _("Network"), 
                           "nautilus --no-desktop --browser Network:///");

  volumes = g_volume_monitor_get_volumes (priv->monitor);
  for (vol = volumes; vol; vol = vol->next)
  {
    GVolume *volume;
    gchar *icon_name;
    
    volume = vol->data;
    icon_name = get_icon_name (volume);
    shortcut = launcher_shortcut_new (icon_name, 
                                      g_volume_get_name (volume),
                                      g_volume_can_eject (volume));
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
    clutter_actor_show (shortcut);
    priv->shortcuts = g_list_append (priv->shortcuts, shortcut);

    g_signal_connect (shortcut, "clicked", 
                      G_CALLBACK (on_volume_clicked), volume);
    g_signal_connect (shortcut, "eject", 
                      G_CALLBACK (on_eject_clicked), volume);


    g_free (icon_name);
  }

  ensure_layout (sidebar);

  shortcut = launcher_shortcut_new ("gtk-quit",
                                    _("Quit..."),
                                    FALSE);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
  clutter_actor_set_position (shortcut, 0, 
                              cfg->win_height - (bar_height)-(PANEL_HEIGHT*2));
  clutter_actor_set_name (shortcut, "gnome-session-save --gui --kill");
  clutter_actor_show (shortcut);
  priv->links = g_list_append (priv->links, shortcut);
  g_signal_connect (shortcut, "clicked", G_CALLBACK (on_link_clicked), NULL);

  shortcut = launcher_shortcut_new ("gtk-preferences",
                                    _("Settings"),
                                    FALSE);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->group), shortcut);
  clutter_actor_set_position (shortcut, 0, 
                             cfg->win_height - (bar_height*2)-(PANEL_HEIGHT*2));
  clutter_actor_set_name (shortcut, "gnome-control-center");
  clutter_actor_show (shortcut);
  priv->links = g_list_prepend (priv->links, shortcut);
  g_signal_connect (shortcut, "clicked", G_CALLBACK (on_link_clicked), NULL);
}

static void
launcher_sidebar_init (LauncherSidebar *sidebar)
{
  LauncherSidebarPrivate *priv;
  
  priv = sidebar->priv = LAUNCHER_SIDEBAR_GET_PRIVATE (sidebar);

  priv->monitor = g_volume_monitor_get ();
  priv->shortcuts = NULL;

  priv->group = clutter_group_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (sidebar), priv->group);
  clutter_actor_set_position (priv->group, 0, 0);
  clutter_actor_show (priv->group);

  priv->timeline = clutter_timeline_new_for_duration (SLOW_TIME);
  priv->temp = clutter_effect_template_new (priv->timeline,
                                            clutter_sine_inc_func);

  load_volumes (sidebar);

  g_signal_connect (priv->monitor, "volume-added", 
                    G_CALLBACK (on_volume_added), sidebar);
  g_signal_connect (priv->monitor, "volume-removed", 
                    G_CALLBACK (on_volume_removed), sidebar);
}

ClutterActor *
launcher_sidebar_get_default (void)

{
  static ClutterActor *sidebar = NULL;

  if (!sidebar)
    sidebar = g_object_new (LAUNCHER_TYPE_SIDEBAR, 
                       NULL);

  return sidebar;
}
