/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>
#include <string.h>

#include "launcher-icon.h"

#include "clutter-drag-server.h"

#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-iconview.h"
#include "launcher-notify.h"
#include "launcher-startup.h"
#include "launcher-util.h"

#include "tidy-texture-frame.h"

G_DEFINE_TYPE (LauncherIcon, launcher_icon, CLUTTER_TYPE_GROUP);

#define LAUNCHER_ICON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ICON, \
  LauncherIconPrivate))

static ClutterActor *bg_texture = NULL;

struct _LauncherIconPrivate
{
  LauncherMenuApplication *app;
  
  ClutterTimeline       *time;
  ClutterEffectTemplate *temp;

  ClutterTimeline       *fast_time;
  ClutterEffectTemplate *fast_temp;

  ClutterTimeline       *current;

  ClutterActor *white;
  ClutterActor *orange;
  ClutterActor *texture;
  ClutterActor *clone;

  gint x;
  gint y;
};

static gboolean
on_motion (ClutterActor       *actor,
           ClutterMotionEvent *event,
           LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;
  ClutterActor *clone;
  
  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_ACTOR (priv->clone))
  {
    g_debug ("Drag already initiated");
    return FALSE;
  }

  if (event->modifier_state & CLUTTER_BUTTON1_MASK 
        && !CLUTTER_IS_ACTOR (priv->clone))
  {
    priv->clone = clone = clutter_clone_texture_new 
                                             (CLUTTER_TEXTURE (priv->texture));
    clutter_actor_set_size (priv->clone,CAW(priv->texture), CAH(priv->texture));
    clutter_container_add_actor (CLUTTER_CONTAINER(clutter_stage_get_default()),
                               priv->clone);  
    clutter_actor_set_position (clone, event->x, event->y);
    clutter_actor_show (clone);

    clutter_drag_server_begin_drag (clutter_drag_server_get_default(),
                                    clone,
                                    priv->app);
   return TRUE;
  }

  return FALSE;
}


static void
position_func (GtkMenu  *menu, 
               gint     *x,
               gint     *y, 
               gboolean *push_in,
               gpointer  data)
{
  LauncherIconPrivate *priv;

  priv = LAUNCHER_ICON_GET_PRIVATE (data);

  *x = priv->x;
  *y = priv->y;
  *push_in = TRUE;
}

static void
launch_app (GtkMenuItem *item, LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GdkPixbuf *pixbuf = NULL;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }
 
  launcher_startup_launch_app (launcher_startup_get_default (),
                               priv->app);
  
  g_object_get (priv->texture, "pixbuf", &pixbuf, NULL);
  launcher_notify_popup (launcher_notify_get_default (),
                         launcher_menu_application_get_name (priv->app),
                         pixbuf,
                         launcher_menu_application_get_pid (priv->app));
  g_object_unref (pixbuf);

  clutter_effect_rotate (priv->temp,
                         CLUTTER_ACTOR (icon),
                         CLUTTER_Y_AXIS,
                         0.0,
                         cfg->icon_width/2, 0, 0, 
                         CLUTTER_ROTATE_CW,
                         NULL, NULL);
}

static void
remove_app (GtkMenuItem *item, LauncherIcon *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  launcher_util_remove_favorite (priv->app);
}

static void
fav_app (GtkMenuItem *item, LauncherIcon *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  launcher_util_create_favorite (priv->app);
}

static void
show_menu (LauncherIcon *icon, ClutterButtonEvent *event)
{
  LauncherIconPrivate *priv;
  GtkWidget *menu, *item;
  LauncherMenuCategory *cat;
  const gchar *icon_name;
  
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  priv->x = event->x;
  priv->y = event->y;  
 
  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_EXECUTE, NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (launch_app), icon);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  cat = launcher_menu_application_get_category (priv->app);
  icon_name = launcher_menu_category_get_icon_name (cat);
  if (icon_name && strcmp (icon_name, FAVCATNAME) == 0)
  {
    item = gtk_image_menu_item_new_from_stock (GTK_STOCK_REMOVE, NULL);
    gtk_widget_show (item);
    g_signal_connect (item, "activate", G_CALLBACK (remove_app), icon);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  }
  else
  {
    item = gtk_image_menu_item_new_with_label (_("Add to Favorites"));
    g_object_set (item, "image", gtk_image_new_from_stock (GTK_STOCK_ADD,
                                                           GTK_ICON_SIZE_MENU),
                  NULL);
    gtk_widget_show (item);
    g_signal_connect (item, "activate", G_CALLBACK (fav_app), icon);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);  
  }

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL, 
                  position_func, icon,
                  3, event->time); 
}

static gboolean
on_app_clicked (ClutterActor       *actor,
                ClutterButtonEvent *event,
                LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;
  
  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_ACTOR (priv->clone))
  {
    clutter_drag_server_finish_drag (clutter_drag_server_get_default ());
    return FALSE;
  }

  if (event->button == 3)
  {
    show_menu (icon, event);
    return TRUE;
  } 
  else if (event->button == 1)
  {
    launch_app (NULL, icon);
    return TRUE;
  }

  return FALSE;
}

static gboolean
on_enter (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherIcon          *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }

  priv->current = clutter_effect_fade (priv->temp, priv->white, 0, NULL, NULL);
  clutter_effect_fade (priv->temp, priv->orange, 255, NULL, NULL);
  return TRUE;
}


static gboolean
on_leave (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherIcon          *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }

  priv->current = clutter_effect_fade (priv->temp, priv->white, 255, NULL, NULL);
  clutter_effect_fade (priv->temp, priv->orange, 0, NULL, NULL);

  return TRUE;
}

void 
launcher_icon_set_application (LauncherIcon            *icon, 
                               LauncherMenuApplication *application)
{
  LauncherIconPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GdkPixbuf *pixbuf;
  ClutterActor *texture;
  ClutterActor *label, *new;;
  ClutterColor color = { 0xff, 0xff, 0xff, 0xff };
  ClutterColor orange = { 221, 119, 27, 255 };
  gint width = cfg->icon_width;
  gint height = cfg->icon_height;
  gint padding = width * 0.08;
 
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  priv->app = application;

  /* Icon */
  pixbuf = launcher_menu_application_get_icon (application);
  texture = priv->texture = clutter_texture_new_from_pixbuf (pixbuf);
  clutter_actor_set_size (texture, 
                          height -(6*padding), 
                          height-(6*padding));
  clutter_actor_set_anchor_point_from_gravity (texture, CLUTTER_GRAVITY_NORTH);
  clutter_actor_set_position (texture, 
                              width/2,
                              padding);
  clutter_container_add_actor (CLUTTER_CONTAINER (icon), texture);
  clutter_actor_show (texture);

  /* Text label */
  label = priv->white = clutter_label_new ();
  clutter_label_set_font_name (CLUTTER_LABEL (label), cfg->font_name);
  clutter_label_set_color (CLUTTER_LABEL (label), &color);
  clutter_label_set_line_wrap (CLUTTER_LABEL (label), FALSE);
  clutter_label_set_alignment (CLUTTER_LABEL (label), PANGO_ALIGN_CENTER);
  clutter_actor_set_width (label, width*0.98);
  clutter_label_set_text (CLUTTER_LABEL (label),
                          launcher_menu_application_get_name (application));

  clutter_container_add_actor (CLUTTER_CONTAINER (icon), label);
  clutter_actor_set_anchor_point_from_gravity (label, CLUTTER_GRAVITY_NORTH);
  clutter_actor_set_position (label, 
                              width/2,
                              clutter_actor_get_y (texture) + CAH(texture)
                                + (padding));
  clutter_actor_show (label);

  /* Text label */
  label = priv->orange = clutter_label_new ();
  clutter_label_set_font_name (CLUTTER_LABEL (label), cfg->font_name);
  clutter_label_set_color (CLUTTER_LABEL (label), &orange);
  clutter_label_set_line_wrap (CLUTTER_LABEL (label), FALSE);
  clutter_label_set_alignment (CLUTTER_LABEL (label), PANGO_ALIGN_CENTER);
  clutter_actor_set_width (label, width);
  clutter_label_set_text (CLUTTER_LABEL (label),
                          launcher_menu_application_get_name (application));

  clutter_container_add_actor (CLUTTER_CONTAINER (icon), label);
  clutter_actor_set_anchor_point_from_gravity (label, CLUTTER_GRAVITY_NORTH);
  clutter_actor_set_position (label, 
                              width/2,
                              clutter_actor_get_y (texture) + CAH(texture)
                                + (padding));
  clutter_actor_set_opacity (label, 0);
  clutter_actor_show (label);

  /* 'New' indicator */
  if (launcher_menu_application_is_new (application))
  {
    new = launcher_util_texture_new_from_file (PKGDATADIR"/new.svg");
    clutter_container_add_actor (CLUTTER_CONTAINER (icon), new);
    clutter_actor_set_size (new, 
                            2*padding,
                            2*padding);
    clutter_actor_set_position (new, (width*0.75) - (2*padding), 0);
    clutter_actor_show (new);
  }


  clutter_actor_set_reactive (CLUTTER_ACTOR (icon), TRUE);
  g_signal_connect (icon, "button-release-event",
                    G_CALLBACK (on_app_clicked), icon);
  g_signal_connect (icon, "enter-event",
                    G_CALLBACK (on_enter), icon);
  g_signal_connect (icon, "leave-event",
                    G_CALLBACK (on_leave), icon);
  clutter_actor_set_reactive (priv->texture, TRUE);
  g_signal_connect (priv->texture, "motion-event", 
                    G_CALLBACK (on_motion), icon);
 
}

void 
launcher_icon_set_focus (LauncherIcon *icon, gboolean has_focus)
{
  LauncherIconPrivate *priv;
    
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  if (has_focus)
  {
    clutter_effect_fade (priv->temp, priv->white, 0, NULL, NULL);
    clutter_effect_fade (priv->temp, priv->orange, 255, NULL, NULL);
  }
  else
  {
    clutter_effect_fade (priv->temp, priv->white, 255, NULL, NULL);
    clutter_effect_fade (priv->temp, priv->orange, 0, NULL, NULL);  
  }
}

void 
launcher_icon_launch (LauncherIcon *icon)
{
  launch_app (NULL, icon);
}


static void
on_show (ClutterActor *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = LAUNCHER_ICON (icon)->priv;

  CLUTTER_ACTOR_CLASS (launcher_icon_parent_class)->show (icon);
 
  clutter_actor_set_opacity (icon, 0);
  clutter_effect_move (priv->temp, icon, 
                       clutter_actor_get_x (icon), 
                       clutter_actor_get_y (icon), NULL, NULL);
  clutter_effect_fade (priv->temp, icon, 255, NULL, NULL);
}


/* GObject stuff */
static void
launcher_icon_finalize (GObject *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = LAUNCHER_ICON_GET_PRIVATE (icon);

  G_OBJECT_CLASS (launcher_icon_parent_class)->finalize (icon);
}

static void
launcher_icon_class_init (LauncherIconClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass   *act_class = CLUTTER_ACTOR_CLASS (klass);

  obj_class->finalize = launcher_icon_finalize;
  act_class->show = on_show;

  g_type_class_add_private (obj_class, sizeof (LauncherIconPrivate));
}

      
static void
launcher_icon_init (LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  
  priv = icon->priv = LAUNCHER_ICON_GET_PRIVATE (icon);

  if (!CLUTTER_IS_ACTOR (bg_texture))
  {
    GdkPixbuf *temp;

    temp = gdk_pixbuf_new_from_file (PKGDATADIR"/icon.png", NULL);
    bg_texture = clutter_texture_new_from_pixbuf (temp);
    clutter_actor_realize (bg_texture);
    g_object_unref (temp);
  }

  priv->time = clutter_timeline_new_for_duration (SLOW_TIME);
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);

  priv->fast_time = clutter_timeline_new_for_duration (MID_TIME);
  priv->fast_temp = clutter_effect_template_new (priv->fast_time,
                                                 clutter_sine_inc_func);
}

ClutterActor *
launcher_icon_new (void)

{
  ClutterActor *icon = NULL;

  icon = g_object_new (LAUNCHER_TYPE_ICON, 
                           NULL);

  return icon;
}
