/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

#ifndef __PGM_CANVAS_H__
#define __PGM_CANVAS_H__

#include <gst/gst.h>
#include "pgmcommon.h"
#include "pgmdrawable.h"

G_BEGIN_DECLS

#define PGM_TYPE_CANVAS (pgm_canvas_get_type ())
#define PGM_CANVAS(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), PGM_TYPE_CANVAS, PgmCanvas))
#define PGM_CANVAS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), PGM_TYPE_CANVAS, PgmCanvasClass))
#define PGM_IS_CANVAS(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), PGM_TYPE_CANVAS))
#define PGM_IS_CANVAS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), PGM_TYPE_CANVAS))
#define PGM_CANVAS_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS((obj), PGM_TYPE_CANVAS, PgmCanvasClass))

typedef struct _PgmCanvas      PgmCanvas;
typedef struct _PgmCanvasClass PgmCanvasClass;

/**
 * PgmCanvas:
 * @far_layer: the list of #PgmDrawable in the #PGM_DRAWABLE_FAR layer.
 * @middle_layer: the list of #PgmDrawable in the #PGM_DRAWABLE_MIDDLE layer.
 * @near_layer: the list of #PgmDrawable in the #PGM_DRAWABLE_NEAR layer.
 * @width: the canvas width.
 * @height: the canvas height.
 * @pixel_formats: the pixel formats mask.
 *
 * The #PgmCanvas structure.
 */
struct _PgmCanvas {
  GstObject parent;

  GHashTable *format_counter;
  guint       nb_formats;

  /*< public >*/ /* with LOCK */
  /* Layer lists */
  GList *far_layer;
  GList *middle_layer;
  GList *near_layer;

  /* Canvas size */
  gfloat width, height;

  /* Mask of supported formats */
  gulong pixel_formats;

  /*< private >*/

  /* for picking */
  PgmDrawable *entered_emission_stopper;

  /* Padding */
  gpointer _pgm_reserved[PGM_PADDING];
};

struct _PgmCanvasClass {
  GstObjectClass parent_class;

  /*< private >*/

  /* Signals */
  void (*drawable_added)     (PgmCanvas *canvas,
                              PgmDrawable *drawable,
                              PgmDrawableLayer layer,
                              gint order);

  void (*drawable_removed)   (PgmCanvas *canvas,
                              PgmDrawable *drawable,
                              PgmDrawableLayer layer);

  void (*drawable_reordered) (PgmCanvas *canvas,
                              PgmDrawable *drawable,
                              PgmDrawableLayer layer);

  void (*size_changed)       (PgmCanvas *canvas);

  void (*regenerated)        (PgmCanvas *canvas);

  gpointer _pgm_reserved[PGM_PADDING];
};

GType      pgm_canvas_get_type          (void);

PgmCanvas *pgm_canvas_new               (void);

PgmError   pgm_canvas_set_size          (PgmCanvas *canvas,
                                         gfloat width,
                                         gfloat height);
PgmError   pgm_canvas_get_size          (PgmCanvas *canvas,
                                         gfloat *width,
                                         gfloat *height);

PgmError   pgm_canvas_add               (PgmCanvas *canvas,
                                         PgmDrawableLayer layer,
                                         PgmDrawable *drawable);
PgmError   pgm_canvas_remove            (PgmCanvas *canvas,
                                         PgmDrawable *drawable);

PgmError   pgm_canvas_add_many          (PgmCanvas *canvas,
                                         PgmDrawableLayer layer,
                                         PgmDrawable *drawable_1,
                                         ...) G_GNUC_NULL_TERMINATED;
PgmError   pgm_canvas_remove_many       (PgmCanvas *canvas,
                                         PgmDrawable *drawable_1,
                                         ...) G_GNUC_NULL_TERMINATED;

PgmError   pgm_canvas_set_order         (PgmCanvas *canvas,
                                         PgmDrawable *drawable,
                                         gint order);
PgmError   pgm_canvas_get_order         (PgmCanvas *canvas,
                                         PgmDrawable *drawable,
                                         PgmDrawableLayer *layer,
                                         gint *order);

PgmError   pgm_canvas_get_layer_count   (PgmCanvas *canvas,
                                         PgmDrawableLayer layer,
                                         gint *count);

PgmError   pgm_canvas_regenerate        (PgmCanvas *canvas);

PgmError   pgm_canvas_get_pixel_formats (PgmCanvas *canvas,
                                         gulong *pixel_formats);

/* Protected methods */

void         _pgm_canvas_update_order                 (PgmCanvas *canvas,
                                                       PgmDrawable *drawable);

void         _pgm_canvas_add_pixel_formats            (PgmCanvas *canvas,
                                                       gulong pixel_formats);

void         _pgm_canvas_remove_pixel_formats         (PgmCanvas *canvas,
                                                       gulong pixel_formats);

PgmDrawable *_pgm_canvas_get_entered_emission_stopper (PgmCanvas *canvas);

void         _pgm_canvas_set_entered_emission_stopper (PgmCanvas *canvas,
                                                       PgmDrawable *drawable);

G_END_DECLS

#endif /* __PGM_CANVAS_H__ */
