/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include "pgmglesimage.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gles_image_debug);
#define GST_CAT_DEFAULT pgm_gles_image_debug

static PgmGlesDrawableClass *parent_class = NULL;

/* Private methods */

/* Computes the image pixel aspect ratio */
static void
update_image_ratio (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);

  GST_OBJECT_LOCK (image);
  glesimage->image_ratio = (gfloat) image->par_n / image->par_d;
  GST_OBJECT_UNLOCK (image);
}

/* Computes the drawable pixel aspect ratio */
static void
update_drawable_ratio (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);

  if (G_LIKELY (glesdrawable->height != 0.0))
    glesimage->drawable_ratio = glesdrawable->width / glesdrawable->height;
  else
    glesimage->drawable_ratio = 1.0f;
}

/* Updates the last position fields with the one from the drawable */
static void
update_last_position (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);

  glesimage->last_x = glesdrawable->x;
  glesimage->last_y = glesdrawable->y;
  glesimage->last_z = glesdrawable->z;
}

/* Updates the alignment */
static void
update_alignment (PgmGlesImage *glesimage)
{
  PgmDrawable *drawable = PGM_GLES_DRAWABLE (glesimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageAlignment align;

  GST_OBJECT_LOCK (image);
  align = image->align;
  GST_OBJECT_UNLOCK (image);

  /* Horizontal alignment */
  if (align & PGM_IMAGE_LEFT)
    glesimage->h_align = 0.0f;
  else if (align & PGM_IMAGE_RIGHT)
    glesimage->h_align = 1.0f;
  else
    glesimage->h_align = 0.5f;

  /* Vertical alignment */
  if (align & PGM_IMAGE_TOP)
    glesimage->v_align = 0.0f;
  else if (align & PGM_IMAGE_BOTTOM)
    glesimage->v_align = 1.0f;
  else
    glesimage->v_align = 0.5f;
}

/* Updates the interpolation */
static void
update_interp (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesTexture *glestexture = glesimage->texture;

  GST_OBJECT_LOCK (image);

  if (image->interp == PGM_IMAGE_BILINEAR)
    glestexture->filter = PGM_GLES_LINEAR;
  else if (image->interp == PGM_IMAGE_NEAREST)
    glestexture->filter = PGM_GLES_NEAREST;

  GST_OBJECT_UNLOCK (image);
}

/* Updates the wrapping */
static void
update_wrapping (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesTexture *glestexture = glesimage->texture;

  GST_OBJECT_LOCK (image);

  if (image->wrap_s == PGM_IMAGE_CLAMP)
    glestexture->wrap_s = PGM_GLES_CLAMP_TO_EDGE;
  else if (image->wrap_s == PGM_IMAGE_REPEAT)
    glestexture->wrap_s = PGM_GLES_REPEAT;

  if (image->wrap_t == PGM_IMAGE_CLAMP)
    glestexture->wrap_t = PGM_GLES_CLAMP_TO_EDGE;
  else if (image->wrap_t == PGM_IMAGE_REPEAT)
    glestexture->wrap_t = PGM_GLES_REPEAT;

  GST_OBJECT_UNLOCK (image);
}

/* Updates the mapping (texture) matrix */
static void
update_mapping_matrix (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);

  GST_OBJECT_LOCK (image);
  pgm_gles_texture_set_matrix (glesimage->texture, image->mapping_matrix);
  GST_OBJECT_UNLOCK (image);
}

/* Updates the color of the inner border vertices */
static void
update_border_inner_color (PgmGlesImage *glesimage)
{
  PgmDrawable *drawable = PGM_GLES_DRAWABLE (glesimage)->drawable;
  PgmGlesFloat *border_color = glesimage->border_color;
  PgmImage *image = PGM_IMAGE (drawable);
  gfloat r, g, b, a;
  guint i;

  /* Get the right color components */
  GST_OBJECT_LOCK (drawable);
  r = image->border_inner_r * INV_255;
  g = image->border_inner_g * INV_255;
  b = image->border_inner_b * INV_255;
  a = image->border_inner_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Then set the color attribute of the inner vertices */
  for (i = 4; i < 48; i += 12)
    {
      border_color[i+0] = r;
      border_color[i+1] = g;
      border_color[i+2] = b;
      border_color[i+3] = a;
    }
}

/* Updates the color of the outer border vertices */
static void
update_border_outer_color (PgmGlesImage *glesimage)
{
  PgmDrawable *drawable = PGM_GLES_DRAWABLE (glesimage)->drawable;
  PgmGlesFloat *border_color = glesimage->border_color;
  PgmImage *image = PGM_IMAGE (drawable);
  gfloat r, g, b, a;

  /* Get the right color components */
  GST_OBJECT_LOCK (drawable);
  r = image->border_outer_r * INV_255;
  g = image->border_outer_g * INV_255;
  b = image->border_outer_b * INV_255;
  a = image->border_outer_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Then set the color attribute of the outer vertices */
  border_color[0] = r;
  border_color[1] = g;
  border_color[2] = b;
  border_color[3] = a;
  border_color[8] = r;
  border_color[9] = g;
  border_color[10] = b;
  border_color[11] = a;
  border_color[12] = r;
  border_color[13] = g;
  border_color[14] = b;
  border_color[15] = a;
  border_color[20] = r;
  border_color[21] = g;
  border_color[22] = b;
  border_color[23] = a;
  border_color[24] = r;
  border_color[25] = g;
  border_color[26] = b;
  border_color[27] = a;
  border_color[32] = r;
  border_color[33] = g;
  border_color[34] = b;
  border_color[35] = a;
  border_color[36] = r;
  border_color[37] = g;
  border_color[38] = b;
  border_color[39] = a;
  border_color[44] = r;
  border_color[45] = g;
  border_color[46] = b;
  border_color[47] = a;
}

/* Updates the border vertices */
static void
set_border_vertices (PgmGlesImage *glesimage,
                     gfloat border_width,
                     gfloat border_height)
{
  PgmGlesFloat *border_vertex = glesimage->border_vertex;
  PgmGlesFloat *img_vertex = glesimage->vertex;

  /* Here is how the border vertices are drawn using a GL_TRIANGLES mode.
   * Remember that the projection matrix flip the scene vertically and the
   * representation of this schema in a Pigment scene is inverted along the y
   * cardinal axis.
   *
   * 3-------------2---6
   * |             |   |
   * 0---10--------1   |
   * |   |         |   |
   * |   |         |   |
   * |   7---------4---5
   * |   |             |
   * 11--8-------------9
   */

  /* Vertex 0 */
  border_vertex[0] = img_vertex[9] - border_width;
  border_vertex[1] = img_vertex[10];
  border_vertex[2] = img_vertex[11];
  /* Vertex 1 */
  border_vertex[3] = img_vertex[6];
  border_vertex[4] = img_vertex[7];
  border_vertex[5] = img_vertex[8];
  /* Vertex 2 */
  border_vertex[6] = img_vertex[6];
  border_vertex[7] = img_vertex[7] + border_height;
  border_vertex[8] = img_vertex[8];
  /* Vertex 3 */
  border_vertex[9] = img_vertex[9] - border_width;
  border_vertex[10] = img_vertex[10] + border_height;
  border_vertex[11] = img_vertex[11];
  /* Vertex 4 */
  border_vertex[12] = img_vertex[3];
  border_vertex[13] = img_vertex[4];
  border_vertex[14] = img_vertex[5];
  /* Vertex 5 */
  border_vertex[15] = img_vertex[3] + border_width;
  border_vertex[16] = img_vertex[4];
  border_vertex[17] = img_vertex[5];
  /* Vertex 6 */
  border_vertex[18] = img_vertex[6] + border_width;
  border_vertex[19] = img_vertex[7] + border_height;
  border_vertex[20] = img_vertex[8];
  /* Vertex 7 */
  border_vertex[21] = img_vertex[0];
  border_vertex[22] = img_vertex[1];
  border_vertex[23] = img_vertex[2];
  /* Vertex 8 */
  border_vertex[24] = img_vertex[0];
  border_vertex[25] = img_vertex[1] - border_height;
  border_vertex[26] = img_vertex[2];
  /* Vertex 9 */
  border_vertex[27] = img_vertex[3] + border_width;
  border_vertex[28] = img_vertex[4] - border_height;
  border_vertex[29] = img_vertex[5];
  /* Vertex 10 */
  border_vertex[30] = img_vertex[9];
  border_vertex[31] = img_vertex[10];
  border_vertex[32] = img_vertex[11];
  /* Vertex 11 */
  border_vertex[33] = img_vertex[0] - border_width;
  border_vertex[34] = img_vertex[1] - border_height;
  border_vertex[35] = img_vertex[2];
}

/* Simply copies glesdrawable background vertices */
static void
set_image_standard_vertices (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesFloat *drb_vertex = glesdrawable->bg_vertex;
  PgmGlesFloat *img_vertex = glesimage->vertex;
  gfloat border_width, border_height;

  /* Get the border sizes avoiding overlapped borders */
  border_width = MIN (glesimage->border_width, glesdrawable->width / 2.0f);
  border_height = MIN (glesimage->border_width, glesdrawable->height / 2.0f);

  /* Copy the drawable vertices applying the border offset */
  img_vertex[0] = drb_vertex[0] + border_width;
  img_vertex[1] = drb_vertex[1] + border_height;
  img_vertex[2] = drb_vertex[2];
  img_vertex[3] = drb_vertex[3] - border_width;
  img_vertex[4] = img_vertex[1];
  img_vertex[5] = drb_vertex[5];
  img_vertex[6] = img_vertex[3];
  img_vertex[7] = drb_vertex[7] - border_height;
  img_vertex[8] = drb_vertex[8];
  img_vertex[9] = img_vertex[0];
  img_vertex[10] = img_vertex[7];
  img_vertex[11] = drb_vertex[11];

  /* Update the border vertices if necessary */
  if (glesimage->border_width > 0.0f)
    set_border_vertices (glesimage, border_width, border_height);
}

/* Modifies vertices for the scaled layout */
static void
set_image_scaled_vertices (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesFloat *drb_vertex = glesdrawable->bg_vertex;
  PgmGlesFloat *img_vertex = glesimage->vertex;
  gfloat border_width, border_height, offset;

  /* Drawable width larger than image width */
  if (glesimage->drawable_ratio >= glesimage->image_ratio)
    {
      offset = glesdrawable->width - glesdrawable->height
        * glesimage->image_ratio;

      /* Get the border sizes avoiding overlapped borders */
      border_width = MIN (glesimage->border_width,
                          (glesdrawable->width - offset) / 2.0f);
      border_height = MIN (glesimage->border_width, glesdrawable->height / 2.0f);

      img_vertex[0] = (drb_vertex[0] + offset *
                       glesimage->h_align) + border_width;
      img_vertex[1] = drb_vertex[1] + border_height;
      img_vertex[3] = (drb_vertex[3] - offset *
                       (1.0f - glesimage->h_align)) - border_width;
      img_vertex[4] = drb_vertex[4] + border_height;
      img_vertex[6] = img_vertex[3];
      img_vertex[7] = drb_vertex[7] - border_height;
      img_vertex[9] = img_vertex[0];
      img_vertex[10] = drb_vertex[10] - border_height;
    }
  /* Drawable height larger than image height */
  else
    {
      offset = glesdrawable->height - glesdrawable->width
        / glesimage->image_ratio;

      /* Get the border sizes avoiding overlapped borders */
      border_width = MIN (glesimage->border_width, glesdrawable->width / 2.0f);
      border_height = MIN (glesimage->border_width,
                           (glesdrawable->height - offset) / 2.0f);

      img_vertex[0] = drb_vertex[0] + border_width;
      img_vertex[1] = (drb_vertex[1] + offset *
                       glesimage->v_align) + border_height;
      img_vertex[3] = drb_vertex[3] - border_width;
      img_vertex[4] = img_vertex[1];
      img_vertex[6] = drb_vertex[6] - border_width;
      img_vertex[7] = (drb_vertex[7] - offset *
                       (1.0f - glesimage->v_align)) - border_height;
      img_vertex[9] = drb_vertex[9] + border_width;
      img_vertex[10] = img_vertex[7];
    }

  /* Then simply copy the z position from the background one */
  img_vertex[2] = drb_vertex[2];
  img_vertex[5] = drb_vertex[5];
  img_vertex[8] = drb_vertex[8];
  img_vertex[11] = drb_vertex[11];

  /* Update the border vertices if necessary */
  if (glesimage->border_width > 0.0f)
    set_border_vertices (glesimage, border_width, border_height);
}

/* Modifies texture coordinates for the zoomed layout */
static void
set_image_zoomed_coordinates (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesTexture *glestexture = glesimage->texture;
  gfloat max_s, max_t;
  gfloat offset;

  max_s = (gfloat) glestexture->width / glestexture->width_pot;
  max_t = (gfloat) glestexture->height / glestexture->height_pot;

  /* Adapt coordinates when drawable width is larger than image width */
  if (glesimage->drawable_ratio >= glesimage->image_ratio)
    {
      gfloat image_height = glesdrawable->width / glesimage->image_ratio;

      offset = ((image_height - glesdrawable->height) * max_t) / image_height;
      glesimage->coord[0] = 0.0f;
      glesimage->coord[1] = offset * glesimage->v_align;
      glesimage->coord[2] = max_s;
      glesimage->coord[3] = glesimage->coord[1];
      glesimage->coord[4] = max_s;
      glesimage->coord[5] = max_t - offset * (1.0f - glesimage->v_align);
      glesimage->coord[6] = 0.0f;
      glesimage->coord[7] = glesimage->coord[5];
    }
  /* Adapt coordinates when drawable height is larger than image height */
  else
    {
      gfloat image_width = glesdrawable->height * glesimage->image_ratio;

      offset = ((image_width - glesdrawable->width) * max_s) / image_width;
      glesimage->coord[0] = offset * glesimage->h_align;
      glesimage->coord[1] = 0.0f;
      glesimage->coord[2] = max_s - offset * (1.0f - glesimage->h_align);
      glesimage->coord[3] = 0.0f;
      glesimage->coord[4] = glesimage->coord[2];
      glesimage->coord[5] = max_t;
      glesimage->coord[6] = glesimage->coord[0];
      glesimage->coord[7] = max_t;
    }
}

/* Modifies texture coordinates to fill the whole stored texture */
static void
set_image_standard_coordinates (PgmGlesImage *glesimage)
{
  PgmGlesTexture *glestexture = glesimage->texture;

  glesimage->coord[0] = 0.0f;
  glesimage->coord[1] = 0.0f;
  glesimage->coord[2] = (gfloat) glestexture->width / glestexture->width_pot;
  glesimage->coord[3] = 0.0f;
  glesimage->coord[4] = glesimage->coord[2];
  glesimage->coord[5] = (gfloat) glestexture->height / glestexture->height_pot;
  glesimage->coord[6] = 0.0f;
  glesimage->coord[7] = glesimage->coord[5];
}

/* Adapts properties to a size change depending on the layout */
static void
update_layout (PgmGlesImage *glesimage)
{
  PgmDrawable *drawable = PGM_GLES_DRAWABLE (glesimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageLayoutType layout;

  GST_OBJECT_LOCK (image);
  layout = image->layout;
  GST_OBJECT_UNLOCK (image);

  switch (layout)
    {
    case PGM_IMAGE_SCALED:
    case PGM_IMAGE_CENTERED: /* FIXME: Not implemented yet */
    case PGM_IMAGE_TILED:    /* FIXME: Not implemented yet */
      set_image_scaled_vertices (glesimage);
      set_image_standard_coordinates (glesimage);
      break;

    case PGM_IMAGE_ZOOMED:
      set_image_standard_vertices (glesimage);
      set_image_zoomed_coordinates (glesimage);
      break;

    case PGM_IMAGE_FILLED:
      set_image_standard_vertices (glesimage);
      set_image_standard_coordinates (glesimage);
      break;

    default:
      break;
    }
}

/* Adapts to a drawable position change */
static void
update_vertices_position (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesFloat *img_vertex = glesimage->vertex;
  PgmGlesFloat *img_border = glesimage->border_vertex;
  gfloat x_offset, y_offset, z_offset;
  guint i;

  x_offset = glesdrawable->x - glesimage->last_x;
  y_offset = glesdrawable->y - glesimage->last_y;
  z_offset = glesdrawable->z - glesimage->last_z;

  /* Add the offsets to the current vertices */
  for (i = 0; i < 12; i += 3)
    {
      img_vertex[i] += x_offset;
      img_vertex[i+1] += y_offset;
      img_vertex[i+2] += z_offset;
    }

  /* Add the offsets to the border vertices, if necessary */
  if (glesimage->border_width > 0.0f)
    for (i = 0; i < 30; i += 3)
      {
        img_border[i] += x_offset;
        img_border[i+1] += y_offset;
        img_border[i+2] += z_offset;
      }

  /* And update the last position with the new one */
  update_last_position (glesimage);
}

/* Update the border_width value stored in the GL image */
static void
update_border_width (PgmGlesImage *glesimage)
{
  PgmDrawable *drawable = PGM_GLES_DRAWABLE (glesimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);

  GST_OBJECT_LOCK (image);
  glesimage->border_width = image->border_width;
  GST_OBJECT_UNLOCK (image);
}

/* Update the image ratio and layout of all the slaves of the given image */
static void
update_slaves (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;
  PgmImage *master = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesImage *slave;
  GList *walk;

  GST_OBJECT_LOCK (master);

  walk = master->slaves;
  while (walk)
    {
      /* Retrieve the slave glesimage */
      GST_OBJECT_LOCK (glesviewport);
      slave = g_hash_table_lookup (glesviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glesviewport);

      /* Update slave glesimage parameters */
      if (slave)
        {
          GST_OBJECT_UNLOCK (master);
          update_image_ratio (slave);
          update_layout (slave);
          GST_OBJECT_LOCK (master);
        }

      walk = walk->next;
    }

  GST_OBJECT_UNLOCK (master);
}

/* PgmGlesDrawable methods */

static void
pgm_gles_image_draw (PgmGlesDrawable *glesdrawable)
{
  static PgmGlesUshort border_indices[] =
    { 0, 1, 2, 2, 3, 0, 2, 4, 5, 5, 6, 2, 5, 7, 8, 8, 9, 5, 8, 10, 0, 0, 11, 8 };
  static PgmGlesUshort indices[] =
    { 0, 1, 2, 2, 3, 0 };
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);
  PgmGlesContextProcAddress *gles;

  GST_LOG_OBJECT (glesdrawable, "draw");

  /* Avoid drawing if it's not visible */
  if (glesimage->empty || (glesimage->fg_color[3] == 0.0f))
    return;

  gles = glesdrawable->glesviewport->context->gles;

  gles->enable_client_state (PGM_GLES_VERTEX_ARRAY);
  gles->enable_client_state (PGM_GLES_COLOR_ARRAY);
  gles->enable_client_state (PGM_GLES_TEXTURE_COORD_ARRAY);

  /* Upload vertices and texture coordinates for the image */
  gles->vertex_pointer (3, PGM_GLES_FLOAT, 0, glesimage->vertex);
  gles->color_pointer (4, PGM_GLES_FLOAT, 0, glesimage->fg_color);
  gles->tex_coord_pointer (2, PGM_GLES_FLOAT, 0, glesimage->coord);

  /* Draw the image */
  pgm_gles_texture_bind (glesimage->texture);
  gles->draw_elements (PGM_GLES_TRIANGLES, 6,
                       PGM_GLES_UNSIGNED_SHORT, indices);
  pgm_gles_texture_unbind (glesimage->texture);

  gles->disable_client_state (PGM_GLES_TEXTURE_COORD_ARRAY);

  if (glesimage->border_width > 0.0f)
    {
      /* Upload vertices and colors for the border */
      gles->vertex_pointer (3, PGM_GLES_FLOAT, 0, glesimage->border_vertex);
      gles->color_pointer (4, PGM_GLES_FLOAT, 0, glesimage->border_color);

      /* Draw the border */
      gles->draw_elements (PGM_GLES_TRIANGLES, 24,
                           PGM_GLES_UNSIGNED_SHORT, border_indices);
    }

  gles->disable_client_state (PGM_GLES_VERTEX_ARRAY);
  gles->disable_client_state (PGM_GLES_COLOR_ARRAY);
}

static void
pgm_gles_image_regenerate (PgmGlesDrawable *glesdrawable)
{
  GST_LOG_OBJECT (glesdrawable, "regenerate");
}

static void
pgm_gles_image_update_projection (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);

  GST_LOG_OBJECT (glesdrawable, "update_projection");

  update_image_ratio (glesimage);
  update_layout (glesimage);
}

static void
pgm_gles_image_set_size (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);

  GST_DEBUG_OBJECT (glesdrawable, "set_size");

  update_drawable_ratio (glesimage);
  update_layout (glesimage);
}

static void
pgm_gles_image_set_position (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);

  GST_DEBUG_OBJECT (glesdrawable, "set_position");

  update_vertices_position (glesimage);
}

static void
pgm_gles_image_set_fg_color (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);
  PgmDrawable *drawable;
  PgmGlesFloat color[4];

  GST_DEBUG_OBJECT (glesdrawable, "set_fg_color");

  drawable = glesdrawable->drawable;

  color[0] = drawable->fg_r * INV_255;
  color[1] = drawable->fg_g * INV_255;
  color[2] = drawable->fg_b * INV_255;
  color[3] = drawable->fg_a * drawable->opacity * INV_255;

  GST_OBJECT_LOCK (drawable);
  glesimage->fg_color[0] = color[0];
  glesimage->fg_color[1] = color[1];
  glesimage->fg_color[2] = color[2];
  glesimage->fg_color[3] = color[3];
  glesimage->fg_color[4] = color[0];
  glesimage->fg_color[5] = color[1];
  glesimage->fg_color[6] = color[2];
  glesimage->fg_color[7] = color[3];
  glesimage->fg_color[8] = color[0];
  glesimage->fg_color[9] = color[1];
  glesimage->fg_color[10] = color[2];
  glesimage->fg_color[11] = color[3];
  glesimage->fg_color[12] = color[0];
  glesimage->fg_color[13] = color[1];
  glesimage->fg_color[14] = color[2];
  glesimage->fg_color[15] = color[3];
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gles_image_set_opacity (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);
  PgmDrawable *drawable = glesdrawable->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmGlesFloat *border_color = glesimage->border_color;
  gfloat fg_opacity, border_opacity;
  guint i;

  GST_DEBUG_OBJECT (glesdrawable, "set_opacity");

  /* Get the opacities */
  GST_OBJECT_LOCK (drawable);
  fg_opacity = drawable->fg_a * drawable->opacity * SQR_INV_255;
  border_opacity = image->border_inner_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Image alpha blending */
  glesimage->fg_color[3] = fg_opacity;
  glesimage->fg_color[7] = fg_opacity;
  glesimage->fg_color[11] = fg_opacity;
  glesimage->fg_color[15] = fg_opacity;

  /* Border alpha blending */
  for (i = 3; i < 48; i += 4)
    border_color[i] = border_opacity;
}

static void
pgm_gles_image_sync (PgmGlesDrawable *glesdrawable)
{
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (glesdrawable);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmImageStorageType type;

  GST_LOG_OBJECT (glesdrawable, "sync");

  /* Synchronize various properties */
  pgm_gles_image_set_fg_color (glesdrawable);
  update_interp (glesimage);
  update_wrapping (glesimage);
  update_alignment (glesimage);
  update_last_position (glesimage);
  update_drawable_ratio (glesimage);
  update_image_ratio (glesimage);
  update_mapping_matrix (glesimage);
  update_border_width (glesimage);
  update_border_inner_color (glesimage);
  update_border_outer_color (glesimage);

  GST_OBJECT_LOCK (image);
  type = image->storage_type;
  GST_OBJECT_UNLOCK (image);

  /* Synchronize the buffer */
  switch (type)
    {
    case PGM_IMAGE_FILE:
      pgm_gles_image_set_from_file (glesimage);
      break;
    case PGM_IMAGE_BUFFER:
      pgm_gles_image_set_from_buffer (glesimage);
      break;
    case PGM_IMAGE_GST_BUFFER:
      pgm_gles_image_set_from_gst_buffer (glesimage);
      break;
    case PGM_IMAGE_PIXBUF:
      pgm_gles_image_set_from_pixbuf (glesimage);
      break;
    case PGM_IMAGE_IMAGE:
      pgm_gles_image_set_from_image (glesimage);
      break;
    case PGM_IMAGE_EMPTY:
      break;
    default:
      break;
    }
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlesImage, pgm_gles_image, PGM_TYPE_GLES_DRAWABLE);

void
pgm_gles_image_register (GTypeModule *module)
{
  pgm_gles_image_register_type (module);
}

static void
pgm_gles_image_dispose (GObject *object)
{

  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (object);
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;
  PgmGlesImage *glesimage = PGM_GLES_IMAGE (object);
  PgmImage *master = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesImage *slave;
  GList *walk;

  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "dispose");

  /* Clean up the slaves */
  GST_OBJECT_LOCK (master);
  walk = master->slaves;
  while (walk)
    {
      GST_OBJECT_LOCK (glesviewport);
      slave = g_hash_table_lookup (glesviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glesviewport);

      if (slave)
        {
          slave->empty = TRUE;
          slave->texture = slave->native_texture;
          slave->image_ratio = 0.0f;
        }

      walk = walk->next;
    }
  GST_OBJECT_UNLOCK (master);

  gst_object_unref (glesdrawable->drawable);
  pgm_gles_context_remove_tasks_with_data
    (glesdrawable->glesviewport->context, glesimage->native_texture);
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_FREE_TEXTURE,
                                    glesimage->native_texture);
  pgm_gles_context_push_immediate_task
    (glesdrawable->glesviewport->context, task);
  glesimage->native_texture = NULL;
  glesimage->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gles_image_class_init (PgmGlesImageClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlesDrawableClass *glesdrawable_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gles_image_debug, "pgm_gles_image", 0,
                           "OpenGL ES plugin: PgmGlesImage");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  glesdrawable_class = PGM_GLES_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gles_image_dispose);

  /* PgmGlesDrawable virtual table */
  glesdrawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gles_image_sync);
  glesdrawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gles_image_draw);
  glesdrawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gles_image_regenerate);
  glesdrawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gles_image_update_projection);
  glesdrawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gles_image_set_size);
  glesdrawable_class->set_position =
    GST_DEBUG_FUNCPTR (pgm_gles_image_set_position);
  glesdrawable_class->set_fg_color =
    GST_DEBUG_FUNCPTR (pgm_gles_image_set_fg_color);
  glesdrawable_class->set_opacity =
    GST_DEBUG_FUNCPTR (pgm_gles_image_set_opacity);
}

static void
pgm_gles_image_class_finalize (PgmGlesImageClass *klass)
{
  return;
}

static void
pgm_gles_image_init (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "init");

  glesimage->empty = TRUE;
  glesimage->border_width = 0.0f;
}

/* Public methods */

PgmGlesDrawable *
pgm_gles_image_new (PgmDrawable *drawable,
                    PgmGlesViewport *glesviewport)
{
  PgmImage *master = PGM_IMAGE (drawable);
  PgmGlesImage *slave;
  PgmGlesImage *glesimage;
  PgmGlesDrawable *glesdrawable;
  GList *walk;

  glesimage = g_object_new (PGM_TYPE_GLES_IMAGE, NULL);

  GST_DEBUG_OBJECT (glesimage, "created new glesimage");

  glesimage->native_texture = pgm_gles_texture_new (glesviewport->context);
  glesimage->texture = glesimage->native_texture;

  glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  glesdrawable->drawable = gst_object_ref (drawable);
  glesdrawable->glesviewport = glesviewport;
  pgm_gles_viewport_connect_changed_callback (glesviewport, glesdrawable);
  pgm_gles_drawable_sync (glesdrawable);

  /* Update the slaves that are in the Canvas */
  GST_OBJECT_LOCK (master);
  walk = master->slaves;
  while (walk)
    {
      GST_OBJECT_LOCK (glesviewport);
      slave = g_hash_table_lookup (glesviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glesviewport);

      if (slave)
        {
          slave->empty = FALSE;
          slave->texture = glesimage->texture;
          update_image_ratio (slave);
          update_layout (slave);
        }

      walk = walk->next;
    }
  GST_OBJECT_UNLOCK (master);

  return glesdrawable;
}

void
pgm_gles_image_clear (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesContextTask *task = NULL;

  GST_DEBUG_OBJECT (glesimage, "clear");

  if (glesimage->empty)
    return;

  /* Request texture clean up in case it was not using a master texture */
  if (glesimage->texture == glesimage->native_texture)
    {
      task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_CLEAN_TEXTURE,
                                        glesimage->texture);
      pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                            task);
    }
  else
    glesimage->texture = glesimage->native_texture;

  GST_OBJECT_LOCK (glesimage);

  glesimage->empty = TRUE;
  glesimage->image_ratio = 0.0f;

  GST_OBJECT_UNLOCK (glesimage);
}

void
pgm_gles_image_set_from_file (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_from_file");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_FILE
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  if (image->data.file.pixbuf == NULL)
    {
      GST_OBJECT_UNLOCK (image);
      _pgm_image_stored_from_file_load (image);
      return;
    }

  pgm_gles_texture_set_pixbuf (glesimage->texture, image->data.file.pixbuf);
  pgm_gles_texture_set_matrix (glesimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  _pgm_image_stored_from_file_free (image);

  glesimage->empty = FALSE;
  update_image_ratio (glesimage);
  update_layout (glesimage);
  update_slaves (glesimage);

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_GEN_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPLOAD_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_deferred_task (glesdrawable->glesviewport->context,
                                       task);
}

void
pgm_gles_image_set_from_buffer (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_from_buffer");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_BUFFER
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  pgm_gles_texture_set_buffer (glesimage->texture, image->data.buffer.buffer,
                               image->data.buffer.format,
                               image->data.buffer.width,
                               image->data.buffer.height,
                               image->data.buffer.size,
                               image->data.buffer.stride);
  pgm_gles_texture_set_matrix (glesimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  glesimage->empty = FALSE;

  update_image_ratio (glesimage);
  update_layout (glesimage);
  update_slaves (glesimage);

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_GEN_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPLOAD_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_deferred_task (glesdrawable->glesviewport->context,
                                       task);
}

void
pgm_gles_image_set_from_gst_buffer (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_from_gst_buffer");

  /* It's the first time the glesimage receives a GstBuffer, let's set the
   * texture buffer with its different informations */
  if (G_UNLIKELY (glesimage->empty))
    {
      GST_OBJECT_LOCK (image);

      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }

      pgm_gles_texture_set_gst_buffer (glesimage->texture,
                                       image->data.gst_buffer.gst_buffer,
                                       image->data.gst_buffer.format,
                                       image->data.gst_buffer.width,
                                       image->data.gst_buffer.height,
                                       image->data.gst_buffer.stride);
      pgm_gles_texture_set_matrix (glesimage->texture, image->mapping_matrix);

      GST_OBJECT_UNLOCK (image);

      glesimage->empty = FALSE;
      update_image_ratio (glesimage);
      update_layout (glesimage);
      update_slaves (glesimage);

      task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_GEN_TEXTURE,
                                        glesimage->texture);
      pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                            task);
    }

  /* It's not the first time, so we just need to send the updated buffer */
  else
    {
      GST_OBJECT_LOCK (image);

      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }

      pgm_gles_texture_update_gst_buffer (glesimage->texture,
                                          image->data.gst_buffer.gst_buffer);
      GST_OBJECT_UNLOCK (image);
    }

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPLOAD_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_deferred_task (glesdrawable->glesviewport->context,
                                       task);
}

void
pgm_gles_image_set_from_pixbuf (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_from_pixbuf");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_PIXBUF
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  pgm_gles_texture_set_pixbuf (glesimage->texture, image->data.pixbuf.pixbuf);
  pgm_gles_texture_set_matrix (glesimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  glesimage->empty = FALSE;
  update_image_ratio (glesimage);
  update_layout (glesimage);
  update_slaves (glesimage);

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_GEN_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPLOAD_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_deferred_task (glesdrawable->glesviewport->context,
                                       task);
}

void
pgm_gles_image_set_from_image (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmImage *image = PGM_IMAGE (glesdrawable->drawable);
  PgmGlesViewport *glesviewport = glesdrawable->glesviewport;
  PgmGlesImage *master;

  GST_DEBUG_OBJECT (glesimage, "set_from_image");

  /* Retrieve the master glesimage */
  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (image->storage_type != PGM_IMAGE_IMAGE))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  GST_OBJECT_LOCK (glesviewport);
  master = g_hash_table_lookup (glesviewport->drawable_hash, image->master);
  GST_OBJECT_UNLOCK (glesviewport);
  GST_OBJECT_UNLOCK (image);

  /* And use its texture */
  if (master)
    {
      glesimage->texture = master->texture;
      glesimage->empty = FALSE;
      update_image_ratio (glesimage);
      update_layout (glesimage);
    }
}

void
pgm_gles_image_set_mapping_matrix (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_mapping_matrix");

  update_mapping_matrix (glesimage);
}

void
pgm_gles_image_set_alignment (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_alignment");

  update_alignment (glesimage);
  update_layout (glesimage);
}

void
pgm_gles_image_set_layout (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_layout");

  update_layout (glesimage);
}

void
pgm_gles_image_set_interp (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_interp");

  update_interp (glesimage);

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPDATE_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
}

void
pgm_gles_image_set_wrapping (PgmGlesImage *glesimage)
{
  PgmGlesDrawable *glesdrawable = PGM_GLES_DRAWABLE (glesimage);
  PgmGlesContextTask *task;

  GST_DEBUG_OBJECT (glesimage, "set_wrapping");

  update_wrapping (glesimage);

  task = pgm_gles_context_task_new (PGM_GLES_CONTEXT_UPDATE_TEXTURE,
                                    glesimage->texture);
  pgm_gles_context_push_immediate_task (glesdrawable->glesviewport->context,
                                        task);
}

void
pgm_gles_image_set_aspect_ratio (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_aspect_ratio");

  update_image_ratio (glesimage);
  update_layout (glesimage);
}

void
pgm_gles_image_set_border_width (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_border_width");

  update_border_width (glesimage);
  update_layout (glesimage);
}

void
pgm_gles_image_set_border_inner_color (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_border_inner_color");

  update_border_inner_color (glesimage);
}

void
pgm_gles_image_set_border_outer_color (PgmGlesImage *glesimage)
{
  GST_DEBUG_OBJECT (glesimage, "set_border_outer_color");

  update_border_outer_color (glesimage);
}
