#!/bin/bash
#
# Resize the OS partition on 8GB and 16GB systems that
# were incorrectly formatted as 4GB disks at the factory.
#
# After resizing the OS partition and rebooting,
# we run 'resize2fs' to make the filesystem take advantage
# of the extra space in the partition (see the runresize2fs
# init script in this package)
#
# Part of a fix for LP: #283494
#
# Thanks to trebonius and the other users 
# on ubuntuforums.org

set -e

die() {
    echo "Error: $1"
    exit 1
}

alldone() {
    echo "Nothing to do: $1"
    exit 0
}

check_existing_partitions() {
    # Is there a disk?
    if [ ! -b /dev/sda ]; then
	alldone "Couldn't find /dev/sda"
    fi

    # Make sure the disk is the internal drive
    if [ `cat /sys/block/sda/removable` != 0 ]; then
	alldone "/dev/sda is a removable drive"
    fi

    # Are we on an 8GB or 16GB unit? (size in 1KB blocks)
    DISKSIZE=`sfdisk -s /dev/sda`
    if [ $DISKSIZE -gt 7000000 ]; then
	if [ $DISKSIZE -lt 8500000 ]; then
	    EIGHTGB=1
	elif [ $DISKSIZE -gt 14500000 -a $DISKSIZE -lt 16500000 ]; then
	    SIXTEENGB=1
	else
	    alldone "Disk is not 8GB or 16GB"
	fi
    else
	alldone "Disk is not 8GB or 16GB"
    fi

    # How many partitions?
    # Some people probably created a new partition with the
    # empty space, we don't want to mess with those systems!
    EXPECTED=2
    PARTCOUNT=`fdisk -l /dev/sda | grep "^/dev/sda" | wc -l`
    if [ $PARTCOUNT -ne $EXPECTED ]; then
	alldone "Expected $EXPECTED partitions on /dev/sda, found $PARTCOUNT"
    fi

    # Is the utility partition still first in line?
    PARTTYPE=`sfdisk -c /dev/sda 1`
    if [ "x$PARTTYPE" != "xde" ]; then
	alldone "First partition is not a utility partition"
    fi

    # Is the second partition the bootable OS partition?
    BOOTABLE=`sfdisk -d | grep sda2 | awk '{print $8}'`
    if [ "x$BOOTABLE" != "xbootable" ]; then
	alldone "Second partition is not bootable, how did we get here?"
    fi

    # Is the second partition ext3?
    PARTTYPE=`sfdisk -c /dev/sda 2`
    if [ "x$PARTTYPE" != "x83" ]; then
	alldone "Second partition is not a Linux partition"
    fi
    if [ `vol_id /dev/sda2 | grep -c "ID_FS_TYPE=ext3"` != 1 ]; then
	alldone "Second partition is not formatted as ext3"
    fi

    # Is the OS partition the wrong size? (size in 1KB blocks)
    PARTSIZE=`sfdisk -s /dev/sda2`
    if [ "x$EIGHTGB" = "x1" ]; then
	if [ $PARTSIZE -gt 7000000 ]; then
	    alldone "8GB unit is partitioned correctly"
	fi
    elif [ "x$SIXTEENGB" = "x1" ]; then
	if [ $PARTSIZE -gt 14500000 ]; then
	    alldone "16GB unit is partitioned correctly"
	fi
    else
	die "Not an 8GB or 16GB unit" # this shouldn't happen
    fi
}

reformat_disk() {
    # Collect starting sector of both partitions and the size of
    # the utility partition
    START1=`sfdisk -d /dev/sda | grep sda1 | awk '{print $4}' | sed 's/,//'`
    SIZE1=`sfdisk -d /dev/sda | grep sda1 | awk '{print $6}' | sed 's/,//'`
    START2=`sfdisk -d /dev/sda | grep sda2 | awk '{print $4}' | sed 's/,//'`
    sync
    # Enlarge the OS partition to fill the rest of the disk while
    # maintaining the position/size of the utility partition and 
    # the position of the OS partition.
    sfdisk -uS --no-reread -q /dev/sda <<EOF
$START1,$SIZE1,de,-
$START2,,L,*
EOF
}


check_existing_partitions
reformat_disk
exit 0
