# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.plugins.pigment.widgets.widget import Widget

from elisa.plugins.pigment.graph.text import Text
from pgm.timing.implicit import AnimatedObject

import gobject
import pgm

import logging


class Entry(Widget):
    """
    A text entry widget, to get single line input (see gtk.Entry).

    Emit these signals:
      - activate: when the Enter key is pressed
      - changed: when the text in the entry has changed

    @ivar text: the content of the entry
    @type text: unicode

    """

    __gsignals__ = {'activate': (gobject.SIGNAL_RUN_LAST,
                                 gobject.TYPE_BOOLEAN,
                                 ()),
                    'changed': (gobject.SIGNAL_RUN_LAST,
                                 gobject.TYPE_BOOLEAN,
                                 ())}

    def __init__(self):
        super(Entry, self).__init__()

        self._text = Text()
        self._text.font_family = "Bitstream DejaVu"
        self._text.font_height = 1.0
        self._text.bg_color = (255, 0, 255, 255)
        self._text.fg_color = (0, 255, 255, 255)
        self._text.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._text.visible = True

        self.add(self._text)

    def clean(self):
        self._text.clean()
        self._text = None

        return super(Entry, self).clean()

    def text__get(self):
        """Get the label's text"""
        return unicode(self._text.label)

    def text__set(self, value):
        """Set the label's text, emitting the 'changed' signal if necessary"""
        value = value or ''

        if self._text.label != value:
            self._text.label = value
            self.emit('changed')

    text = property(text__get, text__set)

    def do_clicked(self, x, y, z, button, time, pressure):
        self.emit('activate')

        self.focus = True
        return True

    def do_activate(self):
        logging.debug('Activate signal')

    def do_changed(self):
        logging.debug('Changed signal')

    def do_key_press_event(self, viewport, event, widget):
        """Take care to filter out modifiers"""

        if event.keyval == pgm.keysyms.Return:
            self.emit('activate')
        elif event.keyval == pgm.keysyms.BackSpace:
            self.text = self.text[0:-1]
        elif event.keyval in [pgm.keysyms.Alt_L,
                              pgm.keysyms.Alt_R,
                              pgm.keysyms.Shift_L,
                              pgm.keysyms.Shift_R,
                              pgm.keysyms.Caps_Lock,
                              pgm.keysyms.Control_L,
                              pgm.keysyms.Control_R]:
            pass
        else:
            self.text = self.text + unichr (event.keyval)

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        widget = cls()
        widget.text = "ciao"
        widget.visible = True

        return widget


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    entry = Entry.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()

