# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.log import Loggable
from elisa.core.common import application
from elisa.core.utils import defer, cancellable_defer
from elisa.core.application import PICTURES_CACHE
from elisa.core.media_uri import MediaUri

from elisa.plugins.base.models.file import FileModel
from elisa.plugins.base.models.media import RawDataModel

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.list import List
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.pigment.widgets.list_horizontal import ListHorizontal
from elisa.plugins.poblesec.player_video import PlayerOsd as VideoPlayerOsd
from elisa.plugins.poblesec.videoplayer_controls import PlayPauseControl, \
                                                        StopControl
from elisa.plugins.poblesec.slideshow_controls import ShowNextControl, \
                                                      ShowPreviousControl, \
                                                      RotationControl
from elisa.plugins.poblesec.widgets.long_loading_image import LongLoadingImage
from elisa.plugins.poblesec.widgets.player.status_display import \
                                                      PictureStatusDisplay, BARS

from elisa.plugins.poblesec.widgets.player.control_ribbon import Control
from elisa.plugins.pigment.widgets.const import *

from elisa.core.utils.i18n import install_translation
from elisa.core import common

_ = install_translation('poblesec')

from twisted.internet import reactor
import gobject, os, os.path, md5, time
import math
import pgm
from pgm.utils import maths


from gtk import gdk

try:
    from elisa.plugins.database import models as db_models
except ImportError:
    # database missing
    db_models = None
else:
    DBFile = db_models.File
    DBImage = db_models.Image
    from elisa.plugins.base.models import image


class PlayerOsd(VideoPlayerOsd):

    status_widget = PictureStatusDisplay


class ListFading(List):

    preloaded = 1

    def __init__(self):
        super(ListFading, self).__init__(widget_class=LongLoadingImage,
                visible_range_size=1)
        self.focus_on_click = False

    def _piecewise_interpolation(self, x, y, factor):
        # specific transformation to map the list items with the given
        # segments to interpolate for
        factor += 0.5
        t = self._visible_range_size
        x = map(lambda a: t*a, x)

        # clamp after lower and upper limits
        if factor < x[0]:
            return y[0]
        elif factor > x[-1]:
            return y[-1]
        else:
            # general case: looking for the segment where factor belongs
            i = 0
            while factor > x[i+1]:
                i += 1

            # factor must be between 0.0 and 1.0
            new_factor = (factor - x[i]) / (x[i+1] - x[i])
            return maths.lerp(y[i], y[i+1], new_factor)

    def do_item_clicked(self, item):
        # avoid default behaviour when an item is clicked: selecting that item
        return True

    def _prepare_widget(self, widget):
        widget.position = (0.0, 0.0, 0.0)
        widget.size = (1.0, 1.0)

    def _layout_widget(self, widget, position):
        widget.opacity = self.compute_opacity(position)

    def compute_opacity(self, index):
        x = [0.0, 0.5, 1.0]
        y = [0, 255, 0]

        return self._piecewise_interpolation(x, y, index)

    def _selected_to_range_start(self, selected):
        visible_range_start = selected
        return visible_range_start

    def _range_start_to_selected(self, visible_range_start):
        selected = visible_range_start
        selected = int(round(selected))
        selected = maths.clamp(selected, 0, len(self.model)-1)
        return selected

def cancel_deferred(widget, name):
    if hasattr(widget, name):
        dfr = getattr(widget, name)
        if dfr != None and not dfr.called:
            # Swallow the error raised by the cancellation
            def _failure(failure):
                failure.trap(cancellable_defer.CancelledError)
            dfr.addErrback(_failure)
            dfr.cancel()
        delattr(widget, name)

class Slideshow(gobject.GObject, Loggable):
    """
    @ivar status:   DOCME
    @type status:   one of [Slideshow.STOPPED, Slideshow.PLAYING, Slideshow.PAUSED]
    @ivar duration: delay between two pictures (in seconds)
    @type duration: float
    """

    STOPPED = 0
    PLAYING = 1
    PAUSED = 2

    __gsignals__ = {'current-index-changed': (gobject.SIGNAL_RUN_LAST,
                                              gobject.TYPE_NONE,
                                          (gobject.TYPE_INT, gobject.TYPE_INT)),
                    'status-changed': (gobject.SIGNAL_RUN_LAST,
                                      gobject.TYPE_BOOLEAN,
                                      (gobject.TYPE_INT,)),
                   }

    def __init__(self):
        super(Slideshow, self).__init__()
        self.logCategory = "slideshow"

        self.status = self.STOPPED
        self.duration = 5.0
        self._call_later = None
        self.playlist = notifying_list.List()
        self.current_index = -1

        self.background = ListFading()

        def on_selected_item_changed(background, item, prev_item):
            if item:
                # item is None when the new selected index is -1 (which happens
                # when the model is empty)
                index = self.background.model.index(item)
                self._set_current_index(index)

        self.background.connect('selected-item-changed',
                                on_selected_item_changed)

        self.background.set_model(self.playlist)
        self.background.set_renderer(self._renderer)
        self.background.visible = True

        # pictures cache directory management
        try:
            os.makedirs(PICTURES_CACHE)
        except OSError:
            # cache directory already existing
            pass

    def _renderer(self, item, widget):

        # should cancel previous dfr_thumbnail and dfr_full created for widget
        cancel_deferred(widget, "dfr_thumbnail")
        cancel_deferred(widget, "dfr_full")

        widget.clear()
        self._set_default_icon(widget)

        if item.references:
            dfr_thumbnail = self._retrieve_thumbnail(item)
            dfr_thumbnail.addCallback(self._load_thumbnail, widget)
            dfr_full = self._retrieve_full_resolution(item)
            dfr_full.addCallback(self._load_full_resolution, widget, dfr_thumbnail)
            # FIXME: storing the deferred in widget seems ugly
            setattr(widget, 'dfr_thumbnail', dfr_thumbnail)
            setattr(widget, 'dfr_full', dfr_full)
        else:
            # We have not got the list of URLs for the image yet, we may be
            # waiting for it (asynchronous filling, see Flickr).
            if hasattr(item, '_references_deferred'):
                # FIXME: Flickr specific hack
                def got_image_sizes(images):
                    delattr(item, '_references_deferred')
                    if self.current_index == self.playlist.index(item):
                        dfr_thumbnail = self._retrieve_thumbnail(item)
                        dfr_thumbnail.addCallback(self._load_thumbnail, widget)
                        dfr_full = self._retrieve_full_resolution(item)
                        dfr_full.addCallback(self._load_full_resolution, widget, dfr_thumbnail)
                        # FIXME: storing the deferred in widget seems ugly
                        setattr(widget, 'dfr_thumbnail', dfr_thumbnail)
                        setattr(widget, 'dfr_full', dfr_full)
                item._references_deferred.addCallback(got_image_sizes)

    def _set_default_icon(self, widget):
        icon = "elisa.plugins.poblesec.file_picture"
        if hasattr(self, "_load_from_theme"):
            self._load_from_theme(icon, widget.quick_image)

    def _retrieve_thumbnail(self, item):
        uri = item.references[0]
        self.debug("retrieving thumbnail image from %s" % uri)
        return self._retrieve_and_cache(uri)

    def _retrieve_full_resolution(self, item):
        uri = item.references[-1]
        self.debug("retrieving full resolution image from %s" % uri)
        return self._retrieve_and_cache(uri)

    def _update_playcount_and_time(self, uri):
        if uri.scheme != 'file':
            # database only support files
            return

        def update(result):
            if not result:
                return

            self.debug("updating playcount")
            result.playcount += 1
            result.last_played = time.time()

        store = common.application.store

        if not store or not db_models:
            return

        dfr = store.get(DBFile, uri.path)
        dfr.addCallback(update)
        return dfr

    def _update_orientation(self, uri, index):
        if uri.scheme != 'file':
            # database only support files
            return

        store = common.application.store

        if not store or not db_models:
            return

        def update(result):
            if not result:
                return

            orientation = result.orientation
            if not orientation:
                orientation = image.ROTATED_0

            self.debug("updating rotation %r on %r", orientation, uri)

            bg_list = self.background
            try:
                widget_index = bg_list._widget_index_from_item_index(index)
                long_loading_image = bg_list._widgets[widget_index]

                self.apply_orientation(long_loading_image.image, orientation)
                self.apply_orientation(long_loading_image.quick_image,
                                   orientation)
            except IndexError:
                # the callback came after the image was already removed again.
                # This happens often when the user is going through the images
                # very fast.
                self.debug("Updating orientation failed. The widget is not " \
                            "loaded anymore")
                return

            model = self.playlist[index]
            model.orientation = orientation
            store.commit()


        dfr = store.get(DBImage, uri.path)
        dfr.addCallback(update)
        return dfr

    def _retrieve_and_cache(self, uri):
        path = self._compute_cached_path(uri)

        if os.path.exists(path):
            self.debug("%s already cached at %s", uri, path)
            # already cached
            dfr = defer.Deferred()
            dfr.callback(path)
        else:
            def cache_it(result):
                if isinstance(result, RawDataModel):
                    self._cache_raw_data(result.data, path)
                    return path
                elif isinstance(result, FileModel):
                    return result.uri.path

            model, dfr = application.resource_manager.get(uri)
            dfr.addCallback(cache_it)

        return dfr

    def _load_thumbnail(self, path, widget):
        # downscaling to 1024 is our safety net here in case some broken
        # resource provider is used
        widget.quick_image.set_from_file(path, 1024)

    def _load_full_resolution(self, path, widget, dfr_thumbnail=None):
        if dfr_thumbnail != None and not dfr_thumbnail.called \
            and isinstance(dfr_thumbnail, cancellable_defer.CancellableDeferred):
            dfr_thumbnail.cancel()

        # FIXME: should downscale the image on loading to a size never bigger
        # than the viewport size nor 1024
        widget.image.set_from_file(path, 1024)

    def _compute_cached_path(self, uri):
        filename = md5.new(str(uri)).hexdigest() + '.' + uri.extension
        path = os.path.join(PICTURES_CACHE, filename)
        return path

    def _cache_raw_data(self, data, path):
        self.debug("caching image at %s", path)
        if not os.path.exists(PICTURES_CACHE):
            os.makedirs(PICTURES_CACHE, 0755)
        fd = open(path, 'wb')
        fd.write(data)
        fd.close()

    def _cancel_last_call_later(self):
        if self._call_later != None and self._call_later.active():
            self._call_later.cancel()

    def apply_orientation(self, img, orientation):
        """
        DOCME
        """
        settings = image.IMAGE_ORIENTATIONS[orientation]
        rotation_settings = settings[1]
        self.debug("Rotation %r on %r", settings[0], img)
        for name, value in rotation_settings.iteritems():
            setattr(img, name, value)

    def apply_pixbuf_orientation(self, uri, img, orientation):
        if uri.scheme == 'file':
            self.debug("Pixbuf rotation %r on %r img=%r", orientation, uri.path, img)
            pixbuf = gdk.pixbuf_new_from_file(uri.path)
            pixbuf_rotations = {image.ROTATED_0: gdk.PIXBUF_ROTATE_NONE,
                                image.ROTATED_90_CCW: gdk.PIXBUF_ROTATE_COUNTERCLOCKWISE,
                                image.ROTATED_180: gdk.PIXBUF_ROTATE_UPSIDEDOWN,
                                image.ROTATED_90_CW:  gdk.PIXBUF_ROTATE_CLOCKWISE}
            img.set_from_pixbuf(pixbuf.rotate_simple(pixbuf_rotations[orientation]))

    def play(self):
        self.status = self.PLAYING
        self._cancel_last_call_later()
        self._call_later = reactor.callLater(self.duration, self.next_image)

        self.emit('status-changed', self.status)

    def pause(self):
        self.status = self.PAUSED
        self.emit('status-changed', self.status)
        self._cancel_last_call_later()

    def stop(self):
        self.playlist[:] = []
        self.current_index = -1
        self.status = self.STOPPED
        self.emit('status-changed', self.status)
        self._cancel_last_call_later()

    def enqueue_to_playlist(self, model):
        self.playlist.append(model)

    def jump_to_index(self, index):
        self.info("Loading image at index %r", index)
        self._set_current_index(index)
        self.background.animated = False
        self.background.selected_item_index = index
        self.background.animated = True
        self.status = self.PAUSED
        self.emit('status-changed', self.status)

    def play_at_index(self, index):
        self.info("Loading image at index %r", index)
        self._set_current_index(index)
        self.background.selected_item_index = index

    def _set_current_index(self, index):
        previous_index = self.current_index
        if index != previous_index:
            self.current_index = index

            def emit_index_changed(result=None):
                # as this is an async callback emit the signal only in case it
                # is still valid.
                if self.current_index == index:
                    self.emit('current-index-changed', index, previous_index)

            def playcount_time_updated(result):
                dfr2 = self._update_orientation(self.playlist[index].references[-1],
                                                index)
                dfr2.addCallback(emit_index_changed)
                return dfr2

            dfr = self._update_playcount_and_time(self.playlist[index].references[-1])
            if dfr:
                dfr.addCallback(playcount_time_updated)
            else:
                self._update_orientation(self.playlist[index].references[-1],
                                         index)
                emit_index_changed()


    def clear_playlist(self):
        # just a API wrapper for stop
        self.stop()

    def next_image(self):
        self._cancel_last_call_later()

        index = self.current_index + 1
        if index < len(self.playlist):
            self.play_at_index(index)
            if self.status == self.PLAYING:
                self._call_later = reactor.callLater(self.duration,
                                                     self.next_image)
            return True
        else:
            self.info("Reached end of slideshow")
            return False

    def previous_image(self):
        self._cancel_last_call_later()

        index = self.current_index - 1
        if index < len(self.playlist) and index >= 0:
            self.play_at_index(index)
            if self.status == self.PLAYING:
                self._call_later = reactor.callLater(self.duration,
                                                     self.previous_image)
            return True
        else:
            self.info("Reached beginning of slideshow")
            return False


class SlideshowController(PigmentController):

    def __init__(self):
        super(SlideshowController, self).__init__()
        self.volume_display_factor = 10.0
        self.player = Slideshow()
        self.widget.add(self.player.background)
        self.player.background.position = (0.0, 0.0, -1.0)
        self.player.background.size = (1.0, 1.0)
        
        self.player.connect('current-index-changed', self._player_index_changed)
        self.player.connect('status-changed', self._player_status_changed)

        self.player.playlist.notifier.connect('items-deleted',
                                              self._player_playlist_changed)
        self.player.playlist.notifier.connect('items-inserted',
                                              self._player_playlist_changed)

        def on_item_clicked(widget, item):
            if self.has_focus():
                self.player_osd.show()
                self.player_osd.control_ribbon.show()
                self.player_osd.mouse_osd.show()
            return True

        self.player.background.connect('item-clicked', on_item_clicked)


        # On Screen Display
        self.player_osd = PlayerOsd()

        self.widget.add(self.player_osd)
        self.player_osd.bg_a = 0
        self.player_osd.opacity = 0
        self.player_osd.status.status_display_details.details.label = ""
        self.player_osd.status.status_display_details.details_2.label = ""
        self.player_osd.visible = True
        
        # connect to the back button
        self.player_osd.mouse_osd.back_button.connect('clicked',
                                                      self._exit_clicked_cb)

        self.player_osd.status.connect \
                                ('seeking-changed', self._seeking_bar_cb)
        self.player_osd.status.connect \
                                ('volume-changed', self._volume_bar_cb)

        # photo controls
        ribbon = self.player_osd.control_ribbon
        ribbon.add_control(ShowPreviousControl(self))
        ribbon.add_control(PlayPauseControl(self))
        ribbon.add_control(StopControl(self))
        ribbon.add_control(ShowNextControl(self))
        # TODO: favorite, fit-screen

        self._rotation_control = RotationControl(self)

        self.music_player = None

        # placeholder for logo, optionally used by plugins
        self.logo = Image()
        self.widget.add(self.logo)
        self.logo.alignment = pgm.IMAGE_BOTTOM_RIGHT
        self.logo.size = (0.1, 0.1)
        # logo is located:
        # - at the bottom right corner of the player
        # - on top of the player but below the on screen display
        self.logo.position = (1.0-self.logo.width, 1.0-self.logo.height,
                              (self.player.background.z+self.player_osd.z)/2.0)
        self.logo.bg_color = (0, 0, 0, 0)
        self.logo.opacity = 150
        self.logo.visible = True

    def _exit_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.exit()

        return True

    def exit(self):
        """
        Exit the player by asking poblesec's main controller to hide it.
        """
        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        self.player_osd.hide()
        main.hide_current_player()

    def _seeking_bar_cb(self, widget, position):
        self.player.jump_to_index(position)
        
    def _volume_bar_cb(self, widget, position):
        self.mute_volume = -1
        self.music_player.set_volume(position / self.volume_display_factor)

    def _player_status_changed(self, player, status):
        if status == player.STOPPED:
            # clean up
            self.player_osd.status.preview.clear()
            self.player_osd.status.status_display_details.title.label = ""

    def _player_index_changed(self, player, index, previous_index):
        st = self.player_osd.status
        dd = self.player_osd.status.status_display_details

        # first clean up
        st.preview.clear()
        dd.title.label = ""

        item = self.player.playlist[index]

        # update title display
        if item.title == None:
            dd.title.label = ""
            icon = 'elisa.plugins.poblesec.file_picture'
            self.frontend.load_from_theme \
                                    (icon, self.player_osd.status.preview)
            
        else:
            dd.title.label = item.title
            # FIXME Not the good way to do it.
            bg_list = player.background
            try:
                widget_index = bg_list._widget_index_from_item_index(index)
                long_loading_image = bg_list._widgets[widget_index]
            except IndexError:
                # FIXME : In this case, the picture is never displayed
                # Only if the user scrolls quickly
                icon = 'elisa.plugins.poblesec.file_picture'
                self.frontend.load_from_theme \
                                    (icon, self.player_osd.status.preview)
            else:
                st.preview.set_from_image(long_loading_image.image)
                
        # apply picture orientation in the status display widget
        uri = self.player.playlist[index].references[-1]
        orientation = item.orientation
        if not orientation:
            orientation = image.ROTATED_0
        self.player.apply_pixbuf_orientation(uri, st.preview, orientation)

        # update ribbon
        self._update_ribbon(item)

        # update index display
        progress_bars = self.player_osd.status.status_display_details.progress_bars
        if not progress_bars.seeking_bar.seek_drag:
            self.player_osd.status.set_seeking_position(index)

        self._update_logo()

    def _update_ribbon(self, model=None):
        ribbon = self.player_osd.control_ribbon
        store = common.application.store

        if self.player.playlist:
            if not model:
                try:
                    model = self.player.playlist[self.player.current_index]
                except IndexError:
                    # current_index not coherent with playlist length
                    # (like in Flickr plugin)
                    model = self.player.playlist[0]

            if model.can_rotate:
                if self._rotation_control not in ribbon.controls:
                    ribbon.add_control(self._rotation_control)

                # hide the new control if the ribbon is invisible
                if not ribbon.is_visible:
                    self._rotation_control.button.square.opacity = 0
            else:
                if self._rotation_control in ribbon.controls:
                    ribbon.remove_control(self._rotation_control)

    def _update_logo(self):
        # if there is no photo displayed or if the current photo does not have
        # a 'logo' attribute display nothing otherwise load the 'logo'
        # resource
        try:
            current_photo = self.player.playlist[self.player.current_index]
            self.frontend.load_from_theme(current_photo.logo, self.logo)
        except (IndexError, AttributeError):
            self.logo.clear()

    def _player_playlist_changed(self, *args):
        self.player_osd.status.set_seeking_duration(len(self.player.playlist))
        self._update_logo()
        self._update_ribbon()

    def set_frontend(self, frontend):
        super(SlideshowController, self).set_frontend(frontend)
        self._load_img_from_theme()
        self.frontend.viewport.connect('motion-notify-event',
                                       self._mouse_motion_cb)
        self.player._load_from_theme = self.frontend.load_from_theme
        
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        self.music_player = main.music_player.player

        self.player_osd.status.set_volume_max \
                        (self.music_player.volume_max * self.volume_display_factor)
        self.player_osd.status.set_volume_position \
                        (self.music_player.get_volume() * self.volume_display_factor)

    def _mouse_motion_cb(self, viewport, event):
        if self.has_focus():
            self.player_osd.show()
            self.player_osd.control_ribbon.show()
            self.player_osd.mouse_osd.show()

    def _load_img_from_theme(self):
        lft = self.frontend.load_from_theme
        #FIXME Still needed ?

    def toggle_play_pause(self):
        if self.player.status == self.player.PLAYING:
            self.player.pause()
            self.player_osd.control_ribbon.show()
            self.player_osd.show()
        else:
            self.player.play()
            self.player_osd.hide()

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_MENU:
            if self.player_osd.animated.opacity == 0:
                self.player_osd.control_ribbon.show()
                self.player_osd.show()
                return True
            elif self.player_osd.control_ribbon.animated.opacity == 0:
                self.player_osd.control_ribbon.show()
                return True
            else:
                self.player_osd.hide()
                self.player.pause()
                return False

        elif input_event.value == EventValue.KEY_OK or \
            input_event.value == EventValue.KEY_SPACE:
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.control_ribbon.selected_control.activate()
            else:
                self.player_osd.control_ribbon.show()
                self.player_osd.show()
            return True

        elif input_event.value == EventValue.KEY_GO_LEFT:
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.show()
                self.player_osd.control_ribbon.select_previous_control()
            else:
                self.player.previous_image()

        elif input_event.value == EventValue.KEY_GO_RIGHT:
            if self.player_osd.control_ribbon.opacity == 255:
                self.player_osd.show()
                self.player_osd.control_ribbon.select_next_control()
            else:
                self.player.next_image()
                
        elif input_event.value == EventValue.KEY_GO_UP:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.VOLUME_BAR)
            self.player_osd.show()
            volume = self.music_player.volume_up() * self.volume_display_factor
            self.player_osd.status.set_volume_position(int(volume))
            return True
        
        elif input_event.value == EventValue.KEY_GO_DOWN:
            pbc = self.player_osd.status.status_display_details.progress_bars
            pbc.set_visible_bar(BARS.VOLUME_BAR)
            self.player_osd.show()
            volume = self.music_player.volume_down() * self.volume_display_factor
            self.player_osd.status.set_volume_position(int(volume))
            return True

        return False
