# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.database.models import *

from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.section import SectionMenuViewMode
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.music_library import ArtistsViewMode, \
                                                 AlbumsViewMode, \
                                                 TracksViewMode

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.database.actions import ArtistPlayAllAction, \
                                           AlbumPlayAllAction, \
                                           TrackPlayAllAction
from elisa.plugins.database.actions import ArtistShuffleAction, \
                                           AlbumShuffleAction, \
                                           TrackShuffleAction
from elisa.plugins.database.actions import AlbumAddToFavoritesAction, \
                                           TrackAddToFavoritesAction, \
                                           PhotoAddToFavoritesAction
from elisa.plugins.database.actions import ViewPhotoAlbumSlideshowAction

from elisa.plugins.favorites.models import FavoritesItem

from twisted.internet import defer, task

from storm.expr import Not

import datetime

_ = install_translation('database')


def music_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music_library'
    link.label = _('Music Library')
    link.icon = 'elisa.plugins.poblesec.music_library'
    controller.model.append(link)

    return defer.succeed(None)


def music_lib_artists_decorator(controller):
    artists = Link()
    artists.controller_path = '/poblesec/database/music/artists'
    artists.label = _('Artists')
    artists.icon = 'elisa.plugins.poblesec.by_artist'
    controller.model.append(artists)
    return defer.succeed(None)


def music_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/music/albums'
    albums.label = _('Albums')
    albums.icon = 'elisa.plugins.poblesec.by_album'
    controller.model.append(albums)
    return defer.succeed(None)


def music_lib_tracks_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/music/tracks'
    tracks.label = _('Tracks')
    tracks.icon = 'elisa.plugins.poblesec.by_track'
    controller.model.append(tracks)
    return defer.succeed(None)


def music_lib_genres_decorator(controller):
    genres = Link()
    genres.controller_path = '/poblesec/database/music/genres'
    genres.label = _('Genres')
    genres.icon = 'elisa.plugins.poblesec.by_genre'
    controller.model.append(genres)
    return defer.succeed(None)


def music_lib_decades_decorator(controller):
    decades = Link()
    decades.controller_path = '/poblesec/database/music/decades'
    decades.label = _('Decades')
    decades.icon = 'elisa.plugins.poblesec.by_decade'
    controller.model.append(decades)
    return defer.succeed(None)


class GenericArtistsController(HierarchyController):

    def initialize(self):
        dfr= super(GenericArtistsController, self).initialize()
        return dfr

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(item, Action):
                item.run()
            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                path = '/poblesec/database/music/albums'
                deferred = browser.history.append_controller(path, item.name, artist=item)

    def make_actions(self):
        actions = []

        action = ArtistPlayAllAction(self)
        action.title = _('Play All')
        # FIXME: there is no two lines support in ArtistController
        action.subtitle = _('Play All Tracks By All Artists')
        actions.append(action)

        action = ArtistShuffleAction(self)
        action.title = _('Shuffle & Play')
        # FIXME: there is no two lines support in ArtistController
        action.subtitle = _('Shuffle & Play All Tracks By All Artists')
        actions.append(action)

        return actions


class ArtistsDbController(GenericArtistsController):

    def initialize(self):
        deferred = super(ArtistsDbController, self).initialize()

        def got_artists(artists):
            artists = sorted(artists, key=lambda a: a.name.lower())
            self.model.extend(artists)
            return self

        def sort_artists(result_set):
            result_set.order_by(Artist.name)
            return result_set.all()

        def get_artists(self):
            return common.application.store.find(Artist)

        deferred.addCallback(get_artists)
        deferred.addCallback(sort_artists)
        deferred.addCallback(got_artists)
        return deferred

class ArtistsDbViewMode(ArtistsViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(ArtistsDbViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return super(ArtistsDbViewMode, self).get_sublabel(item)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        return super(ArtistsDbViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(ArtistsDbViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(ArtistsDbViewMode, self).get_preview_image(item, theme)


class ArtistsDbPreviewListController(ArtistsDbController, MenuItemPreviewListController):
    view_mode = ArtistsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class ArtistsDbCoverflowController(ArtistsDbController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsDbViewMode

class ArtistsDbGridController(ArtistsDbController, GridItemGridController):
    view_mode = ArtistsDbViewMode

class ArtistsDbListSwitcherController(ListSwitcherController):
    modes = [ArtistsDbPreviewListController,
             ArtistsDbCoverflowController,
             ArtistsDbGridController]
    default_mode = ArtistsDbPreviewListController


class GenericAlbumsDbController(HierarchyController):

    def initialize(self, artist=None):
        self.artist = artist
        dfr= super(GenericAlbumsDbController, self).initialize()
        return dfr

    def make_actions(self):
        actions = []

        if self.artist:
            action = AlbumPlayAllAction(self)
            action.title = _('Play All')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _("Play All Tracks By '%s'") % self.artist.name
            actions.append(action)

            action = AlbumShuffleAction(self)
            action.title = _('Shuffle & Play')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _("Shuffle & Play All Tracks By '%s'") % self.artist.name
            actions.append(action)

            action = AlbumAddToFavoritesAction(self)
            # FIXME: there is no two lines support in AlbumController
            action.untoggled_title = _('Add To Favorites')
            action.untoggled_subtitle = _("Add The Artist '%s' To Your Favorites") % self.artist.name
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove The Artist '%s' From Your Favorites") % self.artist.name
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.artist.name,
                                                FavoritesItem.foreign_class == u'Artist')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        else:
            action = AlbumPlayAllAction(self)
            action.title = _('Play All')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _('Play All Tracks From All Albums')
            actions.append(action)

            action = AlbumShuffleAction(self)
            action.title = _('Shuffle & Play')
            # FIXME: there is no two lines support in AlbumController
            action.subtitle = _('Shuffle & Play All Tracks From All Albums')
            actions.append(action)

        return actions

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(item, Action):
                dfr = item.run()
            elif hasattr(item, 'artist'):
                # Dummy "All Tracks" album
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                dfr = browser.history.append_controller('/poblesec/database/music/tracks',
                                                        item.name,
                                                        artist=item.artist)

            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                dfr = browser.history.append_controller('/poblesec/database/music/tracks',
                                                        item.name,
                                                        album=item)
            return dfr


class AlbumsDbController(GenericAlbumsDbController):

    def initialize(self, artist=None):
        deferred = super(AlbumsDbController, self).initialize(artist=artist)

        def sort_albums(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def add_albums(albums):
            # Add an "All Tracks" album for a given artist
            if self.artist:
                all_tracks_album = MusicAlbum()
                all_tracks_album.name = u'All Tracks'
                all_tracks_album.artist = self.artist # dynamic field
                self.model.append(all_tracks_album)

            self.model.extend(albums)
            return self

        if artist is None:
            # Show all albums
            def get_albums(self):
                return common.application.store.find(MusicAlbum)

            deferred.addCallback(get_albums)
            deferred.addCallback(sort_albums)
            deferred.addCallback(add_albums)

        else:
            # Show all albums for a given artist
            def get_albums(album_names):
                store = common.application.store
                return store.find(MusicAlbum,
                                  MusicAlbum.name.is_in(album_names))

            def order_tracks(result_set):
                return result_set.values(MusicTrack.album_name)

            def get_tracks(self):
                return artist.tracks.find()

            deferred.addCallback(get_tracks)
            deferred.addCallback(order_tracks)
            deferred.addCallback(get_albums)
            deferred.addCallback(sort_albums)
            deferred.addCallback(add_albums)

        return deferred


class AlbumsDbViewMode(AlbumsViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(AlbumsDbViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return super(AlbumsDbViewMode, self).get_sublabel(item)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        elif hasattr(item, 'artist'):
            # "All Tracks" fake album
            return 'elisa.plugins.poblesec.by_track'
        return super(AlbumsDbViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        elif hasattr(item, 'artist'):
            # "All Tracks" fake album
            return None
        return super(AlbumsDbViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        elif hasattr(item, 'artist'):
            # "All Tracks" fake album
            return None
        return super(AlbumsDbViewMode, self).get_preview_image(item, theme)


class AlbumsDbPreviewListController(AlbumsDbController, MenuItemPreviewListController):
    view_mode = AlbumsDbViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class AlbumsDbCoverflowController(AlbumsDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class AlbumsDbGridController(AlbumsDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class AlbumsDbListSwitcherController(ListSwitcherController):
    modes = [AlbumsDbPreviewListController,
             AlbumsDbCoverflowController,
             AlbumsDbGridController]
    default_mode = AlbumsDbPreviewListController


class GenericTracksDbController(HierarchyController):

    track_controller_path = '/poblesec/database/music/tracks'

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        self.album = album
        self.track = track
        self.tracks = tracks
        self.artist = artist

        # we don't need the fastscroller, because we're going to order tracks
        # by track number
        if album:
            self.fastscroller = False

        dfr= super(GenericTracksDbController, self).initialize()

        return dfr

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

    def make_actions(self):
        actions = []

        if self.album:
            action = TrackPlayAllAction(self)
            action.title = _('Play All')
            action.subtitle = _("Play All Tracks From '%s'") % self.album.name
            actions.append(action)

            action = TrackShuffleAction(self)
            action.title = _('Shuffle & Play')
            action.subtitle = _("Shuffle & Play All Tracks From '%s'") % self.album.name
            actions.append(action)

            action = TrackAddToFavoritesAction(self)
            action.untoggled_title = _('Add To Favorites')
            action.untoggled_subtitle = _("Add The Album '%s' To Your Favorites") % self.album.name
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove The Album '%s' From Your Favorites") % self.album.name
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.album.name,
                                                FavoritesItem.foreign_class == u'MusicAlbum')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        elif self.track:
            action = TrackPlayAllAction(self)
            action.title = _('Play Track')
            action.subtitle = _("Play '%s'") % self.track.title
            actions.append(action)

            action = TrackAddToFavoritesAction(self)
            action.untoggled_title = _('Add to Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.track.title
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.track.title
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.track.file_path,
                                                FavoritesItem.foreign_class == u'MusicTrack')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        else:
            action = TrackPlayAllAction(self)
            action.title = _('Play All')
            actions.append(action)

            action = TrackShuffleAction(self)
            action.title = _('Shuffle & Play')
            actions.append(action)

        return actions

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            if isinstance(item, Action):
                item.run()
            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                path = self.track_controller_path
                title = item.title
                tracks = [t for t in self.model if isinstance(t, MusicTrack)]
                args = {'track': item,
                        'tracks': tracks}
                dfr = browser.history.append_controller(path, title, **args)


class TracksDbController(GenericTracksDbController):

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        deferred = super(TracksDbController, self).initialize(album=album,
                                                              track=track,
                                                              tracks=tracks,
                                                              artist=artist)

        if track:
            return deferred

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_tracks_number_title(result_set):
            result_set.order_by(MusicTrack.track_number, MusicTrack.title)
            return result_set.all()

        def sort_tracks_title(result_set):
            result_set.order_by(MusicTrack.title)
            return result_set.all()

        def sort_tracks_title_client_side(tracks):
            trash_letters = " .,-_"
            tracks.sort(key=lambda t: t.title.strip(trash_letters).lower())
            return tracks

        def get_tracks(self):
            store = common.application.store
            if self.artist is not None:
                # oh, hey, let's fetch all the tracks of the given artist
                dfr = store.find(MusicTrack,
                                 MusicTrack.file_path == TrackArtist.track_path,
                                 TrackArtist.artist_name == self.artist.name)
                # Add client side ordering - can this be done "server" side? Is
                # there a better way to trash spurious characters?
                dfr.addCallback(sort_tracks_title)
                dfr.addCallback(sort_tracks_title_client_side)
            elif self.album is not None:
                # FIXME: in this case we should not display an alphabetical
                # fastscroller, which does not make sense for a small number of
                # tracks and will be broken anyway, because the tracks are not
                # ordered alphabetically. See bug #253061 on Launchpad.
                dfr = store.find(MusicTrack, MusicTrack.album_name == self.album.name)
                dfr.addCallback(sort_tracks_number_title)
            else:
                dfr = store.find(MusicTrack, MusicTrack.title != None)
                # Add client side ordering - can this be done "server" side? Is
                # there a better way to trash spourious characters?
                dfr.addCallback(sort_tracks_title)
                dfr.addCallback(sort_tracks_title_client_side)

            dfr.addCallback(got_tracks)
            return dfr

        return deferred.addCallback(get_tracks)


class DBTracksViewMode(TracksViewMode):
    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return super(DBTracksViewMode, self).get_label(item)

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)

        def values(result):
            return result.values(Artist.name)

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        def set_values(result_list):
            result_list_len = len(result_list)
            if result_list_len == 0:
                # nothing found
                sublabel = ''
            elif result_list_len == 1:
                sublabel = result_list[0]
            elif result_list_len > 1:
                sublabel = ', '.join(result_list)
            return sublabel

        deferred = item.artists.find()
        deferred.addCallback(set_distinct)
        deferred.addCallback(values)
        deferred.addCallback(set_values)
        return deferred

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        return super(DBTracksViewMode, self).get_default_image(item)

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(DBTracksViewMode, self).get_image(item, theme)

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return super(DBTracksViewMode, self).get_preview_image(item, theme)


class DBTracksInAlbumViewMode(DBTracksViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        # Strip trash letter - defined in TracksViewMode
        title = item.title.strip(self.trash_letters)
        if item.track_number:
            title = '%s. %s' % (str(item.track_number).zfill(2), title)
        return defer.succeed(title)


class TracksPreviewListController(TracksDbController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = True

    def initialize(self, album=None, track=None, tracks=None, artist=None):
        # Change the visualization of tracks, depending on whether they
        # are displayed in one album or not
        if album:
            self.view_mode = DBTracksInAlbumViewMode

        return super(TracksPreviewListController, self).initialize(album=album,
                                                                   track=track,
                                                                   tracks=tracks,
                                                                   artist=artist)

class TracksDbCoverflowController(TracksDbController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode

class TracksDbGridController(TracksDbController, GridItemGridController):
    view_mode = DBTracksViewMode

class TracksDbListSwitcherController(ListSwitcherController):
    modes = [TracksPreviewListController,
             TracksDbCoverflowController,
             TracksDbGridController]
    default_mode = TracksPreviewListController


# A list of important genres
GENRES = [u"Acid Jazz", u"Alternative", u"Alternative & Punk",
          u"Alternative Pop/Rock", u"Alternative Rock", u"AlternRock",
          u"Ambient", u"Blues", u"Blues/R&B", u"Books & Spoken",
          u"Children's Music", u"Classic Rock", u"Classical", u"Comedy",
          u"Dance", u"Easy Listening", u"Electronic", u"Electronica & Dance",
          u"Electronice/Dance", u"Entertainment", u"Folk", u"Funk",
          u"Games", u"Garage/Punk Rock Revival", u"General Alternative",
          u"Hard Rock", u"Hip Hop", u"Hip Hop/Rap", u"Hip-Hop",
          u"Hip-Hop/Rap", u"Holiday", u"House", u"Indi", u"Industrial",
          u"Jazz", u"Latin", u"Metal", u"Music Videos", u"New Age",
          u"Other", u"Pop", u"Punk", u"R&B", u"R&B/Soul", u"Rap", u"Raggae",
          u"Religious", u"Rock", u"Rock & Roll", u"Soundtrack", u"Techno",
          u"Trance", u"World"]


class GenresDbController(HierarchyController):

    def initialize(self, all_genres=False):
        self.all_genres = all_genres

        deferred = super(GenresDbController, self).initialize()

        def got_genres(genres, all_genres):
            def iterate(genres, models):
                for genre in genres:
                    link = Link()
                    link.controller_path = '/poblesec/database/music/genre'
                    link.label = genre
                    link.icon = 'elisa.plugins.poblesec.by_genre'
                    link.controller_args = {'genre': genre}
                    models.append(link)
                    yield link

            def iterate_done(result, models):
                self.model.extend(models)
                return self

            models = []

            if not all_genres:
                link = Link()
                link.controller_path = '/poblesec/database/music/genres'
                link.label = _("All genres")
                link.icon = 'elisa.plugins.poblesec.by_genre'
                link.controller_args = {'all_genres': True}
                models.append(link)

            dfr = task.coiterate(iterate(genres, models))
            dfr.addCallback(iterate_done, models)
            return dfr

        def sort_by_title(result_set):
            result_set.order_by(MusicTrack.genre)
            return result_set

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        def get_genres(result_set):
            return result_set.values(MusicTrack.genre)

        def get_tracks(self):
            store = common.application.store
            if all_genres:
                dfr = store.find(MusicTrack, MusicTrack.genre != None)
            else:
                dfr = store.find(MusicTrack, MusicTrack.genre.is_in(GENRES))
            return dfr

        deferred.addCallback(get_tracks)
        deferred.addCallback(set_distinct)
        deferred.addCallback(sort_by_title)
        deferred.addCallback(get_genres)
        deferred.addCallback(got_genres, all_genres)
        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            if isinstance(item, Action):
                item.run()
            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                path = item.controller_path
                args = item.controller_args
                dfr = browser.history.append_controller(path, item.label, **args)


class GenresDbPreviewListController(GenresDbController, MenuItemPreviewListController):
    view_mode = SectionMenuViewMode

class GenresDbCoverflowController(GenresDbController, ImageWithReflectionCoverflowController):
    view_mode = SectionMenuViewMode

class GenresDbGridController(GenresDbController, GridItemGridController):
    view_mode = SectionMenuViewMode

class GenresDbListSwitcherController(ListSwitcherController):
    modes = [GenresDbPreviewListController,
             GenresDbCoverflowController,
             GenresDbGridController]
    default_mode = GenresDbPreviewListController


class GenreDbController(GenericTracksDbController):

    # FIXME: shouldn't we display a list of albums rather than a list of tracks?

    def initialize(self, genre=None):
        self.genre = genre

        deferred = super(GenreDbController, self).initialize()

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_by_title(result_set):
            result_set.order_by(MusicTrack.genre)
            return result_set.all()

        def get_tracks(self):
            store = common.application.store
            if genre:
                dfr = store.find(MusicTrack, MusicTrack.genre == unicode(genre))
            else:
                dfr = store.find(MusicTrack, Not(MusicTrack.genre.is_in(GENRES)))
            return dfr

        deferred.addCallback(get_tracks)
        deferred.addCallback(sort_by_title)
        deferred.addCallback(got_tracks)
        return deferred


class GenrePreviewListController(GenreDbController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = True

    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.title

class GenreDbCoverflowController(GenreDbController, ImageWithReflectionCoverflowController):
    view_mode = TracksViewMode

class GenreDbGridController(GenreDbController, GridItemGridController):
    view_mode = TracksViewMode

class GenreDbListSwitcherController(ListSwitcherController):
    modes = [GenrePreviewListController,
             GenreDbCoverflowController,
             GenreDbGridController]
    default_mode = GenrePreviewListController


class DecadesDbController(HierarchyController):

    def initialize(self, all_genres=False):
        self.all_genres = all_genres

        deferred = super(DecadesDbController, self).initialize()

        def iterate(models):
            # FIXME: What about other decades (10's, 20's, 30's, 40's, ...)?
            #        And what about other centuries?
            for decade in (50, 60, 70, 80, 90):
                link = Link()
                link.label = _("%d's") % decade
                link.icon = 'elisa.plugins.poblesec.by_decade'
                link.controller_args = \
                    {'begin': datetime.datetime(1900 + decade, 1, 1),
                     'end': datetime.datetime(1909 + decade, 12, 31)}
                models.append(link)
                yield link

        def iterate_done(result, models):
            self.model.extend(models)
            return self

        models = []

        def populate(self):
            dfr = task.coiterate(iterate(models))
            dfr.addCallback(iterate_done, models)
            return dfr

        deferred.addCallback(populate)
        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            if isinstance(item, Action):
                item.run()
            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                path = '/poblesec/database/music/time'
                args = item.controller_args
                dfr = browser.history.append_controller(path, item.label, **args)


class DecadesDbPreviewListController(DecadesDbController, MenuItemPreviewListController):
    view_mode = SectionMenuViewMode

class DecadesDbCoverflowController(DecadesDbController, ImageWithReflectionCoverflowController):
    view_mode = SectionMenuViewMode

class DecadesDbGridController(DecadesDbController, GridItemGridController):
    view_mode = SectionMenuViewMode

class DecadesDbListSwitcherController(ListSwitcherController):
    modes = [DecadesDbPreviewListController,
             DecadesDbCoverflowController,
             DecadesDbGridController]
    default_mode = DecadesDbPreviewListController


class TimeDbController(GenericAlbumsDbController):

    def initialize(self, begin=None, end=None):
        self.begin = begin
        self.end = end

        deferred = super(TimeDbController, self).initialize()

        def got_albums(albums):
            if not albums:
                # FIXME: We should never raise generic exceptions, let's find
                #        a suitable specific exception for this case.
                raise Exception('Missing Data')
            self.model.extend(albums)
            return self

        def sort_by_title(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        def get_albums(self):
            store = common.application.store
            dfr = store.find(MusicAlbum,
                             (MusicAlbum.release_date >= begin) & \
                             (MusicAlbum.release_date <= end))
            return dfr

        def missing_data(failure):
            self.warning("Missing Data for decade.")
            return self

        deferred.addCallback(get_albums)
        deferred.addCallback(sort_by_title)
        deferred.addCallback(got_albums)
        deferred.addErrback(missing_data)
        return deferred

    def nodes_setup(self):
        super(TimeDbController, self).nodes_setup()
        model = [m for m in self.model if not isinstance(m, Action)]
        if len(model) == 0:
            self.model[:] = []
            msg = _("There are no albums for this decade.")
            resource = 'elisa.plugins.poblesec.by_decade'
            img = self.frontend.get_theme().get_resource(resource)
            self.nothing_to_display_widget.icon.set_from_file(img)
            self.nothing_to_display_widget.text.label = msg
            self.nothing_to_display_widget.visible = True


class TimeDbPreviewListController(TimeDbController, MenuItemPreviewListController):
    view_mode = AlbumsDbViewMode

class TimeDbCoverflowController(TimeDbController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsDbViewMode

class TimeDbGridController(TimeDbController, GridItemGridController):
    view_mode = AlbumsDbViewMode

class TimeDbListSwitcherController(ListSwitcherController):
    modes = [TimeDbPreviewListController,
             TimeDbCoverflowController,
             TimeDbGridController]
    default_mode = TimeDbPreviewListController
