# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from twisted.trial.unittest import TestCase
from elisa.plugins.yesfm.searcher import YesfmSearcher
from elisa.plugins.search.models import MusicSearchResultModel
from elisa.plugins.yesfm.models import AlbumModel, ArtistModel, TrackModel, \
                                        GeneralSearchResultModel,\
                                        AlbumsSearchResultModel,\
                                        TracksSearchResultModel,\
                                        ArtistsSearchResultModel

from elisa.core import common
from elisa.core.utils import defer
from elisa.core.media_uri import MediaUri

class Dummy(object): pass

class DummyResourceProvider(object):
    def __init__(self):
        self.uri_to_response = {}

    def get(self, uri, model=None):
        model = self.uri_to_response[unicode(uri)]
        return model, defer.succeed(model)

class DummyResourceManager(object):
    def get_resource_provider_by_path(self, path):
        self.path = path
        return self.provider

class TestYesfmSearcher(TestCase):

    uri_base = u"http://api.yes.fm/v1/search/%s"

    def setUp(self):
        def created(searcher):
            self.searcher = searcher

        self.patch_application()
        return YesfmSearcher.create().addCallback(created)

    def patch_application(self):
        self.application = common.application
        self.manager = DummyResourceManager()
        self.provider = DummyResourceProvider()
        self.manager.provider = self.provider

        application = Dummy()
        application.resource_manager = self.manager

        common.application = application

    def tearDown(self):
        common.application = self.application

    def _create_result(self, query, artists=[], albums=[], tracks=[]):
        model = GeneralSearchResultModel()

        model.artists = ArtistsSearchResultModel()
        artists_lst = model.artists.result

        model.albums = AlbumsSearchResultModel()
        albums_lst = model.albums.result

        model.tracks = TracksSearchResultModel()
        tracks_lst = model.tracks.result
        
        for artist in artists:
            artist_model = ArtistModel()
            artist_model.name = artist
            artists_lst.append(artist_model)

        for album in albums:
            album_model = AlbumModel()
            album_model.album = album
            albums_lst.append(album_model)

        for track in tracks:
            track_model = TrackModel()
            track_model.title = track
            tracks_lst.append(track_model)

        query_str = self.uri_base % query
        self.provider.uri_to_response[query_str] = model

        return model

    def _check(self, result, model, artists=[], albums=[], tracks=[]):
       
        # first check the count
        msg = "Not the same number of %s found"
        self.assertEquals(len(model.artists), len(artists), msg % "artists")
        self.assertEquals(len(model.albums), len(albums), msg % "albums")
        self.assertEquals(len(model.tracks), len(tracks), msg % "tracks")

        # then check the values and the order
        self.assertEquals(artists, [a.name for a in model.artists])
        self.assertEquals(albums, [a.album for a in model.albums])
        self.assertEquals(tracks, [a.title for a in model.tracks])

    def test_provider_requested_once_only(self):
        """
        Test that the Searcher is only requesting once for the resource provider
        and then simply reuses it all the time
        """

        self._create_result('madonna')

        uri = MediaUri('elisa://search/music/madonna')
        model = MusicSearchResultModel()

        self.assertEquals(self.searcher.provider, None)
        self.searcher.search(uri, model)

        self.failUnless(self.searcher.provider)

        self.searcher.search(uri, model)

        self.failUnless(self.searcher.provider)
        
    def test_uses_right_uri(self):
        """
        Test whether the uri is translated right
        """

        self._create_result('Madonna', ['Madonna'])

        uri = MediaUri('elisa://search/music/Madonna')
        model = MusicSearchResultModel()

        dfr = self.searcher.search(uri, model)
        dfr.addCallback(self._check, model, ['Madonna'])
        return dfr

    def test_uses_right_uri_with_spaces(self):
        """
        Test whether the uri is translated right when it contains spaces
        """

        self._create_result('The%20Cranberries', ['The Cranberries'])

        uri = MediaUri('elisa://search/music/The Cranberries')
        model = MusicSearchResultModel()

        dfr = self.searcher.search(uri, model)
        dfr.addCallback(self._check, model, ['The Cranberries'])
        return dfr


    def test_parsing_and_order(self):
        """
        Test whether the data is parsed properly
        """
        # wrong order!
        self._create_result('Coldplay',
                ['Coldplay', 'Nearly Coldplay', 'Hotplay'],
                ['Coldplay', 'Best of Coldplay', 'X&Y', 'or not'],
                ['One', 'two', 'Three', 'four', 'five'])

        uri = MediaUri('elisa://search/music/Coldplay')
        model = MusicSearchResultModel()

        dfr = self.searcher.search(uri, model)
        # check if it is ordered
        result = {'artists' : ['Coldplay', 'Hotplay', 'Nearly Coldplay'],
                  'albums' : ['Best of Coldplay', 'Coldplay', 'or not', 'X&Y'],
                  'tracks' : ['five', 'four', 'One', 'Three', 'two'],}

        dfr.addCallback(self._check, model, **result)
        return dfr

