# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Benjamin Kampmann <benjamin@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>
#          Philippe Normand <philippe@fluendo.com>

"""
Common models related to audio.
"""

from elisa.core.components.model import Model
from elisa.core.utils import defer


class AlbumModel(Model):

    """
    Representation of an audio album.

    An audio album model may contain metadata associated to an audio album,
    such as the name of the artist, name of the album, list of the tracks,
    etc...

    @ivar artist:         the name of the artist
    @type artist:         C{unicode}
    @ivar album:          the name of the audio album
    @type album:          C{unicode}
    @ivar musicbrainz_id: a unique id to request more information from
                          musicbrainz.org
    @type musicbrainz_id: C{unicode}
    @ivar cover:          the cover art for the album
    @type cover:          L{elisa.plugins.base.models.image.ImageModel}
    @ivar tracks:         the list of tracks on the album
    @type tracks:         C{list} of
                          L{elisa.plugins.base.models.audio.TrackModel}
    """

    def __init__(self):
        """
        Constructor. Initialize all the fields.
        """
        super(AlbumModel, self).__init__()
        self.artist = None
        self.album = None
        self.musicbrainz_id = None
        self.cover = None
        self.tracks = []

    def get_tracks(self):
        """
        Return a deferred that will return a list of TrackModel instances
        representing the tracks of the album.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.succeed(self.tracks)

    def get_artist_name(self):
        """
        Return a deferred with the artist name as result.
        """
        return defer.succeed(self.artist)


class TrackModel(Model):

    """
    Representation of an audio track on an album.

    @ivar title:        the title of the track
    @type title:        C{unicode}
    @ivar artist:       the name of the artist of the track
    @type artist:       C{unicode}
    @ivar duration:     the duration of the track (in seconds)
    @type duration:     C{int}
    @ivar track_number: the number of the track on the album
    @type track_number: C{int}
    @ivar playable_uri: the URI to query the resource provider with to get a
                        playable model containing the real URI of the audio
                        file
    @type playable_uri: L{elisa.core.media_uri.MediaUri}
    @ivar album:        album containing the track
    @type album:        L{elisa.plugins.base.models.audio.AlbumModel}
    """

    def __init__(self):
        super(TrackModel, self).__init__()
        self.title = None
        self.artist = None
        self.duration = None
        self.track_number = None
        self.playable_uri = None
        self.album = None

    def get_album(self):
        """
        Return a deferred that will return the album containing the track as an
        instance of L{elisa.plugins.base.models.audio.AlbumModel}.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.succeed(self.album)

    def get_artists(self):
        """
        Return a deferred that will return the list of artist names for the
        track as C{unicode} instances.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.succeed([self.artist])

    def get_playable_model(self):
        """
        Return a deferred that will return an instance of
        L{elisa.plugins.base.models.media.PlayableModel} for the track.

        @rtype:  L{elisa.core.utils.defer.Deferred}
        """
        return defer.fail(NotImplementedError())

class ArtistModel(Model):

    """
    Representation of an artist.

    @ivar name:           name of the artist
    @type name:           C{unicode}
    @ivar musicbrainz_id: a unique id to request more information from
                          musicbrainz.org
    @type musicbrainz_id: C{unicode}
    @ivar images:         visual representations of the artist
    @type images:         C{list} of
                          {elisa.plugins.base.models.image.ImageModel}
    """

    def __init__(self):
        super(ArtistModel, self).__init__()
        self.name = None
        self.musicbrainz_id = None
        self.images = []
