# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.components.resource_provider import ResourceProvider
from elisa.core.components.model import Model
from elisa.core.media_uri import MediaUri
from elisa.plugins.http_client.http_client import ElisaAdvancedHttpClient
from twisted.internet import defer
from twisted.web2 import responsecode
from twisted.web2.stream import BufferedStream
from pymetar import WeatherReport, ReportParser

class WeatherReportError(Exception):
    pass

class WeatherReportModel(Model):
    uri = None
    report_uri = None
    country = ''
    city = ''
    sky = ''
    temperature = ''
    image = ''

class WeatherReportProvider(ResourceProvider):
    default_config = {'base_url': 
            'http://weather.noaa.gov/pub/data/observations/metar/decoded'}
    
    config_doc = {'base_url': 'The URL where to get METAR data from'}

    supported_uri = 'metar://.*'

    def initialize(self):
        self.server_uri = MediaUri(self.config['base_url'])
        self.client = ElisaAdvancedHttpClient(host=self.server_uri.host,
                                              port=self.server_uri.port or 80)
        return defer.succeed(self)

    def clean(self):
        close_defer = self.client.close()
        close_defer.addErrback(lambda x: None)

        def parent_clean(result):
            return super(WeatherReportProvider, self).clean()

        close_defer.addCallback(parent_clean)
        return close_defer

    def get(self, uri, context_model=None):
        model = WeatherReportModel()
        model.uri = uri

        station_file = model.uri.host + '.TXT'
        model.report_uri = self.server_uri.join(station_file)
        dfr = self.client.request(model.report_uri)
        dfr.addCallback(self._request_callback, model)

        return (model, dfr)

    def _request_callback(self, response, model):
        if response.code != responsecode.OK:
            err = '%d error fetching report from %s' % (response.code,
                    model.report_uri)
            return defer.fail(WeatherReportError(err))

        dfr = BufferedStream(response.stream).readExactly()
        dfr.addCallback(self._report_read_callback, model)
        return dfr
    
    def _report_read_callback(self, data, model):
        station = model.uri.host
        report = WeatherReport(station)
        report.reporturl = str(model.report_uri)
        report.fullreport = data

        # parse the report
        parser = ReportParser()
        parser.ParseReport(report)

        # populate the model
        model.country = report.getStationCountry()
        model.city = report.getStationCity()
        model.sky = report.getSkyConditions()
        model.temperature = report.getTemperatureCelsius()
        model.humidity = report.getHumidity()
        model.image = report.getPixmap()

        return model
