# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core.common import application
from elisa.core.media_uri import MediaUri
from elisa.core.input_event import EventValue, key_values, num_values
from elisa.core.components.model import Model
from elisa.core.application import CONFIG_DIR, PICTURES_CACHE
from elisa.core.utils.cancellable_defer import CancellableDeferred, \
                                               CancelledError
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController

from elisa.plugins.base.models.image import ImageModel

from elisa.plugins.flickr.models import FlickrResponseModel, \
                                        FlickrPhotoModel, FlickrTagModel
from elisa.plugins.flickr.resource_provider import API_SERVER
import elisa.plugins.flickr.flickr_api as flickr

from twisted.internet import defer

import os
import hashlib
import pickle


_ = install_translation('flickr')


class FlickrEntryPointModel(Model):

    def __init__(self):
        super(Model, self).__init__()
        self.title = None
        self.uri = None


def flickr_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/flickr/home'
    link.label = _('Flickr')
    link.icon = 'elisa.plugins.flickr.logo'
    controller.model.append(link)
    return defer.succeed(None)


# TODO: factorize this code in the core of Elisa in a caching utilities module
def _thumbnail_file(thumbnail_uri):
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = hashlib.md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


class FlickrController(HierarchyController):

    start_uri = MediaUri('http://%s/services/rest/' % API_SERVER)
    search_uri = MediaUri('flickr://search')

    def initialize(self, uri=start_uri):
        deferred = super(FlickrController, self).initialize()
        self.uri = uri

        def add_entries(self):
            entries = [(_('Last 7 days interesting'),
                        flickr.generate_call_uri(method='flickr.interestingness.getList')),
                       (_('Popular tags'),
                        flickr.generate_call_uri(method='flickr.tags.getHotList')),
                       (_('Most recent uploads'),
                        flickr.generate_call_uri(method='flickr.photos.getRecent')),
                      ]
            entry_models = []
            for title, url in entries:
                entry_model = FlickrEntryPointModel()
                entry_model.title = title
                entry_model.uri = MediaUri(url)
                entry_models.append(entry_model)
            self.model.extend(entry_models)
            return self

        def resource_loaded(resource):
            if isinstance(resource, FlickrResponseModel):
                if hasattr(resource, 'photos'):
                    self.model.extend(resource.photos)
                elif hasattr(resource, 'tags'):
                    self.model.extend(resource.tags)
            return self

        def load_resource(self):
            resource, get_deferred = application.resource_manager.get(self.uri)
            return get_deferred

        if uri == self.start_uri:
            # List of the entry points
            deferred.addCallback(add_entries)
        else:
            # One particular entry point
            deferred.addCallback(load_resource)
            deferred.addCallback(resource_loaded)

        return deferred

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(item, FlickrEntryPointModel):
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                browser = controllers[0]
                if item.uri == self.search_uri:
                    dfr = browser.history.append_controller('/poblesec/flickr/search', item.title, uri=item.uri)
                else:
                    dfr = browser.history.append_controller('/poblesec/flickr', item.title, uri=item.uri)

            elif isinstance(item, FlickrTagModel):
                item.uri = MediaUri(flickr.generate_call_uri(method='flickr.photos.search',
                                                     arguments={'tags': item.label}))
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                browser = controllers[0]
                dfr = browser.history.append_controller('/poblesec/flickr', item.label, uri=item.uri)

            elif isinstance(item, FlickrPhotoModel):
                # We want a Deferred here, too
                self.play_image(item)
                dfr = CancellableDeferred()
                dfr.callback(True)

    def play_image(self, item):
        controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
        slideshow_controller = controllers[0]
        slideshow_controller.player.clear_playlist()

        # Cancel all currently queued requests
        for widget in self.nodes._widgets:
            self.cancel_deferreds(widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        for flickr_photo in self.model:
            if isinstance(flickr_photo, FlickrPhotoModel):
                if flickr_photo == item:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                if not hasattr(flickr_photo, 'images'):
                    dfr, photo = _get_image_available_sizes(flickr_photo)
                    self.register_deferred(photo, dfr)
                    dfr.addErrback(_failure)
                else:
                    photo = flickr_photo.images
                    if not photo.references:
                        dfr, photo = _get_image_available_sizes(flickr_photo)
                        self.register_deferred(photo, dfr)
                        dfr.addErrback(_failure)
                photo.title = flickr_photo.title or '<%s>' % flickr_photo.flickr_id
                slideshow_controller.player.enqueue_to_playlist(photo)

        slideshow_controller.player.jump_to_index(index)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_slideshow_player()
        self.stop_loading_animation()


def _get_photo_thumbnail(item):
    # Load the photo thumbnail
    thumbnail_uri = item.images.references[0]
    try:
        thumbnail_file = item.thumbnail_file
    except AttributeError:
        item.thumbnail_file = _thumbnail_file(thumbnail_uri)
        thumbnail_file = item.thumbnail_file
    if os.path.exists(thumbnail_file):
        return defer.succeed(thumbnail_file)
    else:
        if thumbnail_uri.scheme == 'file':
            # Image is local (default picture)
            thumbnail_file = thumbnail_uri._path
            return defer.succeed(thumbnail_file)

        # Download the thumbnail first
        def got_thumbnail(data_model):
            f = file(thumbnail_file, 'wb')
            f.write(data_model.data)
            f.close()
            return thumbnail_file

        data_model, dfr = application.resource_manager.get(thumbnail_uri)
        dfr.addCallback(got_thumbnail)
        return dfr


def _cached_sizes_file(item):
    cache = os.path.join(CONFIG_DIR, 'flickr_cache')
    if not os.path.exists(cache):
        os.makedirs(cache, 0755)
    return os.path.join(cache, item.flickr_id)


def _get_image_available_sizes(item):
    # Retrieve the list of available sizes (URLs) for an image
    item.images = ImageModel()

    # Check if we have this list in the cache first, will avoid a costly call
    # to the Flickr API
    cache_file = _cached_sizes_file(item)
    if os.path.exists(cache_file):
        fd = open(cache_file, 'rb')
        item.images.references = pickle.load(fd)
        fd.close()
        return (defer.succeed(item.images), item.images)

    request = flickr.generate_call_uri(method='flickr.photos.getSizes',
                                       arguments={'photo_id': item.flickr_id})

    def got_available_sizes(result_model):
        if not item.images.references:
            item.images.references.extend(result_model.sizes.references)
            if not os.path.exists(cache_file):
                fd = open(cache_file, 'wb')
                pickle.dump(item.images.references, fd)
                fd.close()
        return item.images

    result_model, dfr = application.resource_manager.get(request)
    dfr.addCallback(got_available_sizes)
    return (dfr, item.images)


class FlickrViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        if isinstance(item, FlickrEntryPointModel):
            title = item.title
        elif isinstance(item, FlickrTagModel):
            title = item.label
        elif isinstance(item, FlickrPhotoModel):
            title = item.title or '<%s>' % item.flickr_id
        return defer.succeed(title)

    def get_default_image(self, item):
        if isinstance(item, FlickrEntryPointModel):
            resource = 'elisa.plugins.poblesec.pictures_folders'
        elif isinstance(item, FlickrTagModel):
            resource = 'elisa.plugins.poblesec.folder'
        elif isinstance(item, FlickrPhotoModel):
            resource = 'elisa.plugins.poblesec.file_picture'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, FlickrPhotoModel):
            # Load the photo thumbnail
            def get_thumbnail(images):
                return _get_photo_thumbnail(item)

            if not hasattr(item, 'images') or not item.images.references:
                images_deferred, images = _get_image_available_sizes(item)
            else:
                images_deferred = defer.succeed(item)
            images_deferred.addCallback(get_thumbnail)
            return images_deferred
        else:
            return None

    def get_preview_image(self, item, theme):
        if isinstance(item, FlickrPhotoModel):
            try:
                return item.thumbnail_file
            except AttributeError:
                return None
        else:
            return None


class FlickrPreviewListController(FlickrController, MenuItemPreviewListController):
    view_mode = FlickrViewMode


class FlickrCoverflowController(FlickrController, ImageWithReflectionCoverflowController):
    view_mode = FlickrViewMode


class FlickrGridController(FlickrController, GridItemGridController):
    view_mode = FlickrViewMode


class FlickrListSwitcherController(ListSwitcherController):
    modes = [FlickrPreviewListController,
             FlickrCoverflowController,
             FlickrGridController]
    default_mode = FlickrGridController

class FlickrListSwitcherPreviewController(FlickrListSwitcherController):
    default_mode = FlickrPreviewListController
