# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

__maintainer__ = 'Olivier Tilloy <olivier@fluendo.com>'
__maintainer2__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

from elisa.core.tests.elisa_test_case import ElisaTestCase

from elisa.core.components.metadata_capability import MetadataCapability
from elisa.core.components.model import Model
from elisa.core.manager import AlreadyRegistered
from elisa.core.metadata_manager import MetadataManager, \
                                        IncompleteMetadataResponse

from elisa.core.utils import defer

class FlexibleMeta(MetadataCapability):
    able_called_once = False
    get_called_once = False

    def able_to_handle(self, *args):
        self.able_called_once = not self.able_called_once
        return self.able

    def get_metadata(self, *args):
        self.get_called_once = not self.get_called_once
        return defer.succeed("Yay")

class MyException(Exception):
    pass

class TestMetadataManager(ElisaTestCase):

    def setUp(self):
        ElisaTestCase.setUp(self)
        self.manager = MetadataManager()

    def test_register_unregister(self):
        """
        Test whether registering and unregistering work in a very simple case.
        """
        def capability_created(capability):
            capability.rank = 100

            self.manager.register_component(capability)
            self.failUnlessIn(capability, self.manager._components)

            self.manager.unregister_component(capability)
            self.failIfIn(capability, self.manager._components)

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created)
        return dfr

    def test_register_already_registered(self):
        """
        Test that trying to register a capability that has already been
        registered fails.
        """
        def capability_created(capability):
            self.manager.register_component(capability)
            self.failUnless(capability in self.manager._components)
            self.failUnlessRaises(AlreadyRegistered,
                                  self.manager.register_component, capability)

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created)
        return dfr

    def test_ranking(self):
        """
        Test whether the capabilities are correctly ordered by decreasing rank.
        """
        def start_tests(create_result, capabilities):
            self.manager.register_component(capabilities[0])
            self.manager.register_component(capabilities[1])
            self.manager.register_component(capabilities[2])

            # Test that the list is ordered by decreasing rank
            ranks = [capability.rank for capability in self.manager._components]
            sorted_ranks = ranks[:]
            sorted_ranks.sort()
            sorted_ranks.reverse()
            self.assertEquals(ranks, sorted_ranks)

            # make a copy of the list of components
            components = self.manager._components[:]

            self.manager.unregister_component(capabilities[0])
            self.manager.unregister_component(capabilities[1])
            self.manager.unregister_component(capabilities[2])

            # register in different order
            self.manager.register_component(capabilities[1])
            self.manager.register_component(capabilities[0])
            self.manager.register_component(capabilities[2])

            # should be the same
            self.assertEquals(components, self.manager._components)

            self.manager.unregister_component(capabilities[0])
            self.manager.unregister_component(capabilities[1])
            self.manager.unregister_component(capabilities[2])

            # register again in different order
            self.manager.register_component(capabilities[0])
            self.manager.register_component(capabilities[2])
            self.manager.register_component(capabilities[1])

            # should still be the same
            self.assertEquals(components, self.manager._components)

            self.manager.unregister_component(capabilities[0])
            self.manager.unregister_component(capabilities[1])
            self.manager.unregister_component(capabilities[2])

        created = []

        def capability_created(capability, rank, created):
            capability.rank = rank
            created.append(capability)

        s1_dfr = FlexibleMeta.create()
        s1_dfr.addCallback(capability_created, 1, created)

        s3_dfr = FlexibleMeta.create()
        s3_dfr.addCallback(capability_created, 3, created)

        s123_dfr = FlexibleMeta.create()
        s123_dfr.addCallback(capability_created, 123, created)

        dfr = defer.DeferredList([s1_dfr, s3_dfr, s123_dfr])
        dfr.addCallback(start_tests, created)
        return dfr

    def test_able_and_call(self):
        """
        Test whether the able_to_handle(...) method is correctly called.
        """
        def check_called(get_result, capability):
            self.assertTrue(capability.able_called_once)
            self.assertTrue(capability.get_called_once)

        def capability_created(capability):
            capability.rank = 123
            capability.able = True
            capability.failing = False
            self.manager.register_component(capability)

            get_dfr = self.manager.get_metadata('yeah', lambda model: False)
            self.failUnlessFailure(get_dfr, IncompleteMetadataResponse)
            get_dfr.addCallback(check_called, capability)
            return get_dfr

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created)
        return dfr

    def test_multiple_able(self):
        """
        Test whether only the capabilities able to handle a request are hit.
        """
        def check_called(get_result, capabilities):
            self.assertTrue(self.manager._components[2].get_called_once)
            self.assertFalse(self.manager._components[1].get_called_once)
            self.assertTrue(self.manager._components[0].get_called_once)

        def start_tests(create_result, capabilities):
            for capability in capabilities:
                self.manager.register_component(capability)

            get_dfr = self.manager.get_metadata('yay', lambda model: False)
            self.failUnlessFailure(get_dfr, IncompleteMetadataResponse)
            get_dfr.addCallback(check_called, capabilities)
            return get_dfr

        created = []

        def capability_created(capability, rank, able, created):
            capability.rank = rank
            capability.able = able
            created.append(capability)

        first_dfr = FlexibleMeta.create()
        first_dfr.addCallback(capability_created, 100, True, created)

        second_dfr = FlexibleMeta.create()
        second_dfr.addCallback(capability_created, 200, False, created)

        third_dfr = FlexibleMeta.create()
        third_dfr.addCallback(capability_created, 300, True, created)

        dfr = defer.DeferredList([first_dfr, second_dfr, third_dfr])
        dfr.addCallback(start_tests, created)
        return dfr

    def test_metadata_filled(self):
        """
        Test the behaviour when a metadata request is correctly processed.
        Only one metadata capability is registered, it fills the requested
        metadata.
        """
        def got_metadata(get_result, model):
            self.failUnless(model.filled)

        def capability_created(capability):

            def get_metadata(model):
                model.filled = True
                return defer.succeed('Filled')

            capability.get_metadata = get_metadata
            capability.able = True
            self.manager.register_component(capability)
            model = Model()
            model.filled = False
            get_dfr = self.manager.get_metadata(model,
                                                lambda model: model.filled)
            get_dfr.addCallback(got_metadata, model)
            return get_dfr

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created)
        return dfr

    def test_metadata_filled_by_first_capability(self):
        """
        Test the behaviour when a metadata request is correctly processed.
        Two metadata capabilities are registered, the first one fills the
        metadata, therefore the second one is not tried.
        """
        def got_metadata(get_result, model):
            self.failUnless(model.filled)
            self.failIf(self.second_capability.get_called_once)

        def start_tests(create_result):
            model = Model()
            model.filled = False
            get_dfr = self.manager.get_metadata(model,
                                                lambda model: model.filled)
            get_dfr.addCallback(got_metadata, model)
            return get_dfr

        def get_metadata(model):
            model.filled = True
            return defer.succeed('Filled')

        def capability_created(capability, rank):
            capability.rank = rank
            capability.able = True
            if rank == 200:
                capability.get_metadata = get_metadata
            else:
                self.second_capability = capability
            self.manager.register_component(capability)

        first_dfr = FlexibleMeta.create()
        first_dfr.addCallback(capability_created, 200)

        second_dfr = FlexibleMeta.create()
        second_dfr.addCallback(capability_created, 100)

        dfr = defer.DeferredList([first_dfr, second_dfr])
        dfr.addCallback(start_tests)
        return dfr

    def test_metadata_filled_by_second_capability(self):
        """
        Test the behaviour when a metadata request is correctly processed.
        Two metadata capabilities are registered, the first one does not fill
        the metadata, the second one does.
        """
        def got_metadata(get_result, model):
            self.failUnless(model.filled)

        def start_tests(create_result):
            model = Model()
            model.filled = False
            get_dfr = self.manager.get_metadata(model,
                                                lambda model: model.filled)
            get_dfr.addCallback(got_metadata, model)
            return get_dfr

        def get_metadata(model):
            model.filled = True
            return defer.succeed('Filled')

        def capability_created(capability, rank):
            capability.rank = rank
            capability.able = True
            if rank == 100:
                capability.get_metadata = get_metadata
            self.manager.register_component(capability)

        first_dfr = FlexibleMeta.create()
        first_dfr.addCallback(capability_created, 200)

        second_dfr = FlexibleMeta.create()
        second_dfr.addCallback(capability_created, 100)

        dfr = defer.DeferredList([first_dfr, second_dfr])
        dfr.addCallback(start_tests)
        return dfr

    def test_one_fails(self):
        """
        Test what happens when one capability in the list fails.
        """
        def check_called(get_result):
            self.failUnless(self.manager._components[2].get_called_once)
            self.failUnless(self.manager._components[0].get_called_once)

        def start_tests(create_result, capabilities):
            for capability in capabilities:
                self.manager.register_component(capability)

            get_dfr = self.manager.get_metadata('yuhu', lambda model: False)
            self.failUnlessFailure(get_dfr, IncompleteMetadataResponse)
            get_dfr.addCallback(check_called)

            return get_dfr

        created = []

        def capability_created(capability, rank, able, created,
                               get_metadata=None):
            capability.rank = rank
            capability.able = able
            if get_metadata is not None:
                capability.get_metadata = get_metadata
            created.append(capability)

        first_dfr = FlexibleMeta.create()
        first_dfr.addCallback(capability_created, 100, True, created)

        def fail(*args):
            return defer.fail(MyException('no, dude'))

        second_dfr = FlexibleMeta.create()
        # get_metadata(*args) called on the second capability will fail,
        # the manager should swallow this failure and continue the iteration
        second_dfr.addCallback(capability_created, 200, True, created, fail)

        third_dfr = FlexibleMeta.create()
        third_dfr.addCallback(capability_created, 300, True, created)

        dfr = defer.DeferredList([first_dfr, second_dfr, third_dfr])
        dfr.addCallback(start_tests, created)
        return dfr

    def test_external_exception_raised_1(self):
        """
        Test what happens when exceptions are raised outside the twisted part.

        First test, able_to_handle(...) raises a custom exception.
        """
        def fail(*args):
            raise MyException('dead')

        def capability_created(capability, able_to_handle):
            capability.rank = 9
            capability.able_to_handle = able_to_handle

            self.manager.register_component(capability)
            get_dfr = self.manager.get_metadata('beach', lambda model: False)
            self.failUnlessFailure(get_dfr, IncompleteMetadataResponse)

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created, fail)
        return dfr

    def test_external_exception_raised_2(self):
        """
        Test what happens when exceptions are raised outside the twisted part.

        Second test, get_metadata(...) raises a custom exception.
        """
        def fail(*args):
            raise MyException('dead')

        def capability_created(capability):
            capability.rank = 7
            capability.able = True
            capability.get_metadata = fail

            self.manager.register_component(capability)
            get_dfr = self.manager.get_metadata('beach', lambda model: False)
            self.failUnlessFailure(get_dfr, IncompleteMetadataResponse)
            return get_dfr

        dfr = FlexibleMeta.create()
        dfr.addCallback(capability_created)
        return dfr

    def test_cancel_request(self):
        """
        Test that cancelling a metadata request works as expected.
        """
        # TODO: implement this test
        pass

    test_cancel_request.skip = 'This test is not implemented at the moment'
