# -*- coding: utf-8 -*-
#
# Copyright © 2008  Red Hat, Inc. All rights reserved.
# Copyright © 2008  Kushal Das <kushal@fedoraproject.org>
#
# This copyrighted material is made available to anyone wishing to use, modify,
# copy, or redistribute it subject to the terms and conditions of the GNU
# General Public License v.2.  This program is distributed in the hope that it
# will be useful, but WITHOUT ANY WARRANTY expressed or implied, including the
# implied warranties of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.  You should have
# received a copy of the GNU General Public License along with this program; if
# not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
# Floor, Boston, MA 02110-1301, USA. Any Red Hat trademarks that are
# incorporated in the source code or documentation are not subject to the GNU
# General Public License and may only be used or replicated with the express
# permission of Red Hat, Inc.
#
# Author(s): Luke Macken <lmacken@redhat.com>
#            Kushal Das <kushal@fedoraproject.org>

"""
A cross-platform graphical interface for the LiveUSBCreator
"""

import os
import sys

from time import sleep
from datetime import datetime
from PyQt4 import QtCore, QtGui
from PyQt4 import uic
from cStringIO import StringIO

from liveusb import LiveUSBCreator, LiveUSBError
from liveusb.dialog import Ui_Dialog
from liveusb.releases import releases
from liveusb.urlgrabber.grabber import URLGrabber, URLGrabError
from liveusb.urlgrabber.progress import BaseMeter

try:
    import dbus.mainloop.qt
    dbus.mainloop.qt.DBusQtMainLoop(set_as_default=True)
except:
    pass

Ui_Dialog_strIO = StringIO()
uic.compileUi(os.path.join(os.path.dirname(os.path.dirname(__file__)),"data","hp-creator.ui"),Ui_Dialog_strIO)
exec Ui_Dialog_strIO.getvalue().replace("resources_rc","liveusb.resources_rc")

class LiveUSBApp(QtGui.QApplication):
    """ Main application class """
    def __init__(self, opts, args):
        QtGui.QApplication.__init__(self, args) 
        self.mywindow = LiveUSBDialog(opts)
        self.mywindow.show()
        try:
            self.exec_()
        finally:
            self.mywindow.terminate()


class ReleaseDownloader(QtCore.QThread):

    def __init__(self, release, progress, proxies):
        QtCore.QThread.__init__(self)
        self.release = release
        self.progress = progress
        self.proxies = proxies
        for rel in releases:
            if rel['name'] == str(release):
                self.url = rel['url']
                break
        else:
            raise LiveUSBError("Unknown release: %s" % release)

    def run(self):
        self.emit(QtCore.SIGNAL("status(PyQt_PyObject)"),
                  "Downloading %s..." % os.path.basename(self.url))
        grabber = URLGrabber(progress_obj=self.progress, proxies=self.proxies)
        try:
            iso = grabber.urlgrab(self.url, reget='simple')
        except URLGrabError, e:
            self.emit(QtCore.SIGNAL("dlcomplete(PyQt_PyObject)"), e.strerror)
        else:
            self.emit(QtCore.SIGNAL("dlcomplete(PyQt_PyObject)"), iso)


class DownloadProgress(QtCore.QObject, BaseMeter):
    """ A QObject urlgrabber BaseMeter class.

    This class is called automatically by urlgrabber with our download details.
    This class then sends signals to our main dialog window to update the
    progress bar.
    """
    def start(self, filename=None, url=None, basename=None, size=None,
              now=None, text=None):
        if (size==None):
          self.emit(QtCore.SIGNAL("maxprogress(int)"), 0)
        else:
          self.emit(QtCore.SIGNAL("maxprogress(int)"), size)

    def update(self, amount_read, now=None):
        """ Update our download progressbar.

        :read: the number of bytes read so far
        """
        self.emit(QtCore.SIGNAL("progress(int)"), amount_read)

    def end(self, amount_read):
        self.update(amount_read)


class ProgressThread(QtCore.QThread):
    """ A thread that monitors the progress of Live USB creation.

    This thread periodically checks the amount of free space left on the 
    given drive and sends a signal to our main dialog window to update the
    progress bar.
    """
    totalsize = 0
    orig_free = 0
    drive = None
    get_free_bytes = None

    def set_data(self, size, drive, freebytes):
        self.totalsize = size / 1024
        self.drive = drive
        self.get_free_bytes = freebytes
        self.orig_free = self.get_free_bytes()
        self.emit(QtCore.SIGNAL("maxprogress(int)"), self.totalsize)

    def run(self):
        while True:
            free = self.get_free_bytes()
            value = (self.orig_free - free) / 1024
            self.emit(QtCore.SIGNAL("progress(int)"), value)
            if value >= self.totalsize:
                break
            sleep(4)

    def terminate(self):
        if (self.totalsize != 0):
            self.emit(QtCore.SIGNAL("progress(int)"), self.totalsize)
        else:  #stop pulse mode
            self.emit(QtCore.SIGNAL("maxprogress(int)"), 1)
            self.emit(QtCore.SIGNAL("progress(int)"), 1)
        QtCore.QThread.terminate(self)


class LiveUSBThread(QtCore.QThread):

    def __init__(self, live, progress, parent=None):
        QtCore.QThread.__init__(self, parent)
        self.progress = progress
        self.parent = parent
        self.live = live

    def status(self, text):
        self.emit(QtCore.SIGNAL("status(PyQt_PyObject)"), text)

    def run(self):
        now = datetime.now()
        try:
            self.status("Verifying filesystem...")
            self.live.verify_filesystem()
            if not self.live.drive['uuid'] and not self.live.label:
                self.status("Error: Cannot set the label or obtain " 
                            "the UUID of your device.  Unable to continue.")
                return

            #self.live.check_drive_space()

            # If the ISO looks familar, verify it's checksum
            if not self.parent.opts.noverify:
                if self.live.get_release_from_iso():
                    self.status("Verifying checksum of image...")
                    if not self.live.verify_image(progress=self):
                        self.status("Error: The checksum of your image is "
                                    "invalid.  You can run this program with "
                                    "the --noverify argument to bypass this "
                                    "verification check.")
                        return

            self.progress.set_data(size=self.live.totalsize,
                                   drive=self.live.drive['device'],
                                   freebytes=self.live.get_free_bytes)
            self.progress.start()

            self.status("Copying image to USB device...")
            self.live.copy_image()
            if self.live.overlay:
                self.status("Creating %d Mb persistent overlay..." %
                            self.live.overlay)
                self.live.create_persistent_overlay()
            #self.status("Configuring and installing bootloader...")
            #self.live.update_configs()
            #self.live.install_bootloader()
            duration = str(datetime.now() - now).split('.')[0]
            self.status("Complete! (%s)" % duration)
        except LiveUSBError, e:
            self.status(str(e))
            self.status("LiveUSB creation failed!")
        except Exception, e:
            self.status(str(e))
            self.status("LiveUSB creation failed!")
            import traceback
            traceback.print_exc()

        try:
            self.live.unmount_device()
        except Exception, e:
            pass
        self.progress.terminate()

    def set_max_progress(self, maximum):
        self.emit(QtCore.SIGNAL("maxprogress(int)"), maximum)

    def update_progress(self, value):
        self.emit(QtCore.SIGNAL("progress(int)"), value)

    def __del__(self):
        self.wait()


class LiveUSBDialog(QtGui.QDialog, Ui_Dialog):
    """ Our main dialog class """
    def __init__(self, opts):
        QtGui.QDialog.__init__(self)
        Ui_Dialog.__init__(self)
        self.opts = opts
        self.setupUi(self)
        self.live = LiveUSBCreator(opts=opts)
        self.populate_releases()
        self.populate_devices()
        self.downloader = None
        self.progress_thread = ProgressThread()
        self.download_progress = DownloadProgress()
        self.live_thread = LiveUSBThread(live=self.live,
                                        progress=self.progress_thread,
                                        parent=self)
        self.connect_slots()
        self.confirmed = False

    def populate_devices(self, *args, **kw):
        self.driveBox.clear()
        self.textEdit.clear()
        try:
            self.live.detect_removable_drives()
            for device, info in self.live.drives.items():
                if info['label']:
                    self.driveBox.addItem("%s (%s)" % (device, info['label']))
                else:
                    self.driveBox.addItem(device)
            self.startButton.setEnabled(True)
        except LiveUSBError, e:
            self.textEdit.setPlainText(str(e))
            self.startButton.setEnabled(False)

    def populate_releases(self):
        for release in [release['name'] for release in releases]:
            self.downloadCombo.addItem(release)

    def connect_slots(self):
        self.connect(self.isoBttn, QtCore.SIGNAL("clicked()"), self.selectfile)
        self.connect(self.startButton, QtCore.SIGNAL("clicked()"), self.begin)
        #self.connect(self.overlaySlider, QtCore.SIGNAL("valueChanged(int)"),
        #             self.overlay_value)
        self.connect(self.live_thread, QtCore.SIGNAL("status(PyQt_PyObject)"),
                     self.status)
        self.connect(self.live_thread, QtCore.SIGNAL("finished()"),
                     lambda: self.enable_widgets(True))
        self.connect(self.live_thread, QtCore.SIGNAL("terminated()"),
                     lambda: self.enable_widgets(True))
        self.connect(self.live_thread, QtCore.SIGNAL("progress(int)"),
                     self.progress)
        self.connect(self.live_thread, QtCore.SIGNAL("maxprogress(int)"),
                     self.maxprogress)
        self.connect(self.progress_thread, QtCore.SIGNAL("progress(int)"),
                     self.progress)
        self.connect(self.progress_thread, QtCore.SIGNAL("maxprogress(int)"),
                     self.maxprogress)
        self.connect(self.download_progress, QtCore.SIGNAL("maxprogress(int)"),
                     self.maxprogress)
        self.connect(self.download_progress, QtCore.SIGNAL("progress(int)"),
                     self.progress)
        self.connect(self.refreshDevicesButton, QtCore.SIGNAL("clicked()"),
                     self.populate_devices)

        # If we have access to HAL & DBus, intercept some useful signals
        if hasattr(self.live, 'hal'):
            self.live.hal.connect_to_signal('DeviceAdded',
                                            self.populate_devices)
            self.live.hal.connect_to_signal('DeviceRemoved',
                                            self.populate_devices)

    @QtCore.pyqtSignature("QString")
    def on_driveBox_currentIndexChanged(self, drive):
        """ Change the maximum overlay size when each drive is selected.

        This sets the maximum megabyte size of the persistent storage slider
        to the number of free megabytes on the currently selected
        "Target Device".  If the device is not mounted, or if it has more than
        2gigs of free space, set the maximum to 2047mb, which is apparently
        the largest file we can/should store on a vfat partition.
        """
        if not str(drive):
            return
        freespace = self.live.drives[str(drive).split()[0]]['free']
        if not freespace or freespace > 2047:
            freespace = 2047
        #self.overlaySlider.setMaximum(freespace)

    def progress(self, value):
        self.progressBar.setValue(value)

    def maxprogress(self, value):
        self.progressBar.setMaximum(value)

    def status(self, text):
        self.textEdit.append(text)

    def enable_widgets(self, enabled=True):
        self.startButton.setEnabled(enabled)
        self.driveBox.setEnabled(enabled)
        #self.overlaySlider.setEnabled(enabled)
        self.isoBttn.setEnabled(enabled)
        self.downloadCombo.setEnabled(enabled)
        self.refreshDevicesButton.setEnabled(enabled)

    def overlay_value(self, value):
        self.overlayTitle.setTitle("Persistent Storage (%d Mb)" % value)

    def get_selected_drive(self):
        return str(self.driveBox.currentText()).split()[0]

    def begin(self):
        self.enable_widgets(False)
        #self.live.overlay = self.overlaySlider.value()
        self.live.overlay = 0
        self.live.drive = self.get_selected_drive()
        try:
            self.live.mount_device()
        except LiveUSBError, e:
            self.status(str(e))
            self.enable_widgets(True)
            return 

        if self.live.existing_liveos():
            if not self.confirmed:
                self.status("Your device already contains a LiveOS.\nIf you "
                            "continue, this will be overwritten.")
                if self.live.existing_overlay() and self.overlaySlider.value():
                    self.status("Warning: Creating a new persistent overlay "
                                "will delete your existing one.")
                self.status("Press 'Create Live USB' again if you wish to "
                            "continue.")
                self.confirmed = True
                self.live.unmount_device()
                self.enable_widgets(True)
                return
            else:
                # The user has confirmed that they wish to overwrite their
                # existing Live OS.  Here we delete it first, in order to 
                # accurately calculate progress.
                self.status("Removing existing Live OS...")
                try:
                    self.live.delete_liveos()
                except LiveUSBError, e:
                    self.status(str(e))
                    self.live.unmount_device()
                    self.enable_widgets(True)
                    return

        # If the user has selected an ISO, use it.  If not, download one.
        if self.live.iso:
            try:
                self.live.unmount_device()
            except Exception, e:
                pass
            self.live_thread.start()
            try:
                self.live.mount_device()
            except Excepton, e:
                pass
        else:
            self.downloader = ReleaseDownloader(
                    self.downloadCombo.currentText(),
                    progress=self.download_progress,
                    proxies=self.live.get_proxies())
            self.connect(self.downloader,
                         QtCore.SIGNAL("dlcomplete(PyQt_PyObject)"),
                         self.download_complete)
            self.connect(self.downloader,
                         QtCore.SIGNAL("status(PyQt_PyObject)"),
                         self.status)
            self.downloader.start()

    def download_complete(self, iso):
        """ Called by our ReleaseDownloader thread upon completion.

        Upon success, the thread passes in the filename of the downloaded
        release.  If the 'iso' argument is not an existing file, then
        it is assumed that the download failed and 'iso' should contain
        the error message.
        """
        if os.path.exists(iso):
            self.status("Download complete!")
            self.live.iso = iso
            try:
                self.live.unmount_device()
            except Exception, e:
                pass
            self.live_thread.start()
            try:
                self.live.mount_device()
            except Excepton, e:
                pass
        else:
            self.status("Download failed: " + iso)
            self.status("You can try again to resume your download")
            self.enable_widgets(True)

    def selectfile(self):
        isofile = QtGui.QFileDialog.getOpenFileName(self, "Select Image",
                                                    ".", "IMG (*.img)" )
        if isofile:
            try:
                self.live.iso = self._to_unicode(isofile)
            except Exception, e:
                self.live.log.error(str(e))
                self.status("Sorry, I'm having trouble encoding the filename "
                            "of your livecd.  You may have better luck if "
                            "you move your ISO to the root of your drive "
                            "(ie: C:\)")

            self.live.log.info("ISO selected: %s" % repr(self.live.iso))
            self.textEdit.append(os.path.basename(self.live.iso) + ' selected')

    def terminate(self):
        """ Terminate any processes that we have spawned """
        self.live.terminate()

    def _to_unicode(self, obj, encoding='utf-8'):
        if hasattr(obj, 'toUtf8'): # PyQt4.QtCore.QString
            obj = str(obj.toUtf8())
        if isinstance(obj, basestring):
            if not isinstance(obj, unicode):
                obj = unicode(obj, encoding, 'replace')
        return obj
