/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * Context class is used to create and handle the OpenGL context. It retrieves
 * the capabilities of the OpenGL implementation at runtime and stores the
 * results in its version, limits and feature mask data.
 *
 * It creates a thread dedicated to the rendering. Only this thread communicates
 * with the native OpenGL libraries for the entire process. It allows the
 * application side to do blocking operations without stopping the rendering.
 *
 * Context handles two task queues for other threads to push operations. The
 * first queue is used to push tasks that need to be complete as soon as
 * possible in the rendering thread, the tasks pushed in this queue are called
 * "immediate tasks". Typical immediate tasks can be for example requests to
 * change the viewport size or title. The second queue is used to push
 * tasks that only need to be complete on a viewport update request. Such tasks
 * are called "deferred tasks". Deferred tasks can be for example texture
 * uploads or projection matrix updates. The immediate task queue is flushed
 * each time a task is pushed with the pgm_context_push_immediate_task()
 * function. The deferred task queue is flushed each time an update is requested
 * with the pgm_context_update() function.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmcontext.h"

#ifdef G_OS_UNIX
#include <unistd.h> /* pipe, close */
#endif /* G_OS_UNIX */

#ifdef WIN32
#include <windows.h>
#include <fcntl.h>   /* _O_BINARY */
#include <io.h>      /* _pipe */
#define pipe(handle) _pipe(handle, 4096, _O_BINARY)
#endif /* WIN32 */

#include <GL/gl.h>  /* gl* */
#include <string.h> /* strlen */
#include <ctype.h>  /* isdigit */
#include <math.h>   /* tan */

#ifdef HAVE_AGL_AGL_H
#include "pgmaglbackend.h"
#elif defined HAVE_GL_GLX_H
#include "pgmglxbackend.h"
#elif defined HAVE_WGL_H
#include "pgmwglbackend.h"
#endif

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

/* Function prototypes */
static void flush_task_queue (PgmContext *context, GList **queue);
static void render           (PgmContext *context);

/* Map OpenGL extension names and feature bits */
typedef struct {
  const gchar *name;
  gint         feature_mask;
} ExtensionMap;

/* OpenGL extensions checked at runtime */
static ExtensionMap gl_extensions_map[] = {
  { "GL_ARB_texture_rectangle",        PGM_GL_FEAT_TEXTURE_RECTANGLE        },
  { "GL_EXT_texture_rectangle",        PGM_GL_FEAT_TEXTURE_RECTANGLE        },
  { "GL_NV_texture_rectangle",         PGM_GL_FEAT_TEXTURE_RECTANGLE        },
  { "GL_ARB_texture_non_power_of_two", PGM_GL_FEAT_TEXTURE_NON_POWER_OF_TWO },
  { "GL_ARB_texture_mirrored_repeat",  PGM_GL_FEAT_TEXTURE_MIRRORED_REPEAT  },
  { "GL_ARB_texture_border_clamp",     PGM_GL_FEAT_TEXTURE_BORDER_CLAMP     },
  { "GL_ARB_texture_env_combine",      PGM_GL_FEAT_TEXTURE_ENV_COMBINE      },
  { "GL_EXT_texture_env_combine",      PGM_GL_FEAT_TEXTURE_ENV_COMBINE      },
  { "GL_ARB_texture_env_dot3",         PGM_GL_FEAT_TEXTURE_ENV_DOT3         },
  { "GL_ARB_multisample",              PGM_GL_FEAT_MULTISAMPLE              },
  { "GL_NV_multisample_filter_hint",   PGM_GL_FEAT_MULTISAMPLE_FILTER_HINT  },
  { "GL_ARB_multitexture",             PGM_GL_FEAT_MULTITEXTURE             },
  { "GL_EXT_multi_draw_arrays",        PGM_GL_FEAT_MULTI_DRAW_ARRAYS        },
  { "GL_ARB_fragment_program",         PGM_GL_FEAT_FRAGMENT_PROGRAM         },
  { "GL_ARB_vertex_buffer_object",     PGM_GL_FEAT_VERTEX_BUFFER_OBJECT     },
  { "GL_ARB_pixel_buffer_object",      PGM_GL_FEAT_PIXEL_BUFFER_OBJECT      },
  { "GL_EXT_pixel_buffer_object",      PGM_GL_FEAT_PIXEL_BUFFER_OBJECT      },
  { "GL_EXT_blend_func_separate",      PGM_GL_FEAT_BLEND_FUNC_SEPARATE      },
  { "GL_EXT_blend_color",              PGM_GL_FEAT_BLEND_COLOR              },
  { "GL_ARB_imaging",                  PGM_GL_FEAT_BLEND_COLOR              },
  { "GL_APPLE_packed_pixels",          PGM_GL_FEAT_PACKED_PIXELS            },
  { "GL_EXT_framebuffer_object",       PGM_GL_FEAT_FRAMEBUFFER_OBJECT       },
  { NULL,                              0                                    }
};

/* OpenGL functions binding */
static PgmContextProcAddress gl_proc_address = {
  /* OpenGL 1.2 core functions */
  (pgm_gl_enable)                glEnable,
  (pgm_gl_disable)               glDisable,
  (pgm_gl_get_error)             glGetError,
  (pgm_gl_get_string)            glGetString,
  (pgm_gl_begin)                 glBegin,
  (pgm_gl_end)                   glEnd,
  (pgm_gl_vertex_3f)             glVertex3f,
  (pgm_gl_tex_coord_2f)          glTexCoord2f,
  (pgm_gl_enable_client_state)   glEnableClientState,
  (pgm_gl_disable_client_state)  glDisableClientState,
  (pgm_gl_vertex_pointer)        glVertexPointer,
  (pgm_gl_color_pointer)         glColorPointer,
  (pgm_gl_tex_coord_pointer)     glTexCoordPointer,
  (pgm_gl_draw_arrays)           glDrawArrays,
  (pgm_gl_tex_env_f)             glTexEnvf,
  (pgm_gl_tex_env_fv)            glTexEnvfv,
  (pgm_gl_tex_gen_i)             glTexGeni,
  (pgm_gl_tex_gen_fv)            glTexGenfv,
  (pgm_gl_color_4f)              glColor4f,
  (pgm_gl_color_4fv)             glColor4fv,
  (pgm_gl_scissor)               glScissor,
  (pgm_gl_depth_func)            glDepthFunc,
  (pgm_gl_blend_func)            glBlendFunc,
  (pgm_gl_line_width)            glLineWidth,
  (pgm_gl_clear)                 glClear,
  (pgm_gl_clear_color)           glClearColor,
  (pgm_gl_clear_stencil)         glClearStencil,
  (pgm_gl_stencil_func)          glStencilFunc,
  (pgm_gl_stencil_op)            glStencilOp,
  (pgm_gl_push_attrib)           glPushAttrib,
  (pgm_gl_pop_attrib)            glPopAttrib,
  (pgm_gl_matrix_mode)           glMatrixMode,
  (pgm_gl_push_matrix)           glPushMatrix,
  (pgm_gl_pop_matrix)            glPopMatrix,
  (pgm_gl_load_identity)         glLoadIdentity,
  (pgm_gl_load_matrix_f)         glLoadMatrixf,
  (pgm_gl_depth_range)           glDepthRange,
  (pgm_gl_viewport)              glViewport,
  (pgm_gl_raster_pos_2f)         glRasterPos2f,
  (pgm_gl_bitmap)                glBitmap,
  (pgm_gl_read_buffer)           glReadBuffer,
  (pgm_gl_draw_buffer)           glDrawBuffer,
  (pgm_gl_copy_pixels)           glCopyPixels,
  (pgm_gl_flush)                 glFlush,
  (pgm_gl_finish)                glFinish,
  (pgm_gl_pixel_store_i)         glPixelStorei,
  (pgm_gl_frustum)               glFrustum,
  (pgm_gl_ortho)                 glOrtho,
  (pgm_gl_scale_f)               glScalef,
  (pgm_gl_translate_f)           glTranslatef,
  (pgm_gl_rotate_f)              glRotatef,
  (pgm_gl_is_list)               glIsList,
  (pgm_gl_delete_lists)          glDeleteLists,
  (pgm_gl_gen_lists)             glGenLists,
  (pgm_gl_new_list)              glNewList,
  (pgm_gl_end_list)              glEndList,
  (pgm_gl_call_list)             glCallList,
  (pgm_gl_call_lists)            glCallLists,
  (pgm_gl_list_base)             glListBase,
  (pgm_gl_hint)                  glHint,
  (pgm_gl_depth_mask)            glDepthMask,
  (pgm_gl_polygon_mode)          glPolygonMode,
  (pgm_gl_shade_model)           glShadeModel,
  (pgm_gl_color_mask)            glColorMask,
  (pgm_gl_read_pixels)           glReadPixels,
  (pgm_gl_get_tex_image)         glGetTexImage,
  (pgm_gl_tex_sub_image_2d)      glTexSubImage2D,
  (pgm_gl_gen_textures)          glGenTextures,
  (pgm_gl_delete_textures)       glDeleteTextures,
  (pgm_gl_bind_texture)          glBindTexture,
  (pgm_gl_tex_image_2d)          glTexImage2D,
  (pgm_gl_tex_parameter_i)       glTexParameteri,
  (pgm_gl_tex_parameter_fv)      glTexParameterfv,
  (pgm_gl_copy_tex_sub_image_2d) glCopyTexSubImage2D,
  (pgm_gl_get_integer_v)         glGetIntegerv,
  (pgm_gl_get_float_v)           glGetFloatv,

  /* OpenGL 1.2 extension functions */
  (pgm_gl_blend_func_separate)           NULL,
  (pgm_gl_blend_color)                   NULL,
  (pgm_gl_active_texture)                NULL,
  (pgm_gl_client_active_texture)         NULL,
  (pgm_gl_multi_draw_arrays)             NULL,
  (pgm_gl_gen_programs)                  NULL,
  (pgm_gl_delete_programs)               NULL,
  (pgm_gl_program_string)                NULL,
  (pgm_gl_bind_program)                  NULL,
  (pgm_gl_program_local_param_4fv)       NULL,
  (pgm_gl_get_program_iv)                NULL,
  (pgm_gl_gen_buffers)                   NULL,
  (pgm_gl_delete_buffers)                NULL,
  (pgm_gl_bind_buffer)                   NULL,
  (pgm_gl_buffer_data)                   NULL,
  (pgm_gl_buffer_sub_data)               NULL,
  (pgm_gl_get_buffer_sub_data)           NULL,
  (pgm_gl_map_buffer)                    NULL,
  (pgm_gl_unmap_buffer)                  NULL,
  (pgm_gl_gen_framebuffers)              NULL,
  (pgm_gl_delete_framebuffers)           NULL,
  (pgm_gl_bind_framebuffer)              NULL,
  (pgm_gl_framebuffer_renderbuffer)      NULL,
  (pgm_gl_framebuffer_texture_2d)        NULL,
  (pgm_gl_check_framebuffer_status)      NULL,
  (pgm_gl_gen_renderbuffers)             NULL,
  (pgm_gl_delete_renderbuffers)          NULL,
  (pgm_gl_bind_renderbuffer)             NULL,
  (pgm_gl_renderbuffer_storage)          NULL,
  (pgm_gl_get_renderbuffer_parameter_iv) NULL
};

/* Update OpenGL viewport and projection */
static void
update_projection (PgmContext *context)
{
  PgmGlViewport *glviewport = context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);
  PgmContextProcAddress *gl = context->gl;
  gint projected_width, projected_height;
  gint projected_x, projected_y;
  PgmMat4x4 *projection, *transpose;

  /* Retrieve viewport informations with lock */
  GST_OBJECT_LOCK (viewport);
  projected_width = viewport->projected_width;
  projected_height = viewport->projected_height;
  projected_x = viewport->projected_x;
  projected_y = viewport->projected_y;
  projection = viewport->projection;
  GST_OBJECT_UNLOCK (viewport);

  /* Set up viewport */
  gl->viewport (projected_x, projected_y, projected_width, projected_height);

  /* Update projection matrix */
  gl->matrix_mode (PGM_GL_PROJECTION);
  gl->load_identity ();
  transpose = pgm_mat4x4_transpose (projection);
  gl->load_matrix_f (transpose->m);
  pgm_mat4x4_free (transpose);
  gl->matrix_mode (PGM_GL_MODELVIEW);

  /* Update all the drawables inside the viewport */
  pgm_gl_viewport_update_drawable_projection (glviewport);

#if 0
  /* Display projection matrix */
  PgmGlFloat m[16];
  gl->get_float_v (PGM_GL_PROJECTION_MATRIX, m);
  g_print ("[ %7.2f %7.2f %7.2f %7.2f ]\n[ %7.2f %7.2f %7.2f %7.2f ]\n"
           "[ %7.2f %7.2f %7.2f %7.2f ]\n[ %7.2f %7.2f %7.2f %7.2f ]\n\n",
           m[0], m[1], m[2], m[3], m[4], m[5], m[6], m[7], m[8], m[9], m[10],
           m[11], m[12], m[13], m[14], m[15]);
#endif
}

/* Set up initial OpenGL states */
static void
init_opengl_states (PgmContext *context)
{
  PgmContextProcAddress *gl = context->gl;

  /* Enable texture mapping */
  gl->enable (PGM_GL_TEXTURE_2D);

  /* Disable depth testing */
  gl->disable (PGM_GL_DEPTH_TEST);

  /* Set up standard blending with a different alpha function to correctly
   * support compositing with the desktop when the alpha clearing color is
   * available (ARGB visual) */
  if (context->feature_mask & PGM_GL_FEAT_BLEND_FUNC_SEPARATE)
    gl->blend_func_separate (PGM_GL_SRC_ALPHA, PGM_GL_ONE_MINUS_SRC_ALPHA,
                             PGM_GL_SRC_ALPHA, PGM_GL_ONE);
  else
    gl->blend_func (PGM_GL_SRC_ALPHA, PGM_GL_ONE_MINUS_SRC_ALPHA);
  gl->enable (PGM_GL_BLEND);
}

/* Check whether an extension is supported by the OpenGL implementation given
 * the extension name and the list of supported extensions */
static gboolean
has_opengl_extension (const gchar *extensions,
                      const gchar *extension_name)
{
  gchar *end;
  size_t ext_name_len = strlen (extension_name);
  gchar *p = (gchar*) extensions;
  size_t size;

  if (!extensions)
    return FALSE;

  end = p + strlen (p);
  while (p < end)
    {
      size = strcspn (p, " ");
      if ((ext_name_len == size)  && (strncmp (extension_name, p, size) == 0))
        return TRUE;
      p += size + 1;
    }

  return FALSE;
}

/* Retrieve the OpenGL version as a float given the string version */
static gfloat
get_float_opengl_version (const gchar *version_string)
{
  gfloat version = 0.0f;
  gint i;

  for (i = 0; isdigit (version_string[i]); i++)
    version = version * 10.0f + (version_string[i] - 48);

  if (version_string[i++] != '.')
    return 0.0f;

  version = version * 10.0f + (version_string[i] - 48);

  return (version + 0.1f) / 10.0f;
}

/* Bind the OpenGL extension proc addresses considering OpenGL 1.2 as the
 * minimal required base. Depending on the run-time OpenGL implementation
 * version, extensions could be considered as core functionnalities. So
 * depending on the extension and the version, the logic is to retrieve the
 * core proc addresses or the extension proc addresses. */
static void
bind_opengl_extensions (PgmContext *context)
{
  PgmContextProcAddress *gl = context->gl;

  /* Blend func separate */
  if (context->feature_mask & PGM_GL_FEAT_BLEND_FUNC_SEPARATE)
    {
      if (context->version >= 1.4f)
        gl->blend_func_separate = (pgm_gl_blend_func_separate)
          pgm_backend_get_proc_address (context->backend,
                                        "glBlendFuncSeparate");
      else
        gl->blend_func_separate = (pgm_gl_blend_func_separate)
          pgm_backend_get_proc_address (context->backend,
                                        "glBlendFuncSeparateEXT");
      if (!gl->blend_func_separate)
        context->feature_mask &= ~PGM_GL_FEAT_BLEND_FUNC_SEPARATE;
    }

  /* Blend color */
  if (context->feature_mask & PGM_GL_FEAT_BLEND_COLOR)
    {
      if (context->version >= 1.4f)
        gl->blend_color = (pgm_gl_blend_color)
          pgm_backend_get_proc_address (context->backend, "glBlendColor");
      else
        gl->blend_color = (pgm_gl_blend_color)
          pgm_backend_get_proc_address (context->backend, "glBlendColorEXT");
      if (!gl->blend_color)
        context->feature_mask &= ~PGM_GL_FEAT_BLEND_COLOR;
    }

  /* Multi-texturing */
  if (context->feature_mask & PGM_GL_FEAT_MULTITEXTURE)
    {
      if (context->version >= 1.3f)
        {
          gl->active_texture = (pgm_gl_active_texture)
            pgm_backend_get_proc_address (context->backend, "glActiveTexture");
          gl->client_active_texture = (pgm_gl_client_active_texture)
            pgm_backend_get_proc_address (context->backend,
                                          "glClientActiveTexture");
        }
      else
        {
          gl->active_texture = (pgm_gl_active_texture)
            pgm_backend_get_proc_address (context->backend,
                                          "glActiveTextureARB");
          gl->client_active_texture = (pgm_gl_client_active_texture)
            pgm_backend_get_proc_address (context->backend,
                                          "glClientActiveTextureARB");
        }
      if ((!gl->active_texture) || (!gl->client_active_texture))
        context->feature_mask &= ~PGM_GL_FEAT_MULTITEXTURE;
    }

  /* Multi draw arrays */
  if (context->feature_mask & PGM_GL_FEAT_MULTI_DRAW_ARRAYS)
    {
      if (context->version >= 1.4f)
        gl->multi_draw_arrays = (pgm_gl_multi_draw_arrays)
          pgm_backend_get_proc_address (context->backend, "glMultiDrawArrays");
      else
        gl->multi_draw_arrays = (pgm_gl_multi_draw_arrays)
          pgm_backend_get_proc_address (context->backend,
                                        "glMultiDrawArraysEXT");
      if (!gl->multi_draw_arrays)
        context->feature_mask &= ~PGM_GL_FEAT_MULTI_DRAW_ARRAYS;
    }

  /* Fragment program */
  if (context->feature_mask & PGM_GL_FEAT_FRAGMENT_PROGRAM)
    {
      gl->gen_programs = (pgm_gl_gen_programs)
        pgm_backend_get_proc_address (context->backend, "glGenProgramsARB");
      gl->delete_programs = (pgm_gl_delete_programs)
        pgm_backend_get_proc_address (context->backend, "glDeleteProgramsARB");
      gl->program_string = (pgm_gl_program_string)
        pgm_backend_get_proc_address (context->backend, "glProgramStringARB");
      gl->bind_program = (pgm_gl_bind_program)
        pgm_backend_get_proc_address (context->backend, "glBindProgramARB");
      gl->program_local_param_4fv = (pgm_gl_program_local_param_4fv)
        pgm_backend_get_proc_address (context->backend,
                                      "glProgramLocalParameter4fvARB");
      gl->get_program_iv = (pgm_gl_get_program_iv)
        pgm_backend_get_proc_address (context->backend, "glGetProgramivARB");

      if ((!gl->gen_programs) || (!gl->delete_programs) || (!gl->program_string)
          || (!gl->bind_program) || (!gl->program_local_param_4fv))
        context->feature_mask &= ~PGM_GL_FEAT_FRAGMENT_PROGRAM;
    }

  /* VBO and PBO */
  if ((context->feature_mask & PGM_GL_FEAT_VERTEX_BUFFER_OBJECT)
      || (context->feature_mask & PGM_GL_FEAT_PIXEL_BUFFER_OBJECT))
    {
      if (context->version >= 1.5f)
        {
          gl->gen_buffers = (pgm_gl_gen_buffers)
            pgm_backend_get_proc_address (context->backend, "glGenBuffers");
          gl->delete_buffers = (pgm_gl_delete_buffers)
            pgm_backend_get_proc_address (context->backend, "glDeleteBuffers");
          gl->bind_buffer = (pgm_gl_bind_buffer)
            pgm_backend_get_proc_address (context->backend, "glBindBuffer");
          gl->buffer_data = (pgm_gl_buffer_data)
            pgm_backend_get_proc_address (context->backend, "glBufferData");
          gl->buffer_sub_data = (pgm_gl_buffer_sub_data)
            pgm_backend_get_proc_address (context->backend, "glBufferSubData");
          gl->get_buffer_sub_data = (pgm_gl_get_buffer_sub_data)
            pgm_backend_get_proc_address (context->backend, "glGetBufferSubData");
          gl->map_buffer = (pgm_gl_map_buffer)
            pgm_backend_get_proc_address (context->backend, "glMapBuffer");
          gl->unmap_buffer = (pgm_gl_unmap_buffer)
            pgm_backend_get_proc_address (context->backend, "glUnmapBuffer");
        }
      else
        {
          gl->gen_buffers = (pgm_gl_gen_buffers)
            pgm_backend_get_proc_address (context->backend, "glGenBuffersARB");
          gl->delete_buffers = (pgm_gl_delete_buffers)
            pgm_backend_get_proc_address (context->backend, "glDeleteBuffersARB");
          gl->bind_buffer = (pgm_gl_bind_buffer)
            pgm_backend_get_proc_address (context->backend, "glBindBufferARB");
          gl->buffer_data = (pgm_gl_buffer_data)
            pgm_backend_get_proc_address (context->backend, "glBufferDataARB");
          gl->buffer_sub_data = (pgm_gl_buffer_sub_data)
            pgm_backend_get_proc_address (context->backend, "glBufferSubDataARB");
          gl->get_buffer_sub_data = (pgm_gl_get_buffer_sub_data)
            pgm_backend_get_proc_address (context->backend,
                                          "glGetBufferSubDataARB");
          gl->map_buffer = (pgm_gl_map_buffer)
            pgm_backend_get_proc_address (context->backend, "glMapBufferARB");
          gl->unmap_buffer = (pgm_gl_unmap_buffer)
            pgm_backend_get_proc_address (context->backend, "glUnmapBufferARB");
        }

      if ((!gl->gen_buffers) || (!gl->delete_buffers) || (!gl->bind_buffer)
          || (!gl->buffer_data) || (!gl->buffer_sub_data)
          || (!gl->get_buffer_sub_data) || (!gl->map_buffer)
          || (!gl->unmap_buffer))
        {
          context->feature_mask &= ~PGM_GL_FEAT_VERTEX_BUFFER_OBJECT;
          context->feature_mask &= ~PGM_GL_FEAT_PIXEL_BUFFER_OBJECT;
        }
    }

  /* FBO */
  if (context->feature_mask & PGM_GL_FEAT_FRAMEBUFFER_OBJECT)
    {
      gl->gen_framebuffers = (pgm_gl_gen_framebuffers)
        pgm_backend_get_proc_address (context->backend, "glGenFramebuffersEXT");
      gl->delete_framebuffers = (pgm_gl_delete_framebuffers)
        pgm_backend_get_proc_address (context->backend,
                                      "glDeleteFramebuffersEXT");
      gl->bind_framebuffer = (pgm_gl_bind_framebuffer)
        pgm_backend_get_proc_address (context->backend, "glBindFramebufferEXT");
      gl->framebuffer_renderbuffer = (pgm_gl_framebuffer_renderbuffer)
        pgm_backend_get_proc_address (context->backend,
                                      "glFramebufferRenderbufferEXT");
      gl->framebuffer_texture_2d = (pgm_gl_framebuffer_texture_2d)
        pgm_backend_get_proc_address (context->backend,
                                      "glFramebufferTexture2DEXT");
      gl->check_framebuffer_status = (pgm_gl_check_framebuffer_status)
        pgm_backend_get_proc_address (context->backend,
                                      "glCheckFramebufferStatusEXT");
      gl->gen_renderbuffers = (pgm_gl_gen_renderbuffers)
        pgm_backend_get_proc_address (context->backend,
                                      "glGenRenderbuffersEXT");
      gl->delete_renderbuffers = (pgm_gl_delete_renderbuffers)
        pgm_backend_get_proc_address (context->backend,
                                      "glDeleteRenderbuffersEXT");
      gl->bind_renderbuffer = (pgm_gl_bind_renderbuffer)
        pgm_backend_get_proc_address (context->backend,
                                      "glBindRenderbufferEXT");
      gl->renderbuffer_storage = (pgm_gl_renderbuffer_storage)
        pgm_backend_get_proc_address (context->backend,
                                      "glRenderbufferStorageEXT");
      gl->get_renderbuffer_parameter_iv = (pgm_gl_get_renderbuffer_parameter_iv)
        pgm_backend_get_proc_address (context->backend,
                                      "glGetRenderbufferParameterivEXT");

      if ((!gl->gen_framebuffers) || (!gl->delete_framebuffers)
          || (!gl->bind_framebuffer) || (!gl->framebuffer_renderbuffer)
          || (!gl->framebuffer_texture_2d) || (!gl->check_framebuffer_status)
          || (!gl->gen_renderbuffers) || (!gl->delete_renderbuffers)
          || (!gl->bind_renderbuffer) || (!gl->renderbuffer_storage)
          || (!gl->get_renderbuffer_parameter_iv))
        context->feature_mask &= ~PGM_GL_FEAT_FRAMEBUFFER_OBJECT;
    }
}

/* Compute the frame rate per second */
static void
compute_frame_rate (PgmContext *context)
{
  GTimeVal current_time;
  gfloat elapsed_time;
  static guint fps = 0;

  /* Retrieve current time and compute the time elapsed since the last tick */
  g_get_current_time (&current_time);
  elapsed_time = (current_time.tv_sec - context->fps_tick_time.tv_sec)
    + (current_time.tv_usec - context->fps_tick_time.tv_usec) * 0.000001f;

  /* Is the second elapsed? */
  if (elapsed_time >= 1.0f)
    {
      context->fps = fps;

      /* Set the new tick time */
      context->fps_tick_time.tv_sec = current_time.tv_sec;
      context->fps_tick_time.tv_usec = current_time.tv_usec;

      /* Reset the counter */
      fps = 0;
    }

  fps++;

#if 0 /* FIXME: Broken */
  {
    gchar fps_string[16];

    /* Build and rasterize the FPS string */
    g_snprintf (fps_string, 16, "%d FPS", fps);
    context->gl->disable (PGM_GL_TEXTURE_2D);
    pgm_backend_raster_text (context->backend, fps_string, 6.0f, 15.0f,
                             0.0f, 0.0f, 0.0f);
    pgm_backend_raster_text (context->backend, fps_string, 5.0f, 14.0f,
                             1.0f, 1.0f, 1.0f);
    context->gl->enable (PGM_GL_TEXTURE_2D);
  }
#endif
}

/* Viewport update task handler */
static void
do_projection (PgmContext *context,
               gpointer data)
{
  update_projection (context);
  pgm_context_update (context);
}

/* Size update task handler */
static void
do_size (PgmContext *context,
         gpointer data)
{
  gint width, height;

  GST_OBJECT_LOCK (context->glviewport);
  width = PGM_VIEWPORT (context->glviewport)->width;
  height = PGM_VIEWPORT (context->glviewport)->height;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_size (context->backend, width, height);
  pgm_context_update (context);
}

/* Title update task handler */
static void
do_title (PgmContext *context,
          gpointer data)
{
  gchar *title;

  GST_OBJECT_LOCK (context->glviewport);
  title = g_strdup (PGM_VIEWPORT (context->glviewport)->title);
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_title (context->backend, title);

  g_free (title);
}

/* Decoration update task handler */
static void
do_decoration (PgmContext *context,
               gpointer data)
{
  gboolean decorated;

  GST_OBJECT_LOCK (context->glviewport);
  decorated = PGM_VIEWPORT (context->glviewport)->decorated;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_decorated (context->backend, decorated);
}

/* Fullscreen update task handler */
static void
do_fullscreen (PgmContext *context,
               gpointer data)
{
  gboolean fullscreen;

  GST_OBJECT_LOCK (context->glviewport);
  fullscreen = PGM_VIEWPORT (context->glviewport)->fullscreen;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_fullscreen (context->backend, fullscreen);
  pgm_context_update (context);
}

/* Alpha blending update task handler */
static void
do_alpha_blending (PgmContext *context,
                   gpointer data)
{
  gboolean alpha_blending;

  GST_OBJECT_LOCK (context->glviewport);
  alpha_blending = PGM_VIEWPORT (context->glviewport)->alpha_blending;
  GST_OBJECT_UNLOCK (context->glviewport);

  if (alpha_blending)
    context->gl->enable (PGM_GL_BLEND);
  else
    context->gl->disable (PGM_GL_BLEND);

  pgm_context_update (context);
}

/* Resolution update task handler */
static void
do_resolution (PgmContext *context,
               gpointer data)
{
  /* FIXME */
}

/* Opacity task handler */
static void
do_opacity (PgmContext *context,
            gpointer data)
{
  gfloat opacity;

  GST_OBJECT_LOCK (context->glviewport);
  opacity = PGM_VIEWPORT (context->glviewport)->opacity * INV_255;
  GST_OBJECT_UNLOCK (context->glviewport);

  context->gl->clear_color (0.0f, 0.0f, 0.0f, opacity);
  pgm_context_update (context);
}

/* Visibility update task handler */
static void
do_visibility (PgmContext *context,
               gpointer data)
{
  static gboolean first = TRUE;
  gboolean visible;

  GST_OBJECT_LOCK (context->glviewport);
  visible = PGM_VIEWPORT (context->glviewport)->visible;
  GST_OBJECT_UNLOCK (context->glviewport);

  /* First time window is shown */
  if (G_UNLIKELY (first && visible))
    {
      /* The clear color needs an update, to avoid flashes */
      do_opacity (context, NULL);

      pgm_backend_notify_startup_complete (context->backend);
      first = FALSE;
    }

  pgm_backend_set_visibility (context->backend, visible);
}

/* Iconification update task handler */
static void
do_iconification (PgmContext *context,
                  gpointer data)
{
  gboolean iconified;

  GST_OBJECT_LOCK (context->glviewport);
  iconified = PGM_VIEWPORT (context->glviewport)->iconified;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_iconified (context->backend, iconified);
}

/* Focus update task handler */
static void
do_focus (PgmContext *context,
          gpointer data)
{
  pgm_backend_focus (context->backend);
}

/* Cursor update task handler */
static void
do_cursor (PgmContext *context,
           gpointer data)
{
  PgmViewportCursor cursor;

  GST_OBJECT_LOCK (context->glviewport);
  cursor = PGM_VIEWPORT (context->glviewport)->cursor;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_cursor (context->backend, cursor);
}

/* Icon update task handler */
static void
do_icon (PgmContext *context,
         gpointer data)
{
  GdkPixbuf *icon;

  GST_OBJECT_LOCK (context->glviewport);
  icon = PGM_VIEWPORT (context->glviewport)->icon;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_icon (context->backend, icon);
}

/* Message filter update task handler */
static void
do_message_filter (PgmContext *context,
                   gpointer data)
{
  GList *filter;

  GST_OBJECT_LOCK (context->glviewport);
  filter = PGM_VIEWPORT (context->glviewport)->message_filter;
  GST_OBJECT_UNLOCK (context->glviewport);

  pgm_backend_set_message_filter (context->backend, filter);
}

/* Drag status update task handler */
static void
do_drag_status (PgmContext *context,
                gpointer data)
{
  pgm_backend_set_drag_status (context->backend,
                               context->glviewport->drag_status);
}

/* Frame buffer read task handler */
static void
do_read_pixels (PgmContext *context,
                gpointer data)
{
  PgmGlViewportPixelRectangle *rectangle = (PgmGlViewportPixelRectangle*) data;
  PgmContextProcAddress *gl = context->gl;
  PgmViewport *viewport = PGM_VIEWPORT (context->glviewport);
  PgmCanvas *canvas = viewport->canvas;

  /* Flush the queues ensuring last rendering requests will be completed */
  pgm_gl_viewport_flush_update_queue (context->glviewport);
  flush_task_queue (context, &context->immediate_task);
  flush_task_queue (context, &context->deferred_task);

  /* If there's a canvas we need to invert the projection matrix upside-down.
   * Note that if there's no canvas bound, the pixels area will be black. */
  if (G_LIKELY (canvas))
    {
      gl->matrix_mode (PGM_GL_PROJECTION);
      gl->push_matrix ();
      gl->scale_f (1.0f, -1.0f, 1.0f);
      gl->translate_f (0.0f, -canvas->height, 0.0f);
      gl->matrix_mode (PGM_GL_MODELVIEW);
    }

  /* Draw the scene, read it from the back buffer, and clear the back buffer */
  render (context);
  gl->read_buffer (PGM_GL_BACK);
  gl->read_pixels (rectangle->x, rectangle->y, rectangle->width,
                   rectangle->height, PGM_GL_RGBA, PGM_GL_UNSIGNED_BYTE,
                   rectangle->pixels);
  gl->clear (PGM_GL_COLOR_BUFFER_BIT);
  gl->read_buffer (PGM_GL_FRONT);

  /* Pop our inverted projection matrix if needed */
  if (G_LIKELY (canvas))
    {
      gl->matrix_mode (PGM_GL_PROJECTION);
      gl->pop_matrix ();
      gl->matrix_mode (PGM_GL_MODELVIEW);
    }

  /* Push the read pixels back to the viewport */
  pgm_viewport_push_pixels (viewport, rectangle->width, rectangle->height,
                            rectangle->pixels);

  /* Then finally free the rectangle structure */
  g_slice_free (PgmGlViewportPixelRectangle, rectangle);
  rectangle = NULL;
}

/* Quit the rendering thread */
static void
do_quit (PgmContext *context,
         gpointer data)
{
  GSource *source;

  /* Remove sources. Do not use g_source_remove() since this only works for
   * sources attached to the default GMainContext. See bug #261. */
  if (context->immediate_tag)
    {
      source = g_main_context_find_source_by_id (context->render_context,
                                                 context->immediate_tag);
      if (source)
        g_source_destroy (source);
    }
  if (context->update_tag)
    {
      source = g_main_context_find_source_by_id (context->render_context,
                                                 context->update_tag);
      if (source)
        g_source_destroy (source);
    }

  /* Join the rendering thread */
  g_main_loop_quit (context->render_loop);
  g_main_loop_unref (context->render_loop);
  g_main_context_unref (context->render_context);
}

/* Texture generation task handler */
static void
do_gen_texture (PgmContext *context,
                gpointer data)
{
  pgm_texture_generate (PGM_TEXTURE (data));
}

/* Texture clean up task handler */
static void
do_clean_texture (PgmContext *context,
                  gpointer data)
{
  pgm_texture_clean (PGM_TEXTURE (data));
}

/* Texture upload task handler */
static void
do_upload_texture (PgmContext *context,
                   gpointer data)
{
  pgm_texture_upload (PGM_TEXTURE (data));
}

/* Texture update task handler */
static void
do_update_texture (PgmContext *context,
                   gpointer data)
{
  pgm_texture_update (PGM_TEXTURE (data));
}

/* Texture free task handler */
static void
do_free_texture (PgmContext *context,
                 gpointer data)
{
  pgm_texture_free (PGM_TEXTURE (data));
}

/* Traverse the layers and draw the drawables */
static void
render (PgmContext *context)
{
  PgmGlViewport *glviewport = context->glviewport;

  g_mutex_lock (glviewport->layer_lock);

  g_list_foreach (glviewport->far_layer, (GFunc) pgm_gl_drawable_draw, NULL);
  g_list_foreach (glviewport->middle_layer,
                  (GFunc) pgm_gl_drawable_draw, NULL);
  g_list_foreach (glviewport->near_layer, (GFunc) pgm_gl_drawable_draw, NULL);

  g_mutex_unlock (glviewport->layer_lock);
}

/* Push a task in a queue removing a similar one if any */
static void
push_task (PgmContext *context,
           GList **queue,
           PgmContextTask *task)
{
  GList *_queue = *queue;
  PgmContextTask *queued_task;

  PGM_CONTEXT_LOCK (context);

  /* Search and remove a similar task from the queue */
  while (_queue)
    {
      queued_task = PGM_CONTEXT_TASK (_queue->data);

      if (queued_task->type == task->type
          && queued_task->data == task->data)
        {
          GList *next = _queue->next;

          if (_queue->prev)
            _queue->prev->next = next;
          else
            *queue = next;
          if (next)
            next->prev = _queue->prev;

          pgm_context_task_free (PGM_CONTEXT_TASK (_queue->data));
          g_list_free_1 (_queue);

          _queue = NULL;
        }
      else
        _queue = _queue->next;
    }

  /* And prepend the task in the queue */
  *queue = g_list_prepend (*queue, task);

  PGM_CONTEXT_UNLOCK (context);
}

/* Removes all the tasks with the given data from a queue */
static void
remove_tasks_with_data (PgmContext *context,
                        GList **queue,
                        gconstpointer data)
{
  GList *_queue = *queue;

  PGM_CONTEXT_LOCK (context);

  /* Sequentially search for tasks with the corresponding data and remove
   * them until the end of the queue */
  while (_queue)
    {
      if (PGM_CONTEXT_TASK (_queue->data)->data != data)
        _queue = _queue->next;
      else
        {
          GList *next = _queue->next;

          if (_queue->prev)
            _queue->prev->next = next;
          else
            *queue = next;
          if (next)
            next->prev = _queue->prev;

          pgm_context_task_free (PGM_CONTEXT_TASK (_queue->data));
          g_list_free_1 (_queue);

          _queue = next;
        }
    }

  PGM_CONTEXT_UNLOCK (context);
}

/* Frees a task queue with all its remaining tasks */
static void
free_task_queue (GList **queue)
{
  GList *walk = *queue;

  while (walk)
    {
      pgm_context_task_free ((PgmContextTask *) walk->data);
      walk = walk->next;
    }

  g_list_free (*queue);
  *queue = NULL;
}

/* Flushes a task queue calling the correct handler for each tasks */
static void
flush_task_queue (PgmContext *context,
                  GList **queue)
{
  PgmContextTask *task;
  GList *reversed_queue, *walk;

  PGM_CONTEXT_LOCK (context);
  reversed_queue = g_list_reverse (*queue);
  *queue = NULL;
  PGM_CONTEXT_UNLOCK (context);

  walk = reversed_queue;
  while (walk)
    {
      task = PGM_CONTEXT_TASK (walk->data);
      context->task_func[task->type] (context, task->data);
      pgm_context_task_free (task);
      walk = walk->next;
    }

  g_list_free (reversed_queue);
  reversed_queue = NULL;
}

/* i/o watch callback for immediate tasks */
static gboolean
immediate_io_cb (GIOChannel *source,
                 GIOCondition condition,
                 gpointer data)
{
  PgmContext *context = PGM_CONTEXT (data);
  gchar buf;

  g_io_channel_read_chars (source, &buf, 1, NULL, NULL);

  /* Complete the immediate task queue */
  flush_task_queue (context, &context->immediate_task);

  return TRUE;
}

/* Update source */
static gboolean
update_cb (gpointer data)
{
  PgmContext *context = PGM_CONTEXT (data);

  /* User defined update callback */
  pgm_viewport_emit_update_pass (PGM_VIEWPORT (context->glviewport));

  /* Flush update queue */
  pgm_gl_viewport_flush_update_queue (context->glviewport);

  /* Flush rendering queues */
  flush_task_queue (context, &context->immediate_task);
  flush_task_queue (context, &context->deferred_task);

  /* Render the scene */
  render (context);
  pgm_backend_wait_for_vblank (context->backend);
  pgm_backend_swap_buffers (context->backend);
  context->gl->clear (PGM_GL_COLOR_BUFFER_BIT);

  /* Handle frame rate */
  compute_frame_rate (context);

  return TRUE;
}

/* Update timeout used to remove the update idle after 1 second of inactivity */
static gboolean
update_removal_timeout_cb (gpointer data)
{
  PgmContext *context = (PgmContext*) data;
  GTimeVal current_time;

  g_get_current_time (&current_time);

  g_mutex_lock (context->update_mutex);

  /* If the elapsed time since the previous update request rises above one
   * second, remove the auto-update source */
  if ((current_time.tv_sec - context->update_timestamp.tv_sec) >= 2)
    {
      GSource *source;

      /* Get the source from the rendering GMainContext and remove it */
      source = g_main_context_find_source_by_id (context->render_context,
                                                 context->update_tag);
      if (source)
        g_source_destroy (source);

      context->update_tag = 0;
      context->auto_updated = FALSE;
      context->fps = 0;

      g_mutex_unlock (context->update_mutex);

      GST_DEBUG ("removing update source");

      /* Remove ourself from the loop */
      return FALSE;
    }

  g_mutex_unlock (context->update_mutex);

  return TRUE;
}

/* Create a pipe and its i/o channels */
static void
create_io_channels (gint *fd,
                    GIOChannel **in,
                    GIOChannel **out)
{
  if (pipe (fd) == -1)
    {
      GST_ERROR ("cannot create the pipe");
      return;
    }

  *in = g_io_channel_unix_new (fd[1]);
  if (!*in)
    {
      GST_ERROR ("cannot create the input channel");
      return;
    }

  *out = g_io_channel_unix_new (fd[0]);
  if (!*out)
    {
      GST_ERROR ("cannot create the output channel");
      return;
    }

  /* We don't close the fd ourselves because otherwise it deadlocks on windows */
  g_io_channel_set_close_on_unref (*in, TRUE);
  g_io_channel_set_close_on_unref (*out, TRUE);
}

/* Close a pipe and unref its i/o channels */
static void
delete_io_channels (gint *fd,
                    GIOChannel **in,
                    GIOChannel **out)
{
  if (*out)
    {
      g_io_channel_unref (*out);
      *out = NULL;
    }

  if (*in)
    {
      g_io_channel_unref (*in);
      *in = NULL;
    }
}

/* Write a byte in the given i/o channel */
static inline void
write_char (GIOChannel *channel)
{
  if (G_LIKELY (channel))
    {
      g_io_channel_write_chars (channel, "#", 1, NULL, NULL);
      g_io_channel_flush (channel, NULL);
    }
}

/* Add an input watch to a specific context */
static guint
add_input_watch (GIOChannel *channel,
                 GIOFunc func,
                 gpointer user_data,
                 GMainContext *context)
{
  GSource *source;
  guint id;

  source = g_io_create_watch (channel, G_IO_IN);
  g_source_set_callback (source, (GSourceFunc) func, user_data, NULL);
  id = g_source_attach (source, context);
  g_source_unref (source);

  return id;
}

/* Rendering thread entry point, initialize and launch the rendering loop.
 * FIXME: Correct error handling */
static gpointer
render_loop (gpointer data)
{
  PgmContext *context = (PgmContext *) data;
  PgmContextProcAddress *gl = NULL;
  const gchar *env_var = NULL;
  gint i = 0;

  /* Instantiate correct backend depending on the system at compile-time */
#if defined (HAVE_GL_GLX_H)
  context->backend = PGM_BACKEND (pgm_glx_backend_new (context));
#elif defined (HAVE_AGL_AGL_H)
  context->backend = PGM_BACKEND (pgm_agl_backend_new (context));
#elif defined (HAVE_WGL_H)
  context->backend = PGM_BACKEND (pgm_wgl_backend_new (context));
#endif

  if (!pgm_backend_create_window (context->backend))
    {
      g_print ("ERROR: couldn't create the window, exiting...\n");
      goto error_handler;
    }

  context->gl = gl = &gl_proc_address;

  /* Retrieve renderer informations */
  context->version_string = (const gchar *) gl->get_string (PGM_GL_VERSION);
  context->vendor = (const gchar *) gl->get_string (PGM_GL_VENDOR);
  context->renderer = (const gchar *) gl->get_string (PGM_GL_RENDERER);
  context->extensions = (const gchar *) gl->get_string (PGM_GL_EXTENSIONS);

  /* Add some debug info */
  GST_INFO ("OpenGL vendor: %s", context->vendor);
  GST_INFO ("OpenGL renderer: %s", context->renderer);
  GST_INFO ("OpenGL version: %s", context->version_string);
  GST_DEBUG ("OpenGL extensions: %s", context->extensions);

  /* Check that OpenGL minimum version is 1.2 */
  if (!context->version_string)
    {
      g_print ("ERROR: couldn't retrieve OpenGL version, exiting...\n");
      goto error_handler;
    }
  context->version = get_float_opengl_version (context->version_string);
  if (context->version < 1.2f)
    {
      g_print ("ERROR: OpenGL version %.1f (1.2 minimum required), exiting...\n",
               context->version);
      goto error_handler;
    }

  /* Retrieve and bind supported extensions */
  while (gl_extensions_map[i].name)
    {
      if (has_opengl_extension (context->extensions, gl_extensions_map[i].name))
        context->feature_mask |= gl_extensions_map[i].feature_mask;
      i++;
    }

  /* Check in the environment if we have to disable ARB vp/fp */
  if (context->feature_mask & PGM_GL_FEAT_FRAGMENT_PROGRAM)
    {
      GST_INFO ("OpenGL fragment programs are supported");
      env_var = g_getenv ("PGM_GL_PROGRAMS");
      if (env_var && '0' == env_var[0])
        {
          GST_INFO ("Deactivating ARB vertex and fragment programs");
          context->feature_mask &= ~PGM_GL_FEAT_FRAGMENT_PROGRAM;
        }
    }
  else
    GST_INFO ("OpenGL fragment programs are not supported");

  bind_opengl_extensions (context);

  /* Maximum POT and NPOT texture size */
  gl->get_integer_v (PGM_GL_MAX_TEXTURE_SIZE, &context->max_texture_2d_size);
  GST_INFO ("OpenGL max texture 2D size: %d", context->max_texture_2d_size);
  if (context->feature_mask & PGM_GL_FEAT_TEXTURE_RECTANGLE)
    {
      gl->get_integer_v (PGM_GL_MAX_RECTANGLE_TEXTURE_SIZE,
                         &context->max_texture_rect_size);
      GST_INFO ("OpenGL max texture rect size: %d",
                context->max_texture_rect_size);
    }

  /* Pixel store default values */
  gl->get_integer_v (PGM_GL_UNPACK_ROW_LENGTH, &context->row_length);
  gl->get_integer_v (PGM_GL_UNPACK_SKIP_ROWS, &context->skip_rows);
  gl->get_integer_v (PGM_GL_UNPACK_SKIP_PIXELS, &context->skip_pixels);

  /* Multi texturing */
  if (context->feature_mask & PGM_GL_FEAT_MULTITEXTURE)
    {
      gl->get_integer_v (PGM_GL_MAX_TEXTURE_IMAGE_UNITS,
                         &context->max_texture_units);
      GST_INFO ("OpenGL multitexturing is supported, %d image units",
                context->max_texture_units);
      if (context->feature_mask & PGM_GL_FEAT_FRAGMENT_PROGRAM
          && context->max_texture_units >= 3)
        {
          GST_INFO ("OpenGL shader for color space conversions is available");
          context->feature_mask |= PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM;
        }
    }
  else
    GST_INFO ("OpenGL multitexturing is not supported");

  /* Rasterized text display lists */
  pgm_backend_build_text_lists (context->backend);

  /* Init OpenGL */
  init_opengl_states (context);
  gl->clear (PGM_GL_COLOR_BUFFER_BIT);

  /* Compile the ARB fragment programs if supported */
  if (context->feature_mask & PGM_GL_FEAT_FRAGMENT_PROGRAM)
    pgm_program_create (context);

  /* Add the immediate source to the rendering thread GMainContext */
  context->immediate_tag = add_input_watch (context->immediate_out,
                                            (GIOFunc) immediate_io_cb, context,
                                            context->render_context);

  /* Get the framerate requested by the user in the environment */
  env_var = g_getenv ("PGM_GL_FPS");
  if (env_var)
    {
      gint fps = atoi (env_var);
      if (fps > 0)
        {
          GST_INFO ("Requested FPS: %d", fps);
          context->requested_fps = 1000 / fps;
        }
    }

  /* Signal application thread that the rendering thread is now initialized */
  g_mutex_lock (context->init_mutex);
  context->initialized = TRUE;
  g_cond_signal (context->init_cond);
  g_mutex_unlock (context->init_mutex);

  /* And enter the main loop */
  g_main_loop_run (context->render_loop);

  /* Flush the queues */
  flush_task_queue (context, &context->immediate_task);
  flush_task_queue (context, &context->deferred_task);

  /* Delete the ARB fragment programs if supported */
  if (context->feature_mask & PGM_GL_FEAT_FRAGMENT_PROGRAM)
    pgm_program_delete ();

  /* Clean up rendering thread related data */
  pgm_backend_destroy_window (context->backend);
  gst_object_unref (context->backend);
  context->backend = NULL;

  return NULL;

  /* FIXME: The current Pigment architecture doesn't allow to correctly report
   *        to the plugin system that the OpenGL context can't be created. At
   *        the moment we shamefully stop the whole process... On Windows it's
   *        been decided to first display a pop-up explaining the issue. */
 error_handler:
#ifdef HAVE_WGL_H
  MessageBox (NULL,
              "This application cannot run on your system. The issue comes "
              "from either your graphical card which might not be powerful "
              "enough to support it, or from a lack of graphical driver for "
              "it to run correctly.", NULL, MB_OK | MB_ICONERROR);
#endif /* HAVE_WGL_H */
  exit (1);
}

/* Initialize context creating the rendering thread */
static gboolean
init_context (PgmContext *context)
{
  GError *error = NULL;

  /* Structure access */
  context->mutex = g_mutex_new ();

  /* Context and loop creation */
  context->render_context = g_main_context_new ();
  context->render_loop = g_main_loop_new (context->render_context, FALSE);

  /* Initialization lock */
  context->init_mutex = g_mutex_new ();
  context->init_cond = g_cond_new ();
  context->initialized = FALSE;

  /* Immediate task source */
  context->immediate_fd[0] = -1;
  context->immediate_fd[1] = -1;
  context->immediate_in = NULL;
  context->immediate_out = NULL;
  context->immediate_tag = 0;
  create_io_channels (context->immediate_fd, &context->immediate_in,
                      &context->immediate_out);

  /* Task queues */
  context->immediate_task = NULL;
  context->deferred_task = NULL;

  /* Auto-update */
  context->update_mutex = g_mutex_new ();
  context->auto_updated = FALSE;
  g_get_current_time (&context->update_timestamp);
  context->update_tag = 0;
  context->requested_fps = 0;

  /* Frame rate */
  g_get_current_time (&context->fps_tick_time);
  context->fps = 0;

  /* Create rendering thread */
  context->render_thread = g_thread_create (render_loop, context, TRUE, &error);
  if (error != NULL)
    {
      GST_ERROR ("couldn't create rendering thread: %s", error->message);
      return FALSE;
    }

  /* Various tasks function binding */
  context->task_func[PGM_CONTEXT_PROJECTION] = GST_DEBUG_FUNCPTR (do_projection);
  context->task_func[PGM_CONTEXT_SIZE] = GST_DEBUG_FUNCPTR (do_size);
  context->task_func[PGM_CONTEXT_TITLE] = GST_DEBUG_FUNCPTR (do_title);
  context->task_func[PGM_CONTEXT_DECORATION] = GST_DEBUG_FUNCPTR (do_decoration);
  context->task_func[PGM_CONTEXT_FULLSCREEN] =
    GST_DEBUG_FUNCPTR (do_fullscreen);
  context->task_func[PGM_CONTEXT_VISIBILITY] =
    GST_DEBUG_FUNCPTR (do_visibility);
  context->task_func[PGM_CONTEXT_ICONIFICATION] =
    GST_DEBUG_FUNCPTR (do_iconification);
  context->task_func[PGM_CONTEXT_FOCUS] = GST_DEBUG_FUNCPTR (do_focus);
  context->task_func[PGM_CONTEXT_ALPHA_BLENDING] =
    GST_DEBUG_FUNCPTR (do_alpha_blending);
  context->task_func[PGM_CONTEXT_RESOLUTION] =
    GST_DEBUG_FUNCPTR (do_resolution);
  context->task_func[PGM_CONTEXT_OPACITY] = GST_DEBUG_FUNCPTR (do_opacity);
  context->task_func[PGM_CONTEXT_CURSOR] = GST_DEBUG_FUNCPTR (do_cursor);
  context->task_func[PGM_CONTEXT_ICON] = GST_DEBUG_FUNCPTR (do_icon);
  context->task_func[PGM_CONTEXT_MESSAGE_FILTER] =
    GST_DEBUG_FUNCPTR (do_message_filter);
  context->task_func[PGM_CONTEXT_DRAG_STATUS] =
    GST_DEBUG_FUNCPTR (do_drag_status);
  context->task_func[PGM_CONTEXT_READ_PIXELS] =
    GST_DEBUG_FUNCPTR (do_read_pixels);

  /* Texture related tasks function binding */
  context->task_func[PGM_CONTEXT_GEN_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_gen_texture);
  context->task_func[PGM_CONTEXT_CLEAN_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_clean_texture);
  context->task_func[PGM_CONTEXT_UPLOAD_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_upload_texture);
  context->task_func[PGM_CONTEXT_UPDATE_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_update_texture);
  context->task_func[PGM_CONTEXT_FREE_TEXTURE] =
    GST_DEBUG_FUNCPTR (do_free_texture);
  context->task_func[PGM_CONTEXT_QUIT] = GST_DEBUG_FUNCPTR (do_quit);

  /* Wait for the rendering thread initialization completion */
  g_mutex_lock (context->init_mutex);
  if (!context->initialized)
    g_cond_wait (context->init_cond, context->init_mutex);
  g_mutex_unlock (context->init_mutex);

  return TRUE;
}

/* Dispose context */
static void
dispose_context (PgmContext *context)
{
  /* Join the rendering thread */
  g_thread_join (context->render_thread);

  /* Dispose immediate source data */
  delete_io_channels (context->immediate_fd, &context->immediate_in,
                      &context->immediate_out);

  /* Free the task queues */
  free_task_queue (&context->deferred_task);
  free_task_queue (&context->immediate_task);

  /* Auto-update lock */
  g_mutex_free (context->update_mutex);

  /* Initialization lock */
  g_mutex_free (context->init_mutex);
  g_cond_free (context->init_cond);
  context->initialized = FALSE;

  /* Structure access */
  g_mutex_free (context->mutex);
}

/* Public functions */

PgmContext *
pgm_context_new (PgmGlViewport *glviewport)
{
  PgmContext *context;
  gboolean initialized;

  context = g_slice_new0 (PgmContext);

  context->glviewport = glviewport;

  initialized = init_context (context);
  if (!initialized)
    {
      g_slice_free (PgmContext, context);
      context = NULL;
    }

  return context;
}

void
pgm_context_free (PgmContext *context)
{
  g_return_if_fail (context != NULL);

  dispose_context (context);
  context = NULL;

  g_slice_free (PgmContext, context);
}

PgmContextTask *
pgm_context_task_new (PgmContextTaskType type,
                      gpointer data)
{
  PgmContextTask *task;

  task = g_slice_new0 (PgmContextTask);
  task->type = type;
  task->data = data;

  return task;
}

void
pgm_context_task_free (PgmContextTask *task)
{
  g_return_if_fail (task != NULL);

  task->data = NULL;

  g_slice_free (PgmContextTask, task);
}

void
pgm_context_update (PgmContext *context)
{
  g_return_if_fail (context != NULL);

  g_mutex_lock (context->update_mutex);

  /* Get a timestamp of the last update request */
  g_get_current_time (&context->update_timestamp);

  /* Add the auto-update if the rendering loop is stopped */
  if (G_UNLIKELY (!context->auto_updated))
    {
      GSource *source;

      /* Add the update source to the rendering GMainContext. If the user has
       * not specified a framerate through the PGM_GL_FPS environment
       * variable, an idle source blocked on the vsync is attached. Otherwise,
       * a timeout source with the specified framerate is attached. */
      if (G_LIKELY (context->requested_fps == 0))
        {
          GST_DEBUG ("adding update idle source");
          source = g_idle_source_new ();
          g_source_set_priority ((GSource*) source, G_PRIORITY_DEFAULT);
        }
      else
        {
          GST_DEBUG ("adding update timeout source");
          source = g_timeout_source_new (context->requested_fps);
        }
      g_source_set_callback (source, update_cb, context, NULL);
      context->update_tag = g_source_attach (source, context->render_context);
      g_source_unref (source);

      /* Add the update removal timeout to the rendering GMainContext */
      source = g_timeout_source_new (750);
      g_source_set_callback (source, update_removal_timeout_cb, context, NULL);
      g_source_attach (source, context->render_context);
      g_source_unref (source);

      /* Mark the render loop as auto-updated */
      context->auto_updated = TRUE;
    }

  g_mutex_unlock (context->update_mutex);
}

void
pgm_context_push_immediate_task (PgmContext *context,
                                 PgmContextTask *task)
{
  g_return_if_fail (context != NULL);

  push_task (context, &context->immediate_task, task);
  write_char (context->immediate_in);
}

void
pgm_context_push_deferred_task (PgmContext *context,
                                PgmContextTask *task)
{
  g_return_if_fail (context != NULL);

  push_task (context, &context->deferred_task, task);
}

void
pgm_context_remove_tasks_with_data (PgmContext *context,
                                    gconstpointer data)
{
  g_return_if_fail (context != NULL);

  remove_tasks_with_data (context, &context->immediate_task, data);
  remove_tasks_with_data (context, &context->deferred_task, data);
}

/* FIXME */
void
pgm_context_dump_features (PgmContext *context)
{
  g_return_if_fail (context != NULL);
}
