# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
models with (in)direct database access
"""

from elisa.extern.storm_wrapper import wrapper

from storm.locals import Unicode, Int, DateTime

from elisa.core.media_uri import MediaUri, quote

class MusicAlbum(object):
    """
    Represents one album
    """
    __storm_table__ = "music_albums"
    name = Unicode(primary=True)
    cover_uri = Unicode()
    release_date = DateTime()

class Artist(object):
    """
    Represents an artist
    """
    __storm_table__ = 'artists'
    # FIXME: this class is not very precise
    name = Unicode(primary=True)
    # path to a local image
    image_uri = Unicode()


class TrackArtist(object):
    """
    weak table to map artists to tracks
    """
    __storm_table__ = 'track_artists'
    __storm_primary__ = "track_path", "artist_name"

    artist_name = Unicode()
    track_path = Unicode()

class MusicTrack(object):
    """
    Represents one music track
    """
    __storm_table__ = 'music_tracks'
    # and exactly one file
    file_path = Unicode(primary=True)
    # it has a title
    title = Unicode()
    # track number
    track_number = Int()
    # duration
    duration = Int()
    # and is exactly in one album
    album_name = Unicode()
    # genre
    genre = Unicode()

class Image(object):
    """
    represent an image
    """
    __storm_table__ = 'images'
    file_path = Unicode(primary=True)
    size = Unicode()

class Video(object):
    """
    represents an video
    """
    __storm_table__ = 'videos'
    file_path = Unicode(primary=True)
    duration = Int()
    size = Unicode()
    codec = Unicode()
    thumbnail_uri = Unicode()

class File(object):
    """
    represente a file
    """
    __storm_table__ = 'files'
    path = Unicode(primary=True)
    source = Unicode()
    mime_type = Unicode()
    modification_time = Int()
    deleted = Int()
    playcount = Int()
    last_played = Int()

class Tag(object):
    """
    representation of a TAG
    """
    __storm_table__ = "tags"
    name = Unicode(primary=True)

class FileTags(object):
    """
    map multiple tags to multiple Files
    """
    __storm_table__ = 'file_tags'
    __storm_primary__ = "file_path", "tag_name"

    tag_name = Unicode()
    file_path = Unicode()


Artist.tracks= wrapper.DeferredReferenceSet(Artist.name,
        TrackArtist.artist_name,
        TrackArtist.track_path,
        MusicTrack.file_path)
MusicTrack.album = wrapper.DeferredReference(MusicTrack.album_name,
        MusicAlbum.name)
MusicTrack.artists = wrapper.DeferredReferenceSet(MusicTrack.file_path,
        TrackArtist.track_path, TrackArtist.artist_name, Artist.name)
MusicTrack.file = wrapper.DeferredReference(MusicTrack.file_path, File.path)
MusicAlbum.tracks = wrapper.DeferredReferenceSet(MusicAlbum.name,
        MusicTrack.album_name)
File.music_track = wrapper.DeferredReference(File.path, MusicTrack.file_path)
File.image = wrapper.DeferredReference(File.path, Image.file_path)
File.tags = wrapper.DeferredReferenceSet(File.path, FileTags.file_path,
        FileTags.tag_name, Tag.name)
Tag.files = wrapper.DeferredReferenceSet(Tag.name, FileTags.tag_name,
        FileTags.file_path, File.path)
