# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.plugins.database.database_parser import DatabaseParser
from elisa.plugins.database.models import File, MusicTrack, MusicAlbum, \
        Artist, Tag, Image, Video

from elisa.core.media_uri import MediaUri

from elisa.core import common

from storm.locals import create_database
from elisa.extern.storm_wrapper import store

from elisa.plugins.database.media_scanner import SCHEMA

from twisted.trial.unittest import TestCase
from twisted.internet import defer, task
from twisted.python.failure import Failure

import datetime

class DBMixin(object):

    def setUp(self):
        self.patch_application()
        self.db = create_database('sqlite:')
        self.store = store.DeferredStore(self.db)
        dfr = self.store.start()
        dfr.addCallback(lambda x: self._create_tables())
        return dfr

    def tearDown(self):
        self.unpatch_application()
        return self.store.stop()

    def patch_application(self):
        self.application = common.application

        class Dummy(object):
            pass

        def register_component(data):
            pass

        common.application = Dummy()
        common.application.resource_manager = Dummy()
        common.application.resource_manager.register_component = register_component

    def unpatch_application(self):
        common.application = self.application

    def _execute(self, string):
        return self.store.execute(string)

    def _execute_many(self, queries):
        def go_through(queries):
            for query in queries:
                dfr = self._execute(query)
                yield dfr
        dfr = task.coiterate(go_through(queries))
        return dfr

    def _create_tables(self):
        # create the tables
        return self._execute_many(SCHEMA).addCallback(lambda x:
                self.store.commit())



class MetadataDummy(object):
    def __init__(self):
        self.added = []
        self.paused = False
    def initialize(self):
        return defer.succeed(self)
    def get_metadata(self, file):
        if self.paused:
            dfr = defer.Deferred()
        else:
            dfr = defer.succeed({})
        self.added.append( (file, dfr) )
        return dfr

class Dummy(object):pass

class TestDatabaseParser(DBMixin, TestCase):

    def setUp(self):    
        def parser_done(parser):
            self.parser = parser
            self.parser.store = self.store

            self.metadata = MetadataDummy()

            # overwrite some parts
            metadata = self.parser.metadata
            self.parser.metadata = self.metadata

            return metadata.clean()
        dfr = super(TestDatabaseParser, self).setUp()
        dfr.addCallback(lambda x: DatabaseParser.create({}))
        dfr.addCallback(parser_done)
        return dfr

    def test_modification_time_the_same(self):

        def check(res):
            self.assertEquals(self.metadata.added, [])

        def test(res):
            dummy = Dummy()
            dummy.mtime = 13
            dummy.uri = MediaUri('file:///test41')
            return self.parser.query_model(dummy, Dummy())

        file = File()
        file.path = u"/test41"
        file.modification_time = 13
        dfr = self.store.add(file)
        return dfr.addCallback(test).addCallback(check)

    def test_newer_modification_time(self):

        def check(res):
            self.assertEquals(len(self.metadata.added), 1)

        def test(res):
            dummy = Dummy()
            dummy.mtime = 19
            dummy.uri = MediaUri('file:///test42')
            source = Dummy()
            source.files_failed = []
            source.root_uri = dummy.uri
            return self.parser.query_model(dummy, source)

        file = File()
        file.path = u"/test42"
        file.modification_time = 13
        file.source = None
        dfr = self.store.add(file)
        return dfr.addCallback(test).addCallback(check)

    def test_file_not_yet_there(self):
        def check(res):
            self.assertEquals(len(self.metadata.added), 1)

        dummy = Dummy()
        dummy.mtime = 19
        dummy.uri = MediaUri('file:///test4') 

        stat = Dummy()
        stat.files_failed = []
        stat.root_uri = MediaUri('file:///test')
        return self.parser.query_model(dummy, stat).addCallback(check)

    def test_update_modification_time(self):
        file = Dummy()
        file.path = 'test'
        file.modification_time = -1
        self.parser.update_modification_time(Failure(NotImplementedError()), file)
        self.failIf(file.modification_time == -1, "Modifcation time not changed")


    def test_with_new_album_and_new_artist(self):

        def check(res):

            def got_album(album):
                self.assertEquals(album.name, 'music')
                release_date = datetime.datetime(2004, 11, 19)
                self.assertEquals(album.release_date, release_date)

            def got_artists(artists):
                self.assertEquals(artists, ['madonna'])

            def got_track(track):
                self.assertEquals(track.title, u'testie')
                self.assertEquals(track.file_path, u'test')
                self.assertEquals(track.genre, u'rock')
                self.assertEquals(track.duration, 1000)
                self.assertEquals(track.track_number, 10)
                return track.album.addCallback(got_album).addCallback(
                        lambda x: track.artists.values(Artist.name)).addCallback(got_artists)
            
            return self.store.get(MusicTrack, u'test').addCallback(got_track)

        def parse_it(res, file):
            tags = {'artist' : 'madonna',
                    'file_type' : 'audio',
                    'album' : 'music',
                    'song' : 'testie',
                    'duration' : 1000,
                    "genre" : "rock",
                    "date" : 1100818800.0,
                    'track' : 10}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check)


    def test_with_new_album_and_known_artist(self):

        def add_album(res):
            album = MusicAlbum()
            album.name = u'music'
            return self.store.add(album)

        def check(res, reference):
           
            def got_artists(artists):
                self.assertEquals(artists, ['madonna'])

            def got_album(album, reference):
                self.failUnless(album is reference)

            def got_track(track, reference):
                self.assertEquals(track.title, u'testie')
                return track.artists.values(Artist.name).addCallback(
                        got_artists).addCallback(lambda x:
                        track.album).addCallback(got_album, reference)
            
            return self.store.find(MusicTrack, MusicTrack.file_path == \
                        u'testcase').addCallback(lambda x:
                        x.one()).addCallback(got_track, reference)

        def parse_it(album, file):
            tags = {'artist' : 'madonna',
                    'file_type' : 'audio',
                    'album' : 'music',
                    'song' : 'testie',
                    'track' : 10}

            return self.parser.parse_tags(tags, file).addCallback(check,
                    album)

        file = File()
        file.path = u"testcase"
        return self.store.add(file).addCallback(add_album).addCallback(
                parse_it, file)


    def test_reuse_album_and_artist(self):

        def add_artist(album):
            artist = Artist()
            artist.name = u"madonna"
            return self.store.add(artist).addCallback(lambda x: (x,
            album))

        def add_album(res):
            album = MusicAlbum()
            album.name = u'music'
            return self.store.add(album).addCallback(add_artist)

        def check(result, ref_artist, ref_album):

            def got_album(album, ref):
                self.failUnless(album is ref, "Album not reused")
            
            def got_track(track, ref_art, ref_alb):
                self.assertEquals(track.title, u'testie')
                return track.album.addCallback(got_album,
                        ref_album).addCallback(lambda x:
                        track.artists.one())

            return self.store.find(MusicTrack, MusicTrack.file_path == \
                    u'testcase').addCallback(lambda x:
                    x.one()).addCallback(got_track, ref_artist, ref_album)

        def parse_it(refs, file):
            tags = {'artist' : 'madonna',
                    'file_type' : 'audio',
                    'album' : 'music',
                    'song' : 'testie',
                    'track' : 10}
            return self.parser.parse_tags(tags, file).addCallback(check, *refs)

        file = File()
        file.path = u"testcase"
        return self.store.add(file).addCallback(add_album).addCallback(
                parse_it, file)

    def test_reuse_fuzzy_album_and_artist(self):

        def add_artist(album):
            artist = Artist()
            artist.name = u"Madonna"
            return self.store.add(artist).addCallback(lambda x: (x,
            album))

        def add_album(res):
            album = MusicAlbum()
            album.name = u'Music'
            return self.store.add(album).addCallback(add_artist)

        def check(result, ref_artist, ref_album):

            def got_album(album, ref):
                self.failUnless(album is ref, "Fuzzy Album not reused")
            
            def got_track(track, ref_art, ref_alb):
                self.assertEquals(track.title, u'testie')
                return track.album.addCallback(got_album,
                        ref_album).addCallback(lambda x:
                        track.artists.one())

            return self.store.find(MusicTrack, MusicTrack.file_path == \
                    u'testcase').addCallback(lambda x:
                    x.one()).addCallback(got_track, ref_artist, ref_album)

        def parse_it(refs, file):
            tags = {'artist' : 'madonna',
                    'file_type' : 'audio',
                    'album' : 'music',
                    'song' : 'testie',
                    'track' : 10}
            return self.parser.parse_tags(tags, file).addCallback(check, *refs)

        file = File()
        file.path = u"testcase"
        return self.store.add(file).addCallback(add_album).addCallback(
                parse_it, file)

    test_reuse_fuzzy_album_and_artist.todo = 'Not implemented yet'

    def test_reuse_track(self):

        def add_track(file):
            track = MusicTrack()
            track.file_path = file.path
            return self.store.add(track)

        def check(res, reference):
            
            def got_track(track, reference):
                self.failUnless(track is reference, "Track was not reused")
                self.assertEquals(track.title, u'testie')

            return self.store.find(MusicTrack, MusicTrack.file_path == \
                        u'testcase').addCallback(lambda x:
                        x.one()).addCallback(got_track, reference)

        def parse_it(track, file):
            tags = {'artist' : 'madonna',
                    'file_type' : 'audio',
                    'album' : 'music',
                    'song' : 'testie',
                    'track' : 10}

            return self.parser.parse_tags(tags, file).addCallback(check, track)

        file = File()
        file.path = u"testcase"
        return self.store.add(file).addCallback(add_track).addCallback(
                parse_it, file)

    def test_set_mime_type(self):

        def check(res, file):
            self.assertEquals(file.mime_type, u'test')

        def parse_it(track, file):
            tags = {'mime_type':u'test'}

            return self.parser.parse_tags(tags, file).addCallback(check, file)

        file = File()
        file.path = u"testcase"
        return self.store.add(file).addCallback(parse_it, file)

    def test_empty(self):

        def check(res):

            def got_album(album):
                self.failIf(album)

            def got_artists(artists):
                self.assertEquals(artists, [])


            def got_track(track):
                self.assertEquals(track.title, None)
                self.assertEquals(track.genre, None)
                self.assertEquals(track.duration, None)
                return track.album.addCallback(got_album).addCallback(
                        lambda x: track.artists.values(Artist.name)
                        ).addCallback(got_artists)
            
            return self.store.get(MusicTrack, u'test').addCallback(got_track)

        def parse_it(res, file):
            tags = {'file_type' : 'audio'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check)

    def test_none_values(self):

        def check(res):

            def got_album(album):
                self.failIf(album)

            def got_artists(artists):
                self.assertEquals(artists, [])


            def got_track(track):
                self.assertEquals(track.title, None)
                self.assertEquals(track.genre, None)
                self.assertEquals(track.duration, None)
                return track.album.addCallback(got_album).addCallback(
                        lambda x: track.artists.values(Artist.name)
                        ).addCallback(got_artists)
            
            return self.store.get(MusicTrack, u'test').addCallback(got_track)

        def parse_it(res, file):
            tags = {'file_type' : 'audio', 'artist' : None,
                    'date' : None, 'album' : None, 'duration' : None,
                    'genre' : None, 'song' : None}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check)
    def test_audio_tagged(self):

        def check(res, file):
            def got_tags(tags):
                self.assertEquals(tags, ['audio'])

            return file.tags.values(Tag.name).addCallback(got_tags)

        def parse_it(res, file):
            tags = {'file_type' : 'audio'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check, file)

    def test_video_tagged(self):

        def check(res, file):
            def got_tags(tags):
                self.assertEquals(tags, [u'video'])

            return file.tags.values(Tag.name).addCallback(got_tags)

        def parse_it(res, file):
            tags = {'file_type' : 'video'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check, file)

    def test_image_tagged(self):

        def check(res, file):
            def got_tags(tags):
                self.assertEquals(tags, ['image'])

            return file.tags.values(Tag.name).addCallback(got_tags)

        def parse_it(res, file):
            tags = {'file_type' : 'image'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check, file)

    def test_retag(self):

        def check(res, file):
            def got_tags(tags):
                self.assertEquals(tags, ['audio'])

            return file.tags.values(Tag.name).addCallback(got_tags)

        def parse_it(res, file):
            tags = {'file_type' : 'audio'}
            return self.parser.parse_tags(tags, file)

        def add_tag(res, file):
            tag = Tag()
            tag.name = u'audio'
            return self.store.add(tag).addCallback(lambda x:
                    file.tags.add(tag)).addCallback(lambda x:
                    self.store.commit())

        file = File()
        file.path = u"test"
        return self.store.add(file).addCallback(add_tag, file
                ).addCallback(parse_it, file).addCallback(check, file)

    def test_image(self):

        def check(res):

            def got_image(image):
                self.failUnless(image, "Image not in DB")
            
            return self.store.get(Image, u'test').addCallback(got_image)

        def parse_it(res, file):
            tags = {'file_type' : 'image'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check)

    def test_video(self):

        def check(res):

            def got_video(video):
                self.failUnless(video, "Video not in DB")
            
            return self.store.get(Video, u'test').addCallback(got_video)

        def parse_it(res, file):
            tags = {'file_type' : 'video'}
            return self.parser.parse_tags(tags, file)

        file = File()
        file.path = u"test"
        dfr = self.store.add(file)
        return dfr.addCallback(parse_it, file).addCallback(check)

    def test_deleted(self):

        class DummySource(object):pass

        def mark_as_deleted(result, source):
            return self.parser.mark_deleted(source)

        def check_deleted(result, file):
            self.assertTrue(file.deleted, "Was not deleted")

        file = File()
        file.path = u"testcase"
        file.source = u'/tmp'

        return self.store.add(file).addCallback(mark_as_deleted, u'/tmp'
                ).addCallback(check_deleted, file)


