# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Alessandro Decina <alessandro@fluendo.com>

import os
import pkg_resources
import platform
import sys

from twisted.internet import defer, reactor
from twisted.internet.protocol import ProcessProtocol
from twisted.internet.stdio import StandardIO
from twisted.spread.jelly import globalSecurity
from twisted.spread import pb

from elisa.core.log import Loggable
from elisa.core.media_uri import MediaUri
from elisa.core.components.metadata_provider import \
        MetadataProvider, MetadataError
from elisa.core.utils.cancellable_defer import CancellableDeferred, \
        CancelledError
from elisa.plugins.gstreamer.gst_metadata import able_to_handle, \
        supported_schemes, supported_keys

class MetadataPBClientProcessProtocol(ProcessProtocol):
    def __init__(self, isClient=True, security=globalSecurity):
        self.broker = pb.Broker(isClient=isClient, security=security)
        
    def makeConnection(self, transport):
        self.broker.makeConnection(transport)
        ProcessProtocol.makeConnection(self, transport)

    def connectionMade(self):
        ProcessProtocol.connectionMade(self)
        self.broker.connectionMade()

    def outReceived(self, data):
        self.broker.dataReceived(data)
    
    def errReceived(self, data):
        self.factory.debug('Remote output:\n' + data)

    def processEnded(self, reason):
        ProcessProtocol.processEnded(self, reason)
        self.broker.connectionLost(reason)
        self.factory.processEnded(reason)

class MetadataPBClientProcessLauncher(Loggable, pb.PBClientFactory):
    max_retries = 3
    server_script = pkg_resources.resource_filename('elisa.plugins.gstreamer',
            'pb_server.py')

    log_category = 'gst_metadata_pb_client_process_launcher'

    def __init__(self, *args, **kw):
        Loggable.__init__(self)
        pb.PBClientFactory.__init__(self, *args, **kw)

        self.protocol = MetadataPBClientProcessProtocol
        self.path = os.path.split(sys.modules['elisa'].__path__[0])[0]
        self.env = dict(os.environ)
        self.env['PYTHONPATH'] = os.pathsep.join([self.path] + sys.path)
        self.executable = sys.executable
        if os.path.basename(self.executable) == 'elisa.exe':
            self.executable = os.path.join(os.path.dirname(self.executable),
                                           'deps', 'bin', 'elisa_fork.exe')
        self.args = [self.executable, '-u', self.server_script]

        self.process = None
        self.retries = 0
        self.start_defer = None
        self.stop_defer = None

    def buildProtocol(self, addr):
        protocol = pb.PBClientFactory.buildProtocol(self, addr)
        protocol.broker.factory = self

        return protocol

    def startProcess(self):
        def get_root_object_done(component):
            self.debug('initializing remote component')
            dfr = component.callRemote('initialize')
            dfr.addCallbacks(initialize_done, initialize_failure,
                callbackArgs=(component,), errbackArgs=(component,))

            return dfr

        def initialize_done(result, component):
            self.info('metadata server started')
            start_defer = self.start_defer
            self.start_defer = None
            res = start_defer.callback(component)

            return res

        def initialize_failure(failure, component):
            self.warning('failed to initialize remote component: %s', failure)
            start_defer = self.start_defer
            self.start_defer = None
            res = start_defer.errback(failure)

            return res

        self.info('starting metadata server')
        
        assert self.start_defer is None
        # start_defer will be called back after initialize() has been called on
        # the remote object
        self.start_defer = defer.Deferred()

        protocol = self.buildProtocol(None)
        self.process = reactor.spawnProcess(protocol,
                self.executable, self.args, env=self.env)
        
        if platform.system() == 'Windows':
            import win32process
            win32process.SetPriorityClass(self.process.hProcess,
                    win32process.IDLE_PRIORITY_CLASS)

        self.retries = 0
        self.debug('metadata process started')

        # get the remote object and call initialize() on it
        dfr = self.getRootObject()
        dfr.addCallback(get_root_object_done)

        return self.start_defer

    def stopProcess(self):
        assert self.stop_defer is None
        
        self.info('stopping metadata server')

        if self.process is None:
            return defer.succeed(None)

        def get_component_done(component):
            self.debug('cleaning remote component')
            return component.callRemote('clean')

        def clean_done(component):
            self.debug('clean done')
            self.process.loseConnection()

            return component

        # stop_defer will be fired in processEnded after we terminate the child
        # process
        self.stop_defer = defer.Deferred()

        # get the component and call clean() on it
        dfr = self.get_component()
        # FIXME: handle errback
        dfr.addCallback(get_component_done)
        dfr.addCallback(clean_done)

        return self.stop_defer

    def get_component(self):
        if self.process is None:
            return defer.fail(Exception('process not started'))

        if self.start_defer is not None:
            # the remote object is being started
            return self.start_defer
        else:
            return self.getRootObject()

    # FIXME: this should be called by BaseConnector 
    #def clientConnectionLost(self, connector, reason, reconnecting):

    # MetadataPBClientProcessProtocol callback
    def processEnded(self, reason):
        self.process = None

        self.log('process terminated %s', reason.getErrorMessage())

#        if reason.type == error.ProcessDone:
        # FIXME: for some reason sometime we get ProcessDone and sometime we get
        # ProcessTerminated when self.process.loseConnection() is called in
        # stopProcess(). The problem is that ProcessTerminated is also used for
        # segfault so we can't check for that here.
        if self.stop_defer is not None:
            self.info('metadata server stopped')
            stop_defer = self.stop_defer
            self.stop_defer = None
            self._failAll(reason)
            if stop_defer:
                stop_defer.callback(None)
            return

        if self.retries == self.max_retries:
            self.info('%d tries done, giving up' % self.retries)
            # FIXME: this should be called from clientConnectionLost but i don't
            # think that that's called at all with the process api...
            self._failAll(reason)
            
            return

        self.retries += 1
        self.startProcess()

class GstMetadataPBClient(MetadataProvider):
    def __init__(self):
        super(GstMetadataPBClient, self).__init__()
        self.launcher = MetadataPBClientProcessLauncher()
        self._request_id = 0

    def initialize(self):
        def start_process_done(component):
            return self

        dfr = self.launcher.startProcess()
        dfr.addCallback(start_process_done)

        return dfr

    def clean(self):
        def stop_process_done(result):
            return super(GstMetadataPBClient, self).clean()

        dfr = self.launcher.stopProcess()
        dfr.addCallback(stop_process_done)
        
        return dfr

    def get_rank(self):
        return 10

    def able_to_handle(self, metadata):
        return able_to_handle(supported_schemes,
                supported_keys, metadata)
                
    def set_process_interval(self, value):
        def got_root(root):
            return root.callRemote('set_process_interval', value)
        return self.launcher.get_component().addCallback(got_root)

    def _new_request_id(self):
        request_id = self._request_id
        self._request_id += 1

        return request_id

    def _get_metadata_done(self, remote_metadata, cancellable_dfr, metadata):
        # we get a normal dictionary as the result of a remote get_metadata
        # call but we need to return the metadata dictionary that was passed
        # as argument
        for key, value in remote_metadata.iteritems():
            if value is None:
                continue
            
            metadata[key] = value

        cancellable_dfr.callback(metadata)

        return metadata

    def _get_metadata_failure(self, failure, cancellable_dfr):
        if failure.type != 'elisa.core.utils.cancellable_defer.CancelledError':
            cancellable_dfr.errback(failure)

    def _got_root(self, root, cancellable_dfr, request_id, metadata):
        if cancellable_dfr.called:
            # the request has been cancelled
            return root

        # DictObservable can't be marshalled correctly over perspective
        # broker
        dic = dict(metadata)
        dic['request_id'] = request_id
        dfr = root.callRemote('get_metadata', dic)
        dfr.addCallback(self._get_metadata_done, cancellable_dfr, metadata)
        dfr.addErrback(self._get_metadata_failure, cancellable_dfr)

        return dfr
    
    def _call_remote_cancel(self, root, request_id):
        return root.callRemote('cancel_get_metadata', request_id)

    def _cancel_request(self, request_id):
        dfr = self.launcher.get_component()
        dfr.addCallback(self._call_remote_cancel, request_id)

    def get_metadata(self, metadata):
        request_id = self._new_request_id()
        def cancel(dfr):
            self._cancel_request(request_id)
        cancellable_dfr = CancellableDeferred(canceller=cancel)

        root_dfr = self.launcher.get_component()
        root_dfr.addCallback(self._got_root,
                cancellable_dfr, request_id, metadata)

        return cancellable_dfr

globalSecurity.allowInstancesOf(MediaUri)
