# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from twisted.trial.unittest import TestCase
from twisted.internet import reactor, defer, error
from twisted.web2.http import StatusResponse
from twisted.web2.channel.http import HTTPFactory
from twisted.web2 import iweb, server
from twisted.web2 import responsecode
from twisted.web2.stream import BufferedStream, MemoryStream

import os

from zope.interface import implements

from elisa.plugins.http_client.extern.client_http import ClientRequest

from elisa.plugins.http_client.http_client import ElisaHttpClient, \
    ElisaAdvancedHttpClient, ElisaHttpClientNotOpenedError, \
    ElisaHttpClientFactory

SERVER_HOST = 'localhost'
SERVER_PORT = 4242
SERVER_PORT_2 = 4243
SERVER_TITLE = 'Elisa test HTTP Server'

class DummyHttpServerOtherResource(object):
    implements(iweb.IResource)

    """
    This class represents a resource served by our test HTTP server.
    Our test server is able to serve other resources than just '/'.
    """

    def __init__(self, resource_name, close=False):
        # If close is True, the server will close the connection after sending
        # the response.
        self.resource_name = resource_name
        self.close = close

    def renderHTTP(self, request):
        # Render the HTTP response. The body of the response simply contains
        # the name of the resource repeated ten times.
        response = StatusResponse(responsecode.OK, self.resource_name * 10,
                                  SERVER_TITLE)

        dummy_header = request.headers.getRawHeaders('DummyTestHeader')
        if dummy_header:
            response.headers.setRawHeaders('DummyTestHeader', dummy_header)

        if self.close:
            # The connection will be closed by the server after sending the
            # response.
            response.headers.setHeader('Connection', ['close'])
        return response

class DummyHttpServerFattyResource(object):
    implements(iweb.IResource)

    """
    This class represents a large resource served by our test HTTP server.
    This resource contains a lot of content, and is used to test that the
    whole contents get correctly transferred.
    """

    def renderHTTP(self, request):
        # Render the HTTP response. The body of the response contains 10000
        # lines of text, each one prefixed by its line number.
        text = os.linesep.join([str(i + 1).zfill(5) + \
                                ': I am one dumb fatty line of data' \
                                for i in xrange(10000)])
        response = StatusResponse(responsecode.OK, text, SERVER_TITLE)
        return response

class DummyHttpServerRedirectResource(object):
    implements(iweb.IResource)

    """
    This class represents a resource served by our test HTTP server.
    This resource redirects to another resource with a given HTTP redirection
    code. For more details on HTTP redirection codes, see
    http://www.w3.org/Protocols/HTTP/HTRESP.html#z10
    """

    def __init__(self, http_code, resource_name, close=False):
        # If close is True, the server will close the connection after sending
        # the response.
        self.http_code = http_code
        self.resource_name = resource_name
        self.close = close

    def renderHTTP(self, request):
        # Render the HTTP response.
        response = StatusResponse(self.http_code, 'redirecting to /' + \
                                  self.resource_name, SERVER_TITLE)
        response.headers.setHeader('Location', '/' + self.resource_name)

        dummy_header = request.headers.getRawHeaders('DummyTestHeader')
        if dummy_header:
            response.headers.setRawHeaders('DummyTestHeader', dummy_header)

        if self.close:
            # The connection will be closed by the server after sending the
            # response.
            response.headers.setHeader('Connection', ['close'])
        return response

crashed_server = None

class DummyHttpServerOtherResourceCrash(object):
    implements(iweb.IResource)

    """
    This class represents a resource served by our test HTTP server.
    After being requested for this resource, the server will crash.
    """

    def renderHTTP(self, request):
        # Render the HTTP response
        response = StatusResponse(responsecode.OK, 'crashed', SERVER_TITLE)

        # The connection will be closed by the server after sending the
        # response.
        response.headers.setHeader('Connection', ['close'])

        # Crash the server so that it does not respond anymore
        global crashed_server
        crashed_server.http_factory.buildProtocol = lambda addr: None

        return response

class DummyHttpServerRootResource(object):
    implements(iweb.IResource)

    """
    This class represents the root resource served by our test HTTP server
    ('/'). This resource is responsible for addressing the other resources (its
    children) when they are requested.
    """

    def __init__(self):
        # Children resources for our test server
        self.my_children = {'': self,
            'foo': DummyHttpServerOtherResource('foo'),
            'bar': DummyHttpServerOtherResource('bar'),
            'fatty': DummyHttpServerFattyResource(),
            'foo_and_close': DummyHttpServerOtherResource('foo', close=True),
            'simple_redirect': DummyHttpServerRedirectResource(responsecode.MOVED_PERMANENTLY, 'foo'),
            'double_redirect': DummyHttpServerRedirectResource(responsecode.MOVED_PERMANENTLY, 'simple_redirect'),
            'redirect_and_close': DummyHttpServerRedirectResource(responsecode.MOVED_PERMANENTLY, 'foo', close=True),
            'foo_and_crash': DummyHttpServerOtherResourceCrash()}

    def locateChild(self, request, segments):
        res = self.my_children.get(segments[0])
        if res == self:
            return (self, server.StopTraversal)
        elif res == None:
            return (None, segments)
        else:
            return (res, segments[1:])

    def renderHTTP(self, request):
        response = StatusResponse(responsecode.OK,
                                  'Server\'s root resource: /', SERVER_TITLE)
        return response

class DummyHttpServerRequest(server.Request):
    """
    A site with all the resources that need to be served by our test server.
    """
    site = server.Site(DummyHttpServerRootResource())

class SetupServerMixin(object):
    """
    A test HTTP server that listens on a given port and serves our test
    resources.
    """
    def setUpClass(self):
        # Start listening
        self.http_factory = HTTPFactory(DummyHttpServerRequest)
        self._port = reactor.listenTCP(SERVER_PORT, self.http_factory)

    def tearDownClass(self):
        # Stop listening
        return defer.maybeDeferred(self._port.stopListening)


# Actual tests on the HTTP client.

class TestElisaHttpClientConnection(TestCase, SetupServerMixin):

    """
    This test case tests basic connection operations on the ElisaHttpClient
    class.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT)

    def tearDown(self):
        return self.client.close()

    def test_first_request_opens_connection(self):
        """
        Test that the client will open the connection before attempting to send
        the first request to the server.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        # First check that the connection is not opened yet
        self.failUnless(self.client._closed)

        request_dfr = self.client.request('/foo')
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_reset_retries(self):
        """
        Test that after successfully reconnecting the number of retries is
        resetted.
        """
        uri = '/foo_and_close'

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            self.assertEqual(self.client._connector.factory.retries, 0)
            return response

        request_dfrs = []
        for i in xrange(ElisaHttpClientFactory.maxRetries + 1):
            request_dfr = self.client.request(uri)
            request_dfr.addCallback(request_done)
            request_dfrs.append(request_dfr)

        return defer.DeferredList(request_dfrs)


class TestElisaHttpClientConnectionTerminate(TestCase, SetupServerMixin):

    """
    This test case tests basic connection operations on the ElisaHttpClient
    class.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT)

    def tearDown(self):
        return self.client.close()

    def test_close_before_accepted(self):
        """
        Open a connection but terminate before it is accepted.
        Test that cleanup is correctly performed.
        """
        # FIXME: cleanup is not correctly performed...
        def connection_aborted(failure):
            # UserError: User aborted connection.
            failure.trap(error.UserError)

        request_defer = self.client.request('/foo')
        request_defer.addErrback(connection_aborted)
        return defer.succeed(self)


class TestElisaHttpClientConnectionClosed(TestCase, SetupServerMixin):

    """
    This test case tests basic connection operations on the ElisaHttpClient
    class.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT)

    def test_close_not_opened(self):
        """
        Test that trying to close a connection that has not been previously
        opened fails.
        """
        close_dfr = self.client.close()
        self.failUnlessFailure(close_dfr, ElisaHttpClientNotOpenedError)
        return close_dfr


class TestElisaHttpClientConnectionError(TestCase, SetupServerMixin):

    """
    This test case tests connection errors on the ElisaHttpClient class.
    """

    def setUp(self):
        # Connecting on a port greater than 2^16-1 is not allowed
        self.client = ElisaHttpClient(SERVER_HOST, 2**16)

    def test_connection_error(self):
        """
        Test that trying to connect on an invalid port raises the expected
        exception.
        """
        request_dfr = self.client.request('/foo')
        self.assertFailure(request_dfr, error.ConnectionRefusedError)
        return request_dfr


class TestElisaHttpClientConnectionNoServer(TestCase):

    """
    This test case tests the behaviour of the ElisaHttpClient when there is no
    server to respond.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT)

    def test_connection_fails(self):
        """
        Test that trying to connect to an inexistent server raises the expected
        exception.
        """
        request_dfr = self.client.request('/foo')
        self.assertFailure(request_dfr, error.ConnectionRefusedError)
        return request_dfr


class TestElisaHttpClientNoPipelining(TestCase, SetupServerMixin):

    """
    This test case tests the ElisaHttpClient class with no pipelining.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT, pipeline=False)

    def tearDown(self):
        return self.client.close()

    def test_inexistent_resource(self):
        """
        Test that requesting a resource that does not exists returns a 404
        error code.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.NOT_FOUND)
            return response

        request_dfr = self.client.request('/crap')
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_single_request(self):
        """
        Test a single request and validate the response code.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        request_dfr = self.client.request('/foo')
        request_dfr.addCallback(request_done)
        return request_dfr

# FIXME: understand why this test hangs instead of failing with a TypeError as
#        expected.
#    def test_unicode_fails_for_request(self):
#        """
#        Test that passing a unicode string to the request fails
#        (this is a common mistake in client code)
#        """
#        request_dfr = self.client.request(u'/foo')
#        self.assertFailure(request_dfr, TypeError)
#        return request_dfr

    def test_absolute_uri(self):
        """
        Test a request with an absolute URI (not just a relative path to the
        resource on the server).
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        uri = 'http://%s:%s/foo' % (SERVER_HOST, SERVER_PORT)
        request_dfr = self.client.request(uri)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_uri_with_parameters(self):
        """
        Test a request with a URI that contains parameters.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        uri = '/foo?crap=lots&stuff=none'
        request_dfr = self.client.request(uri)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_consecutive_same_uri(self):
        """
        Test two consecutive requests on the same URI. This tests that our
        client reconnects correctly.
        """
        uri = '/foo'
        result_dfr = defer.Deferred()

        def second_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)

            # Send the second request
            second_request_dfr = self.client.request(uri)
            second_request_dfr.addCallback(second_request_done)
            second_request_dfr.chainDeferred(result_dfr)

            return response

        # Send the first request
        first_request_dfr = self.client.request(uri)
        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)

        return result_dfr

    def test_consecutive_different_uris(self):
        """
        Test two consecutive requests on different URIs. This tests that our
        client reconnects correctly.
        """
        first_uri = '/foo'
        second_uri = '/bar'
        result_dfr = defer.Deferred()

        def second_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)

            # Send the second request
            second_request_dfr = self.client.request(second_uri)
            second_request_dfr.addCallback(second_request_done)
            second_request_dfr.chainDeferred(result_dfr)

            return response

        # Send the first request
        first_request_dfr = self.client.request(first_uri)
        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)

        return result_dfr


class TestElisaHttpClientPipelining(TestCase, SetupServerMixin):

    """
    This test case tests the ElisaHttpClient class with pipelining.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT, pipeline=True)

    def tearDown(self):
        return self.client.close()

    def test_single_request(self):
        """
        Test a single request and validate the response code.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        request_dfr = self.client.request('/foo')
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_response_contents(self):
        """
        Test the actual contents of the response returned by the server.
        """
        resource = 'foo'

        def build_response_string(resource):
            content = resource * 10
            response = '<html><head><title>%s</title></head>' % SERVER_TITLE
            response += '<body><h1>%s</h1><p>%s</p></body></html>' % \
                (SERVER_TITLE, content)
            return response

        def response_read(response):
            self.assertEqual(response, build_response_string(resource))
            return response

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            read_dfr = response.stream.read()
            read_dfr.addCallback(response_read)
            return read_dfr

        request_dfr = self.client.request('/' + resource)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_large_response_contents(self):
        """
        Test that a 'large' resource is completely read.
        """
        resource = 'fatty'

        def response_read(response, response_size):
            self.assertEqual(len(response), response_size)
            return response

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            # Read the response's contents
            read_dfr = BufferedStream(response.stream).readExactly()
            read_dfr.addCallback(response_read, response.stream.length)
            return read_dfr

        request_dfr = self.client.request('/' + resource)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_consecutive_same_uri(self):
        """
        Test two consecutive requests on the same URI.
        """
        uri = '/foo'
        result_dfr = defer.Deferred()

        def second_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)

            # Send the second request
            second_request_dfr = self.client.request(uri)
            second_request_dfr.addCallback(second_request_done)
            second_request_dfr.chainDeferred(result_dfr)

            return response

        # Send the first request
        first_request_dfr = self.client.request(uri)
        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)

        return result_dfr

    def test_consecutive_different_uris(self):
        """
        Test two consecutive requests on different URIs.
        """
        first_uri = '/foo'
        second_uri = '/bar'
        result_dfr = defer.Deferred()

        def second_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)

            # Send the second request
            second_request_dfr = self.client.request(second_uri)
            second_request_dfr.addCallback(second_request_done)
            second_request_dfr.chainDeferred(result_dfr)

            return response

        # Send the first request
        first_request_dfr = self.client.request(first_uri)
        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)

        return result_dfr

    def test_consecutive_with_close(self):
        """
        Test two consecutive requests on different URIs with the server closing
        the connection after replying to the first request. This tests that
        the client correctly restores the connection if needed.
        """
        first_uri = '/foo_and_close'
        second_uri = '/bar'
        result_dfr = defer.Deferred()

        def second_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)

            # Send the second request
            second_request_dfr = self.client.request(second_uri)
            second_request_dfr.addCallback(second_request_done)
            second_request_dfr.chainDeferred(result_dfr)

            return response

        # Send the first request
        first_request_dfr = self.client.request(first_uri)
        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)

        return result_dfr

    def test_pipeline_with_close(self):
        """
        Test the pipelining with three requests. The first request returns OK,
        the second request returns OBJECT MOVED and closes the connection, and
        the third request returns OK.
        This checks that our client correctly reconnects if for some reason the
        server closes the connection in the middle of a pipeline
        """
        result_dfr = defer.Deferred()
        got_first_response = False
        got_second_response = False

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            global got_first_response
            got_first_response = True
            return response

        def second_request_done(response):
            global got_second_response
            got_second_response = True
            global got_first_response
            self.failUnless(got_first_response)
            self.assertEqual(response.code, responsecode.MOVED_PERMANENTLY)
            return response

        def third_request_done(response):
            global got_second_response
            self.failUnless(got_second_response)
            self.assertEqual(response.code, responsecode.OK)
            return response

        first_request_dfr = self.client.request('/foo')
        second_request_dfr = self.client.request('/redirect_and_close')
        third_request_dfr = self.client.request('/bar')

        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)
        second_request_dfr.addCallback(second_request_done)
        second_request_dfr.addErrback(result_dfr.errback)
        third_request_dfr.addCallback(third_request_done)
        third_request_dfr.chainDeferred(result_dfr)

        return result_dfr

    def test_cancel_request(self):
        """
        Test that cancelling a request works.
        """
        uri = '/foo'
        request_deferreds = []

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            # Let's cancel the third request after the first one is done
            request_deferreds[2].cancel()
            return request_done(response)

        def second_request_done(response):
            return request_done(response)

        def third_request_done(response):
            self.fail('This should never happen')

        def third_request_cancelled(failure):
            return failure

        def fourth_request_done(response):
            return request_done(response)

        for i in xrange(4):
            request_deferred = self.client.request(uri)
            request_deferreds.append(request_deferred)

        request_deferreds[0].addCallback(first_request_done)
        request_deferreds[1].addCallback(second_request_done)
        request_deferreds[2].addCallback(third_request_done)
        request_deferreds[2].addErrback(third_request_cancelled)
        request_deferreds[3].addCallback(fourth_request_done)

        return defer.DeferredList(request_deferreds, consumeErrors=True)

    def test_cancel_already_cancelled_request(self):
        """
        Test cancelling an already cancelled request.
        """
        uri = '/foo'
        request_deferreds = []

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        def first_request_done(response):
            # Let's cancel the fourth request after the first one is done
            request_deferreds[3].cancel()
            return request_done(response)

        def second_request_done(response):
            # Let's try to cancel again the fourth request after the second one
            # is done
            request_deferreds[3].cancel()
            return request_done(response)

        def third_request_done(response):
            return request_done(response)

        def fourth_request_done(response):
            self.fail('This should never happen')

        def fourth_request_cancelled(failure):
            return failure

        for i in xrange(4):
            request_deferred = self.client.request(uri)
            request_deferreds.append(request_deferred)

        request_deferreds[0].addCallback(first_request_done)
        request_deferreds[1].addCallback(second_request_done)
        request_deferreds[2].addCallback(third_request_done)
        request_deferreds[3].addCallback(fourth_request_done)
        request_deferreds[3].addErrback(fourth_request_cancelled)

        return defer.DeferredList(request_deferreds, consumeErrors=True)


class TestElisaHttpClientPipeliningErrors(TestCase, SetupServerMixin):

    """
    This test case tests the ElisaHttpClient class with pipelining errors.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT, pipeline=True)

    def tearDown(self):
        return self.client.close()

    def test_pipeline_with_errors(self):
        """
        Test the pipelining with three requests. For the first request we get a
        response, for the last two requests the server closes the connection
        without writing a response. This checks that the client doesn't get
        stuck retrying requests if the server is down.
        """
        result_dfr = defer.Deferred()
        got_first_response = False
        got_second_failure = False

        def first_request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            global got_first_response
            got_first_response = True
            return response

        def second_request_done(response):
            raise Exception('this should not be reached')

        def second_request_failure(failure):
            global got_second_failure
            got_second_failure = True
            global got_first_response
            self.failUnless(got_first_response)
            failure.trap(error.ConnectionDone)

        def third_request_done(response):
            raise Exception('this should not be reached')

        def third_request_failure(failure):
            global got_second_response
            self.failUnless(got_second_failure)
            failure.trap(error.ConnectionDone)

        # The server is going to be voluntarily crashed after the first request
        global crashed_server
        crashed_server = self

        first_request_dfr = self.client.request('/foo_and_crash')
        second_request_dfr = self.client.request('/foo')
        third_request_dfr = self.client.request('/bar')

        first_request_dfr.addCallback(first_request_done)
        first_request_dfr.addErrback(result_dfr.errback)
        second_request_dfr.addCallbacks(second_request_done,
                                        second_request_failure)
        third_request_dfr.addCallbacks(third_request_done,
                                       third_request_failure)
        third_request_dfr.chainDeferred(result_dfr)
        return result_dfr


class TestElisaHttpClientMethods(TestCase, SetupServerMixin):

    """
    This test case tests the ElisaHttpClient class on other methods than GET
    (POST, PUT, DELETE).

    What really happens on the server side is not tested here, this test case
    only checks that the server receives the request and answers correctly, and
    it is intended as an example piece of code for how to use methods other
    than GET.
    """

    def setUp(self):
        self.client = ElisaHttpClient(SERVER_HOST, SERVER_PORT)

    def tearDown(self):
        return self.client.close()

    def test_post(self):
        """
        Test a simple POST request.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        stream = MemoryStream('a' * 100)
        request = ClientRequest('POST', '/foo', {}, stream)
        request_dfr = self.client.request_full(request)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_put(self):
        """
        Test a simple PUT request.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        stream = MemoryStream('a' * 100)
        request = ClientRequest('PUT', '/foo', {}, stream)
        request_dfr = self.client.request_full(request)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_delete(self):
        """
        Test a simple DELETE request.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        request = ClientRequest('DELETE', '/foo', {}, None)
        request_dfr = self.client.request_full(request)
        request_dfr.addCallback(request_done)
        return request_dfr


class TestMultipleElisaHttpClients(TestCase):

    """
    This test case tests the interaction of two instances of ElisaHttpClient.
    """

    def setUpClass(self):
        # Set up two servers that listen on two different ports
        self.ports = []
        self.ports.append(reactor.listenTCP(SERVER_PORT, HTTPFactory(DummyHttpServerRequest)))
        self.ports.append(reactor.listenTCP(SERVER_PORT_2, HTTPFactory(DummyHttpServerRequest)))

    def tearDownClass(self):
        # Stop listening
        stop_defers = [defer.maybeDeferred(port.stopListening) for port in self.ports]
        return defer.DeferredList(stop_defers)

    def setUp(self):
        # Set up two clients
        self.clients = []
        self.clients.append(ElisaHttpClient(SERVER_HOST, SERVER_PORT, pipeline=True))
        self.clients.append(ElisaHttpClient(SERVER_HOST, SERVER_PORT_2, pipeline=True))

    def tearDown(self):
        close_defers = [client.close() for client in self.clients]
        return defer.DeferredList(close_defers)

    def test_two_clients_concurrent_requests(self):
        """
        Test two concurrent requests on two clients, and that closing the
        clients at tearDown time succeeds.
        The server will close the connection after responding, and this should
        not prevent the cleanup to be performed correctly.
        """
        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            return response

        first_request_defer = self.clients[0].request('/foo_and_close')
        second_request_defer = self.clients[1].request('/foo_and_close')

        first_request_defer.addCallback(request_done)
        second_request_defer.addCallback(request_done)

        return defer.DeferredList([first_request_defer, second_request_defer])

class TestElisaAdvancedHttpClient(TestCase, SetupServerMixin):

    """
    This test case tests the ElisaAdvancedHttpClient class on HTTP
    redirections and on reading large responses.
    """

    def setUp(self):
        self.client = ElisaAdvancedHttpClient(SERVER_HOST, SERVER_PORT)

    def tearDown(self):
        return self.client.close()

    def request_done(self, response):
        self.assertEqual(response.code, responsecode.OK)
        return response

    def test_no_redirection(self):
        """
        Test a simple HTTP request without redirection.
        """
        result_dfr = defer.Deferred()
        request_dfr = self.client.request('/foo_and_close')
        request_dfr.addCallback(self.request_done)
        request_dfr.addErrback(result_dfr.errback)
        request_dfr.chainDeferred(result_dfr)
        return result_dfr

    def test_simple_redirection(self):
        """
        Test a single HTTP redirection.
        """
        result_dfr = defer.Deferred()
        request_dfr = self.client.request('/simple_redirect')
        request_dfr.addCallback(self.request_done)
        request_dfr.addErrback(result_dfr.errback)
        request_dfr.chainDeferred(result_dfr)
        return result_dfr

    def test_copy_headers_on_redirection(self):
        """
        test that the headers are copied for the redirection
        """
        request = ClientRequest('GET', '/simple_redirect', None, None)

        request.headers.setRawHeaders('DummyTestHeader', ['yehaaa'])

        def check_data(result):
            self.assertEquals(result.headers.getRawHeaders('DummyTestHeader'),
                                ['yehaaa'])
            return result
        request_dfr = self.client.request_full(request)
        request_dfr.addCallback(check_data)
        request_dfr.addCallback(self.request_done)
        return request_dfr



    def test_double_redirection(self):
        """
        Test a double HTTP redirection. The first request redirects to a
        resource which in turn redirects to another one.
        """
        result_dfr = defer.Deferred()
        request_dfr = self.client.request('/double_redirect')
        request_dfr.addCallback(self.request_done)
        request_dfr.addErrback(result_dfr.errback)
        request_dfr.chainDeferred(result_dfr)
        return result_dfr

    def test_redirection_and_close(self):
        """
        Test a single HTTP redirection when the server closes the connection
        after replying to the first request.
        """
        result_dfr = defer.Deferred()
        request_dfr = self.client.request('/redirect_and_close')
        request_dfr.addCallback(self.request_done)
        request_dfr.addErrback(result_dfr.errback)
        request_dfr.chainDeferred(result_dfr)
        return result_dfr

    def test_response_contents(self):
        """
        Test the actual contents of the response returned by the server.
        """
        resource = 'foo'

        def build_response_string(resource):
            content = resource * 10
            response = '<html><head><title>%s</title></head>' % SERVER_TITLE
            response += '<body><h1>%s</h1><p>%s</p></body></html>' % \
                (SERVER_TITLE, content)
            return response

        def response_read(response):
            self.assertEqual(response, build_response_string(resource))
            return response

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            # Read the contents of the response
            read_dfr = response.stream.read()
            read_dfr.addCallback(response_read)
            return read_dfr

        request_dfr = self.client.request('/' + resource)
        request_dfr.addCallback(request_done)
        return request_dfr

    def test_large_response_contents(self):
        """
        Test that a 'large' resource is completely read.
        """
        def response_read(response, response_size):
            self.assertEqual(response_size, len(response))
            return response

        def request_done(response):
            self.assertEqual(response.code, responsecode.OK)
            # Read the contents of the response
            read_dfr = BufferedStream(response.stream).readExactly()
            read_dfr.addCallback(response_read, response.stream.length)
            return read_dfr

        request_dfr = self.client.request('/fatty')
        request_dfr.addCallback(request_done)
        return request_dfr
