from twisted.trial.unittest import TestCase
from twisted.internet import defer
from twisted.python import reflect

from elisa.core.config import Config
from elisa.plugins.pigment.pigment_frontend import PigmentFrontendExtensionMixin
from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.core.utils.misc import pkg_resources_copy_dir 

import pkg_resources
import os.path

class TestController(PigmentController):
    decorated = 0

class GenericAlbumController(TestController):
    pass

class SpecificAlbumController(TestController):
    pass

class AlternateGenericAlbumController(TestController):
    pass

def decorator(controller):
    controller.decorated += 1

    return defer.succeed(None)

def failing_decorator(controller):
    raise Exception('woot')

def failing_decorator_two(controller):
    return defer.fail(Exception('woot'))

class TestPigmentFrontendExtensionMixin(TestCase):
    def setUp(self):
        self.frontend = PigmentFrontendExtensionMixin()

    def test_create(self):
        """
        Register and create two controllers. Test that two patterns don't
        interfere with each other.
        """
        self.frontend.add_controller('/elisa/album/.*',
                GenericAlbumController)
        self.frontend.add_controller('/elisa/album/Acid Food',
                SpecificAlbumController)

        def first_controller_created_callback(controller):
            self.failUnless(isinstance(controller, GenericAlbumController))
            self.failUnlessEqual(controller.path, '/elisa/album/album 1')
            
            return self.frontend.create_controller('/elisa/album/Acid Food',
                    Config())

        def second_controller_created_callback(controller):
            self.failUnless(isinstance(controller, SpecificAlbumController))
            self.failUnlessEqual(controller.path, '/elisa/album/Acid Food')

            return controller

        dfr = self.frontend.create_controller('/elisa/album/album 1',
                Config())
        dfr.addCallback(first_controller_created_callback)
        dfr.addCallback(second_controller_created_callback)

        return dfr

    def test_create_overridden(self):
        """
        Register two controllers for the same path_pattern. Test that the second
        controller overrides the first.
        """
        def first_controller_created_callback(controller):
            self.failUnless(isinstance(controller, GenericAlbumController))
            self.failUnlessEqual(controller.path, '/elisa/album/album 1')

            self.frontend.add_controller('/elisa/album/.*',
                    AlternateGenericAlbumController)
            return self.frontend.create_controller('/elisa/album/album 1',
                    Config())

        def second_controller_created_callback(controller):
            self.failUnless(isinstance(controller,
                    AlternateGenericAlbumController))
            self.failUnlessEqual(controller.path, '/elisa/album/album 1')

            return controller
        
        self.frontend.add_controller('/elisa/album/.*',
                GenericAlbumController)
        dfr = self.frontend.create_controller('/elisa/album/album 1',
                Config())
        dfr.addCallback(first_controller_created_callback)

        return dfr

    def test_decorate(self):
        """
        Register a controller decorator to decorate a controller.
        """
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)
        self.frontend.add_decorator('/elisa/album/.*', decorator)

        def create_controller_callback(controller):
            self.failUnlessEqual(controller.decorated, 1)

        dfr = self.frontend.create_controller('/elisa/album/', Config())
        dfr.addCallback(create_controller_callback)

        return dfr

    def test_decorate_path(self):
        """
        Register a controller decorator using a path string instead of a
        callable.
        """
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)
        path = 'elisa.plugins.pigment.tests.test_pigment_frontend:decorator'
        self.frontend.add_decorator('/elisa/album/.*', path)

        def create_controller_callback(controller):
            self.failUnlessEqual(controller.decorated, 1)

        dfr = self.frontend.create_controller('/elisa/album/', Config())
        dfr.addCallback(create_controller_callback)

        return dfr
    
    def test_decorate_unexisting_path(self):
        """
        Register a controller decorator using a path string instead of a
        callable.
        """
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)
        path = 'elisa.plugins.pigment.tests.test_pigment_frontend:i_am_not_here'
        self.frontend.add_decorator('/elisa/album/.*', path)

        def create_controller_callback(controller):
            self.failUnlessEqual(controller.decorated, 0)

        dfr = self.frontend.create_controller('/elisa/album/', Config())
        dfr.addCallback(create_controller_callback)

        return dfr

    def test_decorate_multiple(self):
        """
        Test multiple decorators.
        """
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)
        self.frontend.add_decorator('/elisa/album/.*', decorator)
        self.frontend.add_decorator('/elisa/album/.*', decorator)
        self.frontend.add_decorator('/elisa/album/.*', decorator)

        def create_controller_callback(controller):
            self.failUnlessEqual(controller.decorated, 3)

        dfr = self.frontend.create_controller('/elisa/album/', Config())
        dfr.addCallback(create_controller_callback)

        return dfr
    
    def test_decorate_multiple_with_errors(self):
        """
        Test multiple decorators.
        """
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)
        self.frontend.add_decorator('/elisa/album/.*', decorator)
        self.frontend.add_decorator('/elisa/album/.*', failing_decorator)
        self.frontend.add_decorator('/elisa/album/.*', decorator)
        path = 'elisa.plugins.pigment.tests.test_pigment_frontend:i_am_not_here'
        self.frontend.add_decorator('/elisa/album/.*', path)
        self.frontend.add_decorator('/elisa/album/.*', failing_decorator_two)
        self.frontend.add_decorator('/elisa/album/.*', decorator)

        def create_controller_callback(controller):
            self.failUnlessEqual(controller.decorated, 3)

        dfr = self.frontend.create_controller('/elisa/album/', Config())
        dfr.addCallback(create_controller_callback)

        return dfr

    def test_retrieve_single_controller(self):
        """
        Test retrieval of an existing instance of a controller.
        """
        path = '/elisa/album/album 1'
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)

        def controller_created_callback(controller):
            saved_controllers = self.frontend.retrieve_controllers(path)
            self.failUnless(controller in saved_controllers)

        dfr = self.frontend.create_controller(path, Config())
        dfr.addCallback(controller_created_callback)

        return dfr

    def test_retrieve_multiple_controllers(self):
        """
        Test retrieval of multiple controller instances that correspond to
        paths with a common part.
        """
        self.frontend.add_controller('/elisa/album/.*',
                GenericAlbumController)
        self.frontend.add_controller('/elisa/album/Acid Food',
                SpecificAlbumController)

        self.first_controller = None
        self.second_controller = None

        def first_controller_created_callback(controller):
            self.first_controller = controller
            return self.frontend.create_controller('/elisa/album/Acid Food',
                    Config())

        def second_controller_created_callback(controller):
            self.second_controller = controller

            path = '/elisa/album/album 1'
            saved_controllers = self.frontend.retrieve_controllers(path)
            self.failUnless(self.first_controller in saved_controllers)
            self.failUnless(self.second_controller not in saved_controllers)

            path = '/elisa/album/Acid Food'
            saved_controllers = self.frontend.retrieve_controllers(path)
            self.failUnless(self.second_controller in saved_controllers)
            self.failUnless(self.first_controller not in saved_controllers)

        dfr = self.frontend.create_controller('/elisa/album/album 1',
                Config())
        dfr.addCallback(first_controller_created_callback)
        dfr.addCallback(second_controller_created_callback)

        return dfr

    def test_retrieve_dropped_controllers(self):
        """
        Test that the frontend releases controllers that are not used anymore.
        """
        path = '/elisa/album/album 1'
        self.frontend.add_controller('/elisa/album/.*', GenericAlbumController)

        def controller_created_callback(controller):
            pass

        def controller_dropped_callback(result):
            saved_controllers = self.frontend.retrieve_controllers(path)
            self.failUnlessEqual(len(saved_controllers), 0)

        dfr = self.frontend.create_controller(path, Config())
        dfr.addCallback(controller_created_callback)
        dfr.addCallback(controller_dropped_callback)

        return dfr


class PigmentFrontendExtensionMixinStub(PigmentFrontendExtensionMixin):

    def __init__(self):
        self.added_controllers = []
        self.added_decorators = []
        super(PigmentFrontendExtensionMixinStub, self).__init__()

    def add_controller(self, path_pattern, controller):
        self.added_controllers.append((path_pattern, controller))

    def add_decorator(self, path_pattern, decorator):
        self.added_decorators.append((path_pattern, decorator))

class TestPigmentFrontendExtensionMixinMappings(TestCase):
    def setUp(self):
        # do everything under _trial_temp/test_pigment_plugin
        self.test_dir = os.path.abspath('test_pigment_plugin')
        pkg_resources.ensure_directory(self.test_dir)

        self.copy_plugin('tests/data/plugins/test_simple')
        self.load_plugins()

        self.frontend = PigmentFrontendExtensionMixinStub()

    def copy_plugin(self, path):
        # copy the plugin to _trial_temp/test_pigment_plugin
        pkg_resources_copy_dir('elisa.plugins.pigment', path, self.test_dir)

    def load_plugins(self):
        # look up the plugin directory
        env = pkg_resources.Environment([self.test_dir])
        distributions, errors = pkg_resources.working_set.find_plugins(env)

        # add the distributions to the working set
        for dist in distributions:
            pkg_resources.working_set.add(dist)

    def test_loading_controller_mappings(self):
        """
        Test loading controller mappings from a fake plugin.
        """
        controller_mappings = set([("test_controller_key1", "test_controller_value1"),
                                   ("test_controller_key2", "test_controller_value2")])

        loaded = set(self.frontend.added_controllers)
        self.failUnless(controller_mappings.issubset(loaded))

    def test_loading_decorator_mappings(self):
        """
        Test loading decorator mappings from a fake plugin.
        """
        decorator_mappings = set([("test_decorator_key1", "test_decorator_value1"),
                                  ("test_decorator_key2", "test_decorator_value2")])

        loaded = set(self.frontend.added_decorators)
        self.failUnless(decorator_mappings.issubset(loaded))

    def test_initialize_theme(self):
        self.fail()

    def test_get_theme(self):
        self.fail()

    def test_set_theme(self):
        self.fail()

    def test_load_from_theme_theme(self):
        self.fail()
