# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.plugins.pigment.graph.group import Group

from elisa.plugins.pigment.widgets.style import Style
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.widgets.const import *

import pgm
import gobject

import copy
import logging


class Widget(Group):
    """
    Basic widget for the python Pigment scenegraph.

    It handles a simple focus system (at most one widget having the focus at
    one time) and a simple style system (with style and style-properties change
    notifications, also bound to state changes).

    Emit these signals:
      - focus
      - state-changed
      - style-set
      - key-press-event

    @ivar name: the name of the widget
    @type name: string
    @ivar style: the present style of the widget
    @type style: L{pgm.widget.Style}
    @ivar state: the present state of the widget
    @type state: enum(STATE_NORMAL, STATE_ACTIVE, STATE_PRELIGHT,
                      STATE_SELECTED, STATE_INSENSITIVE)
    """

    __gsignals__ = {'focus': (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_BOOLEAN,)),
                    'state-changed': (gobject.SIGNAL_RUN_LAST,
                                      gobject.TYPE_BOOLEAN,
                                      (gobject.TYPE_PYOBJECT,)),
                    'style-set': (gobject.SIGNAL_RUN_LAST,
                                  gobject.TYPE_BOOLEAN,
                                  (gobject.TYPE_PYOBJECT,)),
                    'key-press-event': (gobject.SIGNAL_RUN_LAST,
                                        gobject.TYPE_BOOLEAN,
                                        (gobject.TYPE_PYOBJECT,
                                         gobject.TYPE_PYOBJECT,
                                         gobject.TYPE_PYOBJECT))}

    def __init__(self):
        """
        Set the default styles for the widget. Subclasses should override and
        call "self._update_style_properties()" after calling this
        base class constructor.
        """

        super(Widget, self).__init__()

        self.name = None

        self._initializing = True

        self._state = None
        self._styles = {}
        self._focus = False

        self._init_styles()

        self.state = STATE_NORMAL

        self._initializing = False

    def _init_styles(self):
        theme = Theme.get_default()

        styles = {}
        for state in (STATE_NORMAL, STATE_ACTIVE, STATE_PRELIGHT,
                      STATE_SELECTED, STATE_INSENSITIVE):
            styles[state] = []

        base = self.__class__
        while issubclass(base, Widget):
            module_name = base.__module__
            # HACK, to allow interactive testing of some widgets
            if module_name == '__main__':
                module_name = 'elisa.plugins.pigment.widgets'
            else:
                module_name = '.'.join(module_name.split('.')[0:-1])

            widget_class = module_name + '.' + base.__name__
            for state in (STATE_NORMAL,):# STATE_ACTIVE, STATE_PRELIGHT,
                          #STATE_SELECTED, STATE_INSENSITIVE):
                styles[state].insert(0, theme.get_style_for_widget(widget_class, state))
            base = base.__base__

        logging.debug('STATE_NORMAL widget style:' + str(styles[STATE_NORMAL]))

        normal_style = Style()
        for s in styles[STATE_NORMAL]:
            if s is not None:
                normal_style = normal_style.merge(s)
        self._styles[STATE_NORMAL] = normal_style

        """
        for state in (STATE_ACTIVE, STATE_PRELIGHT,
                      STATE_SELECTED, STATE_INSENSITIVE):
            self._styles[state] = Style(**normal_style.get_items())
            for s in styles[state]:
                if s is not None:
                    self._styles[state].update(s)
        """

    def state__get(self):
        """The present state of the widget"""
        return self._state

    def state__set(self, state):
        """
        Set the state of the widget and emit the "state-changed" signal if
        necessary.

        @param state: the new state
        @type state: enum(STATE_NORMAL, STATE_ACTIVE, STATE_PRELIGHT,
                          STATE_SELECTED, STATE_INSENSITIVE)
        """
        if self._state != state:
            logging.debug("State changed from %s to %s" % (self._state, state))
            self._state = state
            self.emit("state-changed", state)

    state = property(state__get, state__set)

    def do_state_changed(self, previous_state):
        """Default 'state-changed' handler"""
        try:
            self.style = self._styles[self.state]
        except KeyError:
            logging.warning("Cannont change the status of widget '%s' to %s" % (self, self.state))

    def style__get(self):
        """The present style"""
        return self._style

    def style__set(self, style):
        """
        Set the present style, after binding it to the widget and subscribing
        for property change notifications.

        @param style: the style to set
        @type param: L{pgm.widget.Style}
        """
        logging.debug("Style changed")
        style.connect('property-changed', self._style_property_changed)
        style.widget = self
        self._style = style
        self.emit("style-set", style)

    style = property(style__get, style__set)

    def _style_property_changed(self, style, key, value):
        if not self._initializing:
            self._update_style_properties({key : value})

    def do_style_set(self, style):
        """Default 'style-set' signal handler"""
        if not self._initializing:
            self._update_style_properties(self.style.get_items())

    def _update_style_properties(self, props):
        """
        Update the widget's appearence basing on the properties set.

        This is intended to be overriden by subclasses, that will implement the
        actual code to bind style properties to widget ones.

        @param props: the properties that have to be updated
        @type props: dictionary of strings ==> anything
        """
        if props is None:
            return

        for key, value in props.iteritems():
            if key != 'widget' and not key.startswith('_'):
                logging.debug("Fake updating '%s' to %s" % (key, value))

    def get_parent(self):
        """
        Get the parent, if exists.

        @return: L{elisa.plugins.pigment.widgets.Widget}
        """
        return self.parent

    def get_children(self):
        """
        Get the list of direct children.

        @return: list of L{elisa.plugins.pigment.widgets.Widget}
        """
        return filter (lambda c: isinstance(c, Widget), self._children)

    def get_root(self):
        """
        Get the root of the widget's tree hierarchy

        @return: L{elisa.plugins.pigment.widgets.Widget}
        """
        while self.parent is not None:
            self = self.parent

        return self

    def get_descendants(self):
        """
        Get the list of nodes in the subtree

        @return: list of L{elisa.plugins.pigment.widgets.Widget}
        """
        children = filter (lambda c: isinstance(c, Widget), self._children)
        items = copy.copy(children)

        for child in children:
            items += child.get_descendants()

        return items

    def focus__get(self):
        """Whether the widget has focus"""
        return self._focus

    def focus__set(self, value):
        """
        Set or unset the focus on the widget and emit the 'focus' signal.

        If setting the focus on the current widget, walk throught the widget
        hierarchy it belongs to and unset the previously focused one.

        @param value: the focus value.
        @type value: boolean.
        """

        #if self._focus == value:
        #    return

        if not value:
            self._focus = value
            self.emit('focus', value)
            return

        for widget in self.get_root().get_descendants():
            if widget != self:
                widget.focus = False

        self._focus = value
        self.emit('focus', value)

    focus = property(focus__get, focus__set)

    def focus_child__get(self):
        """Get the child with focus"""
        for child in self.get_descendants():
            if child.focus:
                return child

    focus_child = property(focus_child__get)

    def do_key_press_event(self, viewport, event, widget):
        """
        Default 'key-press-event' signal handler.

        Forward the signal to the focused child, if any.
        """
        focus_child = self.focus_child
        if focus_child:
            focus_child.emit("key-press-event", viewport, event, widget)

        logging.debug("Key pressed")

    def _do_real_key_press_event(self, viewport, event, widget):
        self.emit("key-press-event", viewport, event, widget)

    @classmethod
    def _demo_create_viewport(cls, plugin):
        viewport = pgm.viewport_factory_make('opengl')
        viewport.title = cls.__name__
        viewport.show()
        return viewport

    @classmethod
    def _on_demo_delete(self, viewport, event):
        try:
            __IPYTHON__
            sys.exit(0)
        except NameError:
            pgm.main_quit()

    @classmethod
    def _demo_widget(cls):
        """
        Meant to be overidden by inheriting widgets for widget creation and
        setup at demo time.

        @return: L{elisa.plugins.pigment.widgets.Widget}
        """
        widget = cls()
        return widget

    @classmethod
    def _set_demo_widget_defaults(cls, widget, canvas, viewport):
        widget.canvas = canvas
        # FIXME: connection to the viewport key presses should be handled in
        # the Widget class so that the key-press-event actually works at all
        # times not only at demo time.
        # The key-press-event should be emitted for the widgets having the
        # focus
        viewport.connect('key-press-event', widget._do_real_key_press_event,
                         widget)

        def handle_basic_keys (viewport, event, widget):
            if event.keyval == pgm.keysyms.Escape:
                viewport.emit('delete-event', None)
            elif event.keyval == pgm.keysyms.f:
                viewport.set_fullscreen(not viewport.get_fullscreen())

        viewport.connect('key-press-event', handle_basic_keys, widget)

        viewport.connect('delete-event', cls._on_demo_delete)

    @classmethod
    def demo(cls, plugin='opengl'):
        """
        Create a demo widget, put it on a canvas and show it in a viewport.

        Just start a pgm.main() or an "ipython -gthread" shell to interactively
        test your widget. See the __main__ block at the end of this file.

        @return: the demo L{elisa.plugins.pigment.widgets.Widget}
        """

        # viewport creation
        # keep a reference to the viewport in the global namespace so that it
        # does not get garbage collected
        global viewport
        viewport = cls._demo_create_viewport(plugin)

        # create and bind the canvas to the viewport
        canvas = pgm.Canvas()
        viewport.set_canvas(canvas)

        # create the widget for the demo and set it up
        widget = cls._demo_widget()
        widget.visible = True
        cls._set_demo_widget_defaults(widget, canvas, viewport)

        return widget


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    widget = Widget.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()

