# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.media_uri import MediaUri
from elisa.plugins.poblesec.link import Link

# FIXME: we should factor out useful stuff, like some widgets
from elisa.plugins.poblesec.filesystem_controller import ImageWithReflection
from elisa.plugins.poblesec.filesystem_controller import GridItem

from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.poblesec.section import MenuItemWidget, DoubleLineMenuItemWidget
from elisa.plugins.poblesec.list_preview import PreviewListController
from elisa.plugins.poblesec.list_controller import ListController
from elisa.plugins.poblesec.coverflow import CoverflowListController, ListCoverflow
from elisa.plugins.poblesec.grid import GridController
from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.fast_scroll_controller import FastScrollListController

from elisa.core import common
from elisa.core.application import PICTURES_CACHE

from elisa.core.utils.cancellable_defer import CancelledError

from elisa.core.resource_manager import NoMatchingResourceProvider, ResourceProviderNotFound
from elisa.plugins.base.models.audio import AlbumModel

# FIXME: cover retrieval is hardcoded for amazon covers
try:
    from elisa.plugins.amazon.metadata import AmazonCovers
except ImportError:
    AmazonCovers = None

from twisted.internet import defer

import os, hashlib

class ArtistPictureRetriever(object):
    def __init__(self):
        self.enabled = True
        self.cache = {}

    def get_image(self, model):
        if not self.enabled:
            # we are disabled
            return

        model_id = model.name.replace(" ","+")

        if model_id in self.cache:
            if self.cache[model_id]:
                # we did a request and we have it still in cache
                model.image_uri = self.cache[model_id]
                return defer.succeed(model)
            # we are in doing the request
            return

        self.cache[model_id] = None
             
        def got_thumbnail(result, model, model_id):
            if result.images:
                res = unicode(result.images[0].references[0])
                model.image_uri = res
                self.cache[model_id] = res
                return model

        def failed(result, model_id):
            # if it was cancelled, we want to remove the model_id to try again
            # later
            if result.type is CancelledError:
                self.cache.pop(model_id)
            # we swallow anything
            return None

        uri = MediaUri("http://www.discogs.com/artist/%s" %
                model.name.replace(" ", "+"))
        try:
            data_model, dfr = common.application.resource_manager.get(uri)
        except NoMatchingResourceProvider:
            self.enabled = False
            return

        dfr.addCallback(got_thumbnail, model, model_id)
        dfr.addErrback(failed, model_id)
        return dfr

# FIXME: a global instance at this level does not prevent from having multiple
# instances, this should be made a singleton or an attribute of the
# application, or something...
artist_picture_retriever = ArtistPictureRetriever()

class AlbumCoverRetrieval(object):
    def __init__(self):
        self.cache = {}
        self.provider = None

        if not AmazonCovers:
            # plugin missing
            return

        def set_provider(provider):
            self.provider = provider
        try:
            dfr = AmazonCovers.create({})
            dfr.addCallback(set_provider)
        except ResourceProviderNotFound:
            # provider is disable, do nothing
            pass

    def get_image(self, model, artist=None):
        if not self.provider:
            # we are disabled
            return

        model_id = unicode(model.name)

        if model_id in self.cache:
            if self.cache[model_id]:
                # we did a request and we have it still in cache
                model.cover_uri = self.cache[model_id]
                return defer.succeed(model)
            # we are in doing the request
            return
        
        self.cache[model_id] = None

        album_model = AlbumModel()
        album_model.artist = artist
        album_model.album = model_id.encode('UTF-8')
        
        def got_album(result_model, return_model, model_id):
            if result_model.cover:
                uri = unicode(result_model.cover.references[-1])
                self.cache[model_id] = uri
                return_model.cover_uri = uri
                return return_model

        def failed(result, model_id):
            # if it was cancelled, we want to remove the model_id to try again
            # later
            if result.type is CancelledError:
                self.cache.pop(model_id)
            # we swallow anything
            return None


        dfr = self.provider.get_metadata(album_model)
        dfr.addCallback(got_album, model, model_id)
        dfr.addErrback(failed, model_id)
        return dfr

# FIXME: a global instance at this level does not prevent from having multiple
# instances, this should be made a singleton or an attribute of the
# application, or something...
album_cover_retriever = AlbumCoverRetrieval()

class RetrievalMixin(object):

    resource = ''
    retriever = None
    def image_from_item(self, item):
 
        thumb = self._get_image_path_for_item(item)
        if thumb:
            return thumb

        theme = self.frontend.get_theme()
        return theme.get_resource(self.resource)

    def _update_item(self, result, item):
        if not result: 
            # nothing found
            return
        
        path = self._get_path_for_item(item)

        if path:
            thumb = _thumbnail_file(path)
            if os.path.exists(thumb):
                index = self.model.index(item)
                self.model.notifier.emit('items-changed', index, [item])
                return

            # no thumbnail yet, so lets download it
            dfr = _get_thumbnail(MediaUri(path))
            dfr.addCallback(self._update_item, item)

    def node_renderer(self, item, widget):
        # cancel previously made typefinding and thumbnailing request relative
        # to widget
        if hasattr(widget, "dfr") and not widget.dfr.called:
            widget.dfr.cancel()
            delattr(widget, "dfr")

        thumb = self._get_image_path_for_item(item)
        if not thumb:
            self._retrieve(widget, item)

        return super(RetrievalMixin, self).node_renderer(item, widget)

    def _retrieve(self, widget, item):
        if self.retriever:
            # we don't have our image yet, so let's do the necessary calls
            dfr = self.retriever.get_image(item)
            if dfr:
                dfr.addCallback(self._update_item, item)
                widget.dfr = dfr
        
    def _get_path_for_item(self, item):
        raise NotImplementedError()

    def _get_image_path_for_item(self, item):
        path = self._get_path_for_item(item)

        if path:
            thumb = _thumbnail_file(path)
            if os.path.exists(thumb):
                return thumb

            # no thumbnail yet, so lets download it
            dfr = _get_thumbnail(MediaUri(path))
            dfr.addCallback(self._update_item, item)

class AlbumsController(RetrievalMixin, ListController):

    resource = 'elisa.plugins.poblesec.cover'
    retriever = album_cover_retriever
    tracks_controller_path = ''

    def initialize(self, artist=None):
        dfr = super(AlbumsController, self).initialize()
        dfr.addCallback(self._start, artist)
        return dfr

    def _get_path_for_item(self, item):
        return item.cover_uri

    def _get_title(self, item):
        return item.name

    def _start(self, result, artist):
        raise NotImplementedError()

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller(self.tracks_controller_path,
                                                    model.name, album=model)


class ArtistsController(RetrievalMixin, ListController):

    resource = 'elisa.plugins.poblesec.artist'
    retriever = artist_picture_retriever
    albums_controller_path = ''

    def initialize(self, uri=None):
        self.uri = uri
        dfr = super(ArtistsController, self).initialize()
        dfr.addCallback(self._start)
        return dfr

    def _get_title(self, item):
        return item.name

    def _get_path_for_item(self, item):
        return item.image_uri

    def _start(self, result):
        raise NotImplementedError()

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller(self.albums_controller_path,
                                                    model.name, artist=model)


class TracksController(RetrievalMixin, ListController):

    resource = 'elisa.plugins.poblesec.cover'

    def initialize(self, album=None):
        dfr = super(TracksController, self).initialize()
        dfr.addCallback(self._start, album)
        return dfr

    def _get_title(self, item):
        title = "%s" % item.title

        if item.track_number:
            num = str(item.track_number).zfill(2)
            title = "%s. %s" % (num, title)
        # FIXME: currently disabled because it is ugly :(
        #if item.duration:
        #    title = "%s (%d)" % (title, item.duration)

        return title

    def _get_path_for_item(self, item):
        return None

    def _start(self, result, album):
        raise NotImplementedError()

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        # TODO: videos & pictures playback support
        if selected_item != self._previous_clicked:
            self._previous_clicked = selected_item
            self.play_audio(model)

    def play_audio(self, model):
        raise NotImplementedError()


def _thumbnail_file(thumbnail_uri):
    if not thumbnail_uri:
        return None
    if not os.path.exists(PICTURES_CACHE):
        os.makedirs(PICTURES_CACHE, 0755)
    thumbnail = hashlib.md5(str(thumbnail_uri)).hexdigest() + '.jpg'
    return os.path.join(PICTURES_CACHE, thumbnail)


def _get_thumbnail(thumbnail_uri):
    # Load the photo thumbnail
    thumbnail_file = _thumbnail_file(thumbnail_uri)
    if os.path.exists(thumbnail_file):
        return defer.succeed(thumbnail_file)
    else:
        # Download the thumbnail first
        def got_thumbnail(data_model):
            f = file(thumbnail_file, 'wb')
            f.write(data_model.data)
            f.close()
            return thumbnail_file

        try:
            data_model, dfr = common.application.resource_manager.get(thumbnail_uri)
        except NoMatchingResourceProvider, e:
            # TODO: investigate why this seems to happen from time to time
            return defer.fail(e)
        dfr.addCallback(got_thumbnail)
        return dfr


# general implementations

class VerticalWithPreview(PreviewListController, FastScrollListController):

    node_widget = MenuItemWidget
    shortcuts = list("#abcdefghijklmnopqrstuvwxyz")

    def node_renderer(self, item, widget):
        widget.icon.clear()
        widget.label.label = self._get_title(item)

        thumb = self._get_image_path_for_item(item)
        if thumb:
            widget.icon.set_from_file(thumb)
        else:
            self.frontend.load_from_theme(self.resource, widget.icon)

    def item_to_letter(self, item):
        letter = self._get_title(item)[0]
        if letter.isdigit():
            return '#'
        else:
            return letter.lower()

    def set_frontend(self, frontend):
        super(VerticalWithPreview, self).set_frontend(frontend)
        # reset the stolen focus
        self.nodes.focus = True


class Coverflow(CoverflowListController):

    node_widget = ImageWithReflection

    def nodes_setup(self):
        super(Coverflow, self).nodes_setup()
        # FIXME: bad naming: looks like a model not a widget
        self.nodes = ListCoverflow(self.node_widget, 9)
        # We want our coverflow without opacity
        self.nodes.compute_opacity = lambda index: 255
        self.widget.add(self.nodes)
        self.nodes.width, self.nodes.height = (1.0, 1.0)
        self.nodes.x = 0.0
        self.nodes.y = 0.1
        self.nodes.visible = True
        self.nodes.focus = True

        self.titles_setup()
    def set_title_from_item(self, item):
        self.title.label = self._get_title(item)

    def node_renderer(self, item, widget):
        widget.clear()

        thumb = self._get_image_path_for_item(item)
        if thumb:
            widget.image.set_from_file(thumb)
            return

        self.frontend.load_from_theme(self.resource, widget.image)

    def set_frontend(self, frontend):
        super(Coverflow, self).set_frontend(frontend)
        # reset the stolen focus
        self.nodes.focus = True


class Grid(GridController):

    node_widget = GridItem

    def set_title_from_item(self, item):
        self.title.label = self._get_title(item)

    def node_renderer(self, item, widget):
        widget.image._image.clear()

        thumb = self._get_image_path_for_item(item)
        if thumb:
            widget.image._image.set_from_file(thumb)
            return

        self.frontend.load_from_theme(self.resource, widget.image.quick_image)

    def set_frontend(self, frontend):
        super(Grid, self).set_frontend(frontend)
        # reset the stolen focus
        self.nodes.focus = True


# tracks

class TracksVerticalWithPreview(VerticalWithPreview):
    album = None
    node_widget = DoubleLineMenuItemWidget

    def _get_path_for_item(self, item):
        if item is self.album:
            return self.album.cover_uri
        return super(TracksVerticalWithPreview, self)._get_path_for_item(item)

    def set_frontend(self, frontend):
        super(TracksVerticalWithPreview, self).set_frontend(frontend)
        # reset the stolen focus
        self.nodes.focus = True

        if self.album:
            self._load_preview(self.album)

    def _load_sublabel(self, item, widget):
        return

    def node_renderer(self, item, widget):
        super(TracksVerticalWithPreview, self).node_renderer(item, widget)
        self._load_sublabel(item, widget)

    def item_to_letter(self, item):
        letter = item.title[0]
        if letter.isdigit():
            return '#'
        else:
            return letter.lower()

    def _initiate_load_preview(self, widget, item, previous_item):
        if self.album:
            # don't refresh!
            return
        else:
            su = super(TracksVerticalWithPreview, self)
            return su._initiate_load_preview(widget, item, previous_item)
