# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import os.path

from elisa.core.common import application
from elisa.core.media_uri import MediaUri

from elisa.plugins.base.messages.device import DeviceRemoved

from elisa.plugins.poblesec.section import LoadableMixin
from elisa.plugins.poblesec.list_controller import ListController
from elisa.plugins.base.models.file import FileModel, DirectoryModel
from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.poblesec.notifying_dict import NotifyingDictionary

from elisa.plugins.poblesec.vertical_list import VerticalListController
from elisa.plugins.poblesec.section import MenuItemWidget
from elisa.plugins.poblesec.fast_scroll_controller import FastScrollListController
from elisa.plugins.poblesec.list_preview import PreviewListController
from elisa.plugins.poblesec.coverflow import CoverflowListController
from elisa.plugins.poblesec.grid import GridController
from elisa.plugins.poblesec.long_loading_image import LongLoadingImage
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.core.utils import typefinding

import pgm
from pgm.utils.image import cairo_gradient
from pgm.timing import implicit

from twisted.internet import defer

from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.core.utils.i18n import install_translation

_ = install_translation('poblesec')

class FilesystemController(ListController):

    default_uri = MediaUri("file://"+os.path.expanduser("~"))
    hide_files = False
    media_type = None

    filetype_to_resource = {
        "video": "elisa.plugins.poblesec.file_video",
        "audio": "elisa.plugins.poblesec.file_music",
        "image": "elisa.plugins.poblesec.file_picture",
        "directory": "elisa.plugins.poblesec.folder",
       }

    def initialize(self, uri=None, media_type=None):
        dfr = super(FilesystemController, self).initialize()
        bus = application.bus
        bus.register(self._device_remove_cb, DeviceRemoved)

        self._load_error = None
        self.uri = uri
        self.media_type = media_type
        self.gone_back = False

        def resource_loaded(resource):
            def is_not_hidden(file_model):
                return not file_model.name.startswith('.')

            non_hidden_files = filter(is_not_hidden, resource.files)
            non_hidden_files.sort(cmp=lambda x, y: cmp(x.name.lower(), y.name.lower()))

            if self.hide_files == True:
                def is_directory(file_model):
                    return isinstance(file_model, DirectoryModel)

                non_hidden_files = filter(is_directory, non_hidden_files)

            for my_file_model in non_hidden_files:
                try:
                    media_types = typefinding.typefind(MediaUri(my_file_model.uri))
                except typefinding.UnknownFileExtension, e:
                    media_types = []

                if self.media_type in media_types:
                    my_file_model.media_type = self.media_type
                else:
                    continue

            if self.media_type:
                def is_controller_mediatype(file_model):
                    if isinstance(file_model, DirectoryModel):
                        return True
                    return file_model.media_type == self.media_type

                non_hidden_files = filter(is_controller_mediatype, non_hidden_files)

            self.model.extend(non_hidden_files)
            return self

        def got_error(failure, resource):
            self._load_error = failure
            return resource

        def load_resource(controller):
            # we assume that self.uri is a directory
            # adds a slash in order to get the content of the directory
            self.uri = self.uri.join("")
            resource, dfr_resource = application.resource_manager.get(self.uri)
            dfr_resource.addErrback(got_error, resource)
            return dfr_resource

        if self.uri != None:
            dfr.addCallback(load_resource)
            dfr.addCallback(resource_loaded)

        return dfr


    def _device_remove_cb(self, msg, sender):
        if None not in (self.uri, msg.udi) and msg.udi == self.uri:
            # get history and go back to Music menu
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            if not self.gone_back:
                self.gone_back = True
                browser.history.go_back()


    def nodes_setup(self):
        super(FilesystemController, self).nodes_setup()
        media_type = self.media_type
        # FIXME: please clean the media_type values mess.
        if self._load_error:
            msg = _("There was an error loading the folder.")
        if media_type == 'video':
            msg = _("This folder does not contain any video files.")
        elif media_type in ('music', 'audio'):
            msg = _("This folder does not contain any audio files.")
        elif media_type in ('pictures', 'image'):
            msg = _("This folder does not contain any image files.")
        else:
            msg = _("This folder does not contain any multimedia files")
        self.nothing_to_display_widget.text.label = msg
        if not len(self.model) or self._load_error:
            self.nothing_to_display_widget.visible = True


    def set_frontend(self, frontend):
        # retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        self.gst_metadata = controllers[0].gst_metadata

        super(FilesystemController, self).set_frontend(frontend)

        self.nodes.focus = True

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(model, DirectoryModel):
                file_uri = model.uri

                # we assume there was only one controller created with that path
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                browser = controllers[0]
                dfr = browser.history.append_controller(self.path, model.name,
                                                        uri=file_uri, media_type=self.media_type)

            elif isinstance(model, FileModel):
                if model.media_type == "image":
                    self.play_image(model)
                elif model.media_type == "audio":
                    self.play_music_or_video(model, 'music')
                elif model.media_type == "video":
                    self.play_music_or_video(model, 'video')

                dfr = defer.succeed(True)

    def play_image(self, model):
        controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
        slideshow_controller = controllers[0]
        slideshow_controller.player.clear_playlist()

        for file in self.model:
            if isinstance(file, FileModel) and file.media_type == "image":
                if file == model:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                image = ImageModel()
                if file.thumbnail != None:
                    thumbnail_uri = MediaUri("file://"+file.thumbnail)
                    image.references.append(thumbnail_uri)
                image.references.append(file.uri)
                image.title = file.name
                slideshow_controller.player.enqueue_to_playlist(image)

        slideshow_controller.player.jump_to_index(index)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_slideshow_player()
        self.stop_loading_animation()

    def play_music_or_video(self, model, player):
        # player should be one of ('music', 'video')
        controllers = self.frontend.retrieve_controllers('/poblesec/%s_player' % player)
        player_controller = controllers[0]

        # enqueue and play the clicked file
        playable_model = PlayableModel()
        playable_model.uri = model.uri
        player_controller.player.play_model(playable_model)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        eval('main.show_%s_player()' % player)
        self.stop_loading_animation()

        # enqueue all the following files of the directory
        index = self.model.index(model)
        for file in self.model[index+1:]:
            if isinstance(file, DirectoryModel):
                continue
            playable_model = PlayableModel()
            playable_model.uri = file.uri
            player_controller.player.enqueue_to_playlist(playable_model)

    def _request_thumbnail(self, item, widget):
        return self._request_type_and_thumbnail(item, widget)

    def _request_type_and_thumbnail(self, item, widget):
        def on_metadata_set(notifier, metadata, key, value):
            if key == 'file_type':
                if item.media_type == None:
                    item.media_type = value
                    self._display_generic_icon(item, widget)
            elif key == 'thumbnail':
                item.thumbnail = value
                self._display_thumbnail(item, widget)

        metadata = {'uri': item.uri, 'thumbnail': None, 'file_type': None, \
                    'mime_type': None}
        metadata = NotifyingDictionary(metadata)
        handler_id = metadata.notifier.connect('item-changed', on_metadata_set)

        if self.gst_metadata.able_to_handle(metadata):
            dfr = self.gst_metadata.get_metadata(metadata)
            dfr.addBoth(lambda result: self._disconnect_metadata(metadata, handler_id))

            # FIXME: storing the deferred in the widget is ugly
            widget.dfr = dfr

    def _disconnect_metadata(self, metadata, handler_id):
        metadata.notifier.disconnect(handler_id)

    def _display_thumbnail(self, item, widget):
        pass

    def _display_generic_icon(self, item, widget):
        pass

    def clean(self):
        dfr = super(FilesystemController, self).clean()
        # FIXME: should cancel all ongoing thumbnailing requests
        return dfr

class Vertical(FilesystemController, VerticalListController):

    node_widget = MenuItemWidget

    def _display_thumbnail(self, item, widget):
        widget.icon.set_from_file(item.thumbnail, 64)

    def _display_generic_icon(self, item, widget):
        fallback = "elisa.plugins.poblesec.file"
        resource = self.filetype_to_resource.get(item.media_type, fallback)
        self.frontend.load_from_theme(resource, widget.icon)

    def node_renderer(self, item, widget):
        # cancel previously made typefinding and thumbnailing request relative
        # to widget
        if hasattr(widget, "dfr") and not widget.dfr.called:
            widget.dfr.cancel()
            delattr(widget, "dfr")

        widget.label.label = item.name
        widget.icon.clear()

        if item.media_type != None:
            if item.thumbnail != None:
                self._display_thumbnail(item, widget)
            else:
                self._display_generic_icon(item, widget)
                self._request_thumbnail(item, widget)
        else:
            self._display_generic_icon(item, widget)
            self._request_type_and_thumbnail(item, widget)

class VerticalWithFastScroll(Vertical, FastScrollListController):

    shortcuts = list("#abcdefghijklmnopqrstuvwxyz")

    def item_to_letter(self, item):
        letter = item.name[0]
        if letter.isdigit():
            return '#'
        else:
            return letter.lower()

class VerticalWithPreview(Vertical, PreviewListController):

    def image_from_item(self, item):
        if item.thumbnail != None:
            return item.thumbnail
        else:
            if isinstance(item, DirectoryModel):
                resource = 'elisa.plugins.poblesec.folder'
            else:
                if item.media_type in self.filetype_to_resource.keys():
                    resource = self.filetype_to_resource[item.media_type]
                else:
                    resource = 'elisa.plugins.poblesec.file'

            theme = self.frontend.get_theme()
            return theme.get_resource(resource)

class VerticalWithFastScrollAndPreview(VerticalWithFastScroll, VerticalWithPreview):
    pass


class ImageWithReflection(Widget, LoadableMixin):

    loading_attribute = 'image'

    def __init__(self):
        super(ImageWithReflection, self).__init__()

        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.x, self.image.y = (0.0, 0.0)
        self.image.width, self.image.height = (1.0, 0.5)
        self.image.visible = True

        # reflection
        self.reflection = Image()
        self.add(self.reflection)
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.x, self.reflection.y = (0.0, 0.5)
        self.reflection.width, self.reflection.height = (1.0, 0.5)
        self.reflection.opacity = 50
        self.reflection.visible = True

    def set_from_file(self, image_path):
        self.image.set_from_file(image_path)
        # FIXME: deactivated because of Python Cairo bindings not releasing
        # the GIL thus locking Pigment's rendering thread when the viewport
        # update-pass signal is emitted
        #cairo_gradient(image_path, self.reflection, 0.4)

    def clear(self):
        self.image.clear()
        self.reflection.clear()

class Coverflow(CoverflowListController, FilesystemController):

    node_widget = ImageWithReflection

    def node_renderer(self, item, widget):
        widget.clear()
        resource = "elisa.plugins.poblesec.file"
        theme = self.frontend.get_theme()
        file = theme.get_resource(resource)
        widget.set_from_file(file)

        # cancel previously made typefinding and thumbnailing request relative
        # to widget
        if hasattr(widget, "dfr") and not widget.dfr.called:
            widget.dfr.cancel()
            delattr(widget, "dfr")

        widget.clear()

        if item.media_type != None:
            if item.thumbnail != None:
                self._display_thumbnail(item, widget)
            else:
                self._display_generic_icon(item, widget)
                self._request_thumbnail(item, widget)
        else:
            self._display_generic_icon(item, widget)
            self._request_type_and_thumbnail(item, widget)

    def _display_thumbnail(self, item, widget):
        widget.set_from_file(item.thumbnail)

    def _display_generic_icon(self, item, widget):
        fallback = "elisa.plugins.poblesec.file"
        resource = self.filetype_to_resource.get(item.media_type, fallback)

        theme = self.frontend.get_theme()
        file = theme.get_resource(resource)
        widget.set_from_file(file)

    def set_title_from_item(self, item):
        self.title.label = item.name

class GridItem(Widget, LoadableMixin):

    loading_attribute = 'image'

    def __init__(self):
        super(GridItem, self).__init__()

        self.image = LongLoadingImage()
        self.add(self.image)
        self.image.width, self.image.height = (0.7, 0.7)
        self.image.x = (1.0-self.image.width)/2.0
        self.image.y = (1.0-self.image.height)/2.0
        self.image.border_outer_color = (255, 255, 255, 255)
        self.image.visible = True

        self.image.image.border_width = 0.010
        self.image.image.layout = pgm.IMAGE_ZOOMED


class Grid(GridController, FilesystemController):

    node_widget = GridItem

    def _display_thumbnail(self, item, widget):
        widget.image.image.set_from_file(item.thumbnail)

    def _display_generic_icon(self, item, widget):
        fallback = "elisa.plugins.poblesec.file"
        resource = self.filetype_to_resource.get(item.media_type, fallback)
        self.frontend.load_from_theme(resource, widget.image.quick_image)

    def node_renderer(self, item, widget):
        # cancel previously made typefinding and thumbnailing request relative
        # to widget
        if hasattr(widget, "dfr") and not widget.dfr.called:
            widget.dfr.cancel()
            delattr(widget, "dfr")

        widget.image.clear()
        self._display_generic_icon(item, widget)

        if item.media_type != None:
            if item.thumbnail != None:
                self._display_thumbnail(item, widget)
            else:
                self._request_thumbnail(item, widget)
        else:
            self._display_generic_icon(item, widget)
            self._request_type_and_thumbnail(item, widget)

    def set_title_from_item(self, item):
        self.title.label = item.name

class ListSwitcher(ListSwitcherController):

    modes = [VerticalWithFastScrollAndPreview, Coverflow, Grid]

class ListSwitcherVertical(ListSwitcher):

    default_mode = VerticalWithFastScrollAndPreview

class ListSwitcherCoverflow(ListSwitcher):

    default_mode = Coverflow

class ListSwitcherGrid(ListSwitcher):

    default_mode = Grid
