# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.poblesec.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.list_controller import ListController

from elisa.plugins.pigment.widgets.grid_horizontal import GridHorizontal
from elisa.plugins.pigment.graph.text import Text

import pgm
from pgm.timing import implicit

from elisa.core.input_event import *

class GridController(ListController):

    def nodes_setup(self):
        super(GridController, self).nodes_setup()
        # FIXME: default number of rows should be configurable
        rows = 3
        # default dummy value
        columns = rows

        # FIXME: bad naming: looks like a model not a widget
        self.nodes = GridHorizontal(self.node_widget, rows, columns)
        self.widget.add(self.nodes)
        # FIXME: hardcoded value
        self.nodes.width, self.nodes.height = (1.0, 0.75)
        self.nodes.x = 0.0
        # FIXME: hardcoded value
        self.nodes.y = 1.0-self.nodes.height-0.10
        self.nodes.visible = True
        self.nodes.focus = True

        # selector setup
        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.grid_selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.grid_selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.grid_selector_body')
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)

        self.nodes.set_selector(selector)

        self.titles_setup()

    def titles_setup(self):
        self.title = Text()
        self.widget.add(self.title)
        self.title.weight = pgm.TEXT_WEIGHT_BOLD
        self.title.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self.title.alignment = pgm.TEXT_ALIGN_CENTER
        self.title.bg_a = 0
        # FIXME: hardcoded font
        self.title.font_family = "Liberation Sans"
        # FIXME: hardcoded values
        self.title.width = 0.65
        self.title.height = 0.04
        self.title.x = (1.0-self.title.width)/2.0
        self.title.y = (1.0- (self.nodes.height + 0.05) -self.title.height)/2.0
        self.title.visible = True

        self.animated_title = implicit.AnimatedObject(self.title)

    def node_selected(self, widget, item, previous_item):
        self._fade_out_title()

    def set_title_from_item(self, item):
        pass

    def _fade_out_title(self):
        # do not do anything if it was already fading out
        if self.animated_title.opacity == 0.0:
            return

        def faded_out(timer):
            self._fade_in_title()
            selected_item = self.model[self.nodes.selected_item_index]
            self.set_title_from_item(selected_item)

        # fade out
        self.animated_title.setup_next_animations(duration=200,
                                      transformation=implicit.DECELERATE,
                                      end_callback=faded_out)
        self.animated_title.opacity = 0.0

    def _fade_in_title(self):
        # fade in
        self.animated_title.setup_next_animations(duration=200,
                                        transformation=implicit.DECELERATE)
        self.animated_title.opacity = 255

    def prepare(self):
        aspect_ratio = self.nodes.absolute_width/self.nodes.absolute_height
        columns = self.nodes.rows*aspect_ratio
        self.nodes.visible_range_size = columns

    def handle_input(self, manager, input_event):
        if self.nothing_to_display_widget.visible:
            return False

        if input_event.value == EventValue.KEY_GO_LEFT:
            index = self.nodes.selected_item_index - self.nodes.rows
            if index >= 0:
                self.nodes.selected_item_index = index
            return True
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            index = self.nodes.selected_item_index + self.nodes.rows
            if index < len(self.model):
                self.nodes.selected_item_index = index
            return True
        if input_event.value == EventValue.KEY_GO_UP:
            if self.nodes.selected_item_index % self.nodes.rows == 0:
                # first row of the grid (top)
                return False
            else:
                self.nodes.selected_item_index -= 1
                return True
        elif input_event.value == EventValue.KEY_GO_DOWN:
            if self.nodes.selected_item_index % self.nodes.rows == self.nodes.rows-1:
                # last row of the grid (bottom)
                return False
            else:
                self.nodes.selected_item_index += 1
                return True
        else:
            return super(GridController, self).handle_input(manager, input_event)
