# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.poblesec.section import MenuItemWidget
from elisa.plugins.poblesec.list_controller import ListController
from elisa.plugins.poblesec.vertical_list import VerticalListController
from elisa.plugins.poblesec.fast_scroll_controller import FastScrollListController
from elisa.plugins.poblesec.section import SectionMenuController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.coverflow import CoverflowListController
from elisa.plugins.poblesec.grid import GridController
from elisa.plugins.poblesec.nothing_to_display import NothingToDisplay

# FIXME: we should factor out useful stuff, like some widgets
from elisa.plugins.poblesec.filesystem_controller import ImageWithReflection
from elisa.plugins.poblesec.filesystem_controller import GridItem


from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.models.network import NetworkServiceModel
from elisa.plugins.coherence.upnp_resource import IMAGE_MAGIC
from elisa.plugins.coherence.models import UpnpContainerModel
from elisa.plugins.coherence.models import UpnpItemModel
from elisa.plugins.coherence.models import UpnpImageModel
from elisa.plugins.coherence.models import UpnpTrackModel
from elisa.plugins.coherence.models import UpnpVideoModel
from elisa.plugins.coherence.models import UpnpAlbumModel
from elisa.plugins.coherence.models import UpnpArtistModel

from elisa.plugins.base.messages.network import NetworkServiceAppearedMessage
from elisa.plugins.base.messages.network import NetworkServiceDisappearedMessage

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets import notifying_list

from elisa.core.input_event import *
from elisa.core.media_uri import MediaUri
from elisa.core.common import application
from elisa.core.utils.i18n import install_translation

_ = install_translation('poblesec')

from twisted.internet import defer
import pgm
import copy
import os


"""
TODO
  * images slideshow
  * actually activate item ordering
  ✓ on the fly updating of new/removed shares
  ✓ playing of audio and video
  ✓ media type filtering / item icons
  ✓ item ordering
  ✓ check whether the share is actually accessible
"""


class NetworkSharesList(SectionMenuController):

    def initialize(self, media_type=None):
        dfr = super(NetworkSharesList, self).initialize()
        self.media_type = media_type
        self.shares = []
        return dfr

    def set_frontend(self, frontend):
        super(NetworkSharesList, self).set_frontend(frontend)
        return self.list_network_shares(self.media_type)

    def list_network_shares(self, media_type):
        # Keep 3 slashes
        uri = MediaUri("network-services:///")
        model, dfr = application.resource_manager.get(uri, None)

        def do_list_network_shares(model):
            for share in model.services:
                # FIXME: loosing deferreds?
                self._add_share(share)

            application.bus.register(self._share_added_cb, NetworkServiceAppearedMessage)
            application.bus.register(self._share_removed_cb, NetworkServiceDisappearedMessage)

            return model

        def do_list_network_shares_error(err, *args):
            return err

        dfr.addCallback(do_list_network_shares)
        dfr.addErrback(do_list_network_shares_error)

        return dfr

    def _add_share(self, share):
        link = Link()
        link.controller_path = "/poblesec/network"
        link.controller_args = {'uri': share.elisa_uri,
                                'media_type': self.media_type}
        link.label = share.name
        if self.media_type:
            link.icon = "elisa.plugins.poblesec.%s_folders" % (self.media_type)
        else:
            link.icon = "elisa.plugins.poblesec.folder"

# FIXME: activate this code (alphabetic ordering of shares) when the list
# refreshing is fixed.
#        i = 0
#        for i, s in enumerate(self.shares):
#            if s.name.lower() > share.name.lower():
#                break
#        self.model.insert(i, link)
#        self.shares.insert(i, share)
        self.model.append(link)
        self.shares.append(share)

        def set_share_status(share, link):
            return share

        def set_share_status_error(err, link):
            self.debug("Error while accessing share '%s': %s" % (link.label, err))
            index = self.model.index(link)
            link = copy.copy(link)
            link.icon = "elisa.plugins.poblesec.folder_unaccesible"
            self.model[index] = link

        resource, dfr = application.resource_manager.get(share.elisa_uri)

        dfr.addCallback(set_share_status, link)
        dfr.addErrback(set_share_status_error, link)

        return dfr

    def _share_added_cb(self, msg, sender):
        self.debug("New share appeared: %s" % msg.model.name)
        # FIXME: loosing a deferred
        self._add_share(msg.model)

    def _share_removed_cb(self, msg, sender):
        uids = [s.uid for s in self.shares]

        if msg.uid in uids:
            index = uids.index(msg.uid)
            share = self.shares.pop(index)
            self.debug("Removing share '%s'" % share.name)
            self.model.pop(index)
        else:
            self.warning("Unknown share disappeared: '%s'" % msg.uid)

    def clean(self):
        dfr = super(NetworkSharesList, self).clean()
        application.bus.unregister(self._share_added_cb)
        application.bus.unregister(self._share_removed_cb)
        return dfr


class NetworkShareBrowser(ListController):

    itemclass_to_resource = {
        UpnpItemModel: "elisa.plugins.poblesec.file",
        UpnpContainerModel: "elisa.plugins.poblesec.folder",
        UpnpTrackModel: "elisa.plugins.poblesec.file_music",
        UpnpVideoModel: "elisa.plugins.poblesec.file_video",
        UpnpImageModel: "elisa.plugins.poblesec.file_picture",
        UpnpAlbumModel: "elisa.plugins.poblesec.folder",
        UpnpArtistModel: "elisa.plugins.poblesec.folder",
        NetworkServiceModel: "elisa.plugins.poblesec.folder",
       }
       
    # translate the type of the section into a "media type"
    # (types defined in elisa.plugins.base.models.file.FileModel)
    translation = {'music': 'audio',
                   'video': 'video',
                   'pictures': 'image'}

    def initialize(self, uri=None, media_type=None):
        dfr = super(NetworkShareBrowser, self).initialize()

        self.uri = uri
        self.media_type = media_type

        self.nothing_to_display_widget = NothingToDisplay(self.media_type)
        self.widget.add(self.nothing_to_display_widget)
        self.nothing_to_display_widget.visible = False

        def upnp_resource_loaded(resource):
            if self.media_type == 'pictures':
                items = [i for i in resource.items if isinstance(i, (UpnpImageModel, UpnpContainerModel))]
            elif self.media_type == 'music':
                items = [i for i in resource.items if isinstance(i, (UpnpTrackModel, UpnpContainerModel))]
            elif self.media_type == 'video':
                items = [i for i in resource.items if isinstance(i, (UpnpVideoModel, UpnpContainerModel))]
            else:
                items = resource.items

            self.debug("Filtering %s" % self.media_type)

            self.model = sorted(items, key=lambda i: i.name.lower())

            return self
        
        def smb_resource_loaded(resource):
            items = resource.servers

            self.debug("Filtering %s" % self.media_type)

            self.model = sorted(items, key=lambda i: i.name.lower())

            return self
        
        def general_resource_loaded(resource):
            items = resource.files

            self.debug("Filtering %s" % self.media_type)

            self.model = sorted(items, key=lambda i: i.name.lower())

            return self

        def load_resource(controller):
            # we assume that self.uri is a directory
            resource, dfr_resource = application.resource_manager.get(self.uri)
            return dfr_resource

        if self.uri != None:
            dfr.addCallback(load_resource)
            if self.uri.scheme == 'upnp':
                loaded_cb = upnp_resource_loaded
            elif self.uri.scheme == 'smb':
                loaded_cb = smb_resource_loaded
            else:
                loaded_cb = general_resource_loaded
            
            dfr.addCallback(loaded_cb)

        return dfr

    def set_frontend(self, frontend):
        super(NetworkShareBrowser, self).set_frontend(frontend)
        # reset the stolen focus
        self.nodes.focus = True

    def nodes_setup(self):
        super(NetworkShareBrowser, self).nodes_setup()
        media_type = self.media_type
        # FIXME: please clean the media_type values mess.
        if media_type == 'video':
            msg = _("This shared folder does not contain any video files.")
        elif media_type in ('music', 'audio'):
            msg = _("This shared folder does not contain any audio files.")
        elif media_type in ('pictures', 'image'):
            msg = _("This shared folder does not contain any image files.")
        else:
            msg = _("This shared folder does not contain any multimedia files")
        self.nothing_to_display_widget.text.label = msg
        if not len(self.model):
            self.nothing_to_display_widget.visible = True

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(model, (NetworkServiceModel, UpnpContainerModel)):
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                browser = controllers[0]

                if model.elisa_uri.scheme == 'file':
                    path = '/poblesec/%s/filesystem' % self.media_type
                    file_media_type = self.translation[self.media_type]
                else:
                    path = self.path
                    file_media_type = self.media_type
                dfr = browser.history.append_controller(path,
                                                  model.name,
                                                  uri=model.elisa_uri,
                                                  media_type=file_media_type)
            elif isinstance(model, UpnpImageModel):
                self.play_image(model)
                dfr = defer.succeed(True)
            elif isinstance(model, UpnpVideoModel):
                self.play_audio_video(model, 'video')
                dfr = defer.succeed(True)
            elif isinstance(model, UpnpTrackModel):
                self.play_audio_video(model, 'music')
                dfr = defer.succeed(True)

    def play_image(self, model):
        controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
        slideshow_controller = controllers[0]
        slideshow_controller.player.clear_playlist()

        for item in self.model:
            if isinstance(item, UpnpImageModel):
                if item == model:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                image = ImageModel()
                uri = MediaUri(item.playable_uri + IMAGE_MAGIC)
                image.references.append(uri)
                image.title = item.name
                slideshow_controller.player.enqueue_to_playlist(image)

        slideshow_controller.player.jump_to_index(index)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_slideshow_player()
        self.stop_loading_animation()

    def play_audio_video(self, model, player):
        # player should be one of ('music', 'video')
        controllers = self.frontend.retrieve_controllers('/poblesec/%s_player' % player)
        player_controller = controllers[0]

        # enqueue and play the clicked item
        model.playable_model.title = model.name
        player_controller.player.play_model(model.playable_model)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        eval('main.show_%s_player()' % player)
        self.stop_loading_animation()

        # enqueue all the following items of the container
        index = self.model.index(model)
        for item in self.model[index+1:]:
            if isinstance(item, UpnpContainerModel):
                continue
            item.playable_model.title = item.name
            player_controller.player.enqueue_to_playlist(item.playable_model)

    def _disconnect_metadata(self, metadata, handler_id):
        metadata.notifier.disconnect(handler_id)


class NetworkVertical(NetworkShareBrowser, VerticalListController):

    node_widget = MenuItemWidget

    def node_renderer(self, item, widget):
        widget.label.label = item.name
        widget.icon.clear()
        fallback = "elisa.plugins.poblesec.file"
        resource = self.itemclass_to_resource.get(item.__class__, fallback)
        self.frontend.load_from_theme(resource, widget.icon)


class NetworkVerticalWithFastScroll(NetworkVertical, FastScrollListController):

    shortcuts = list("#abcdefghijklmnopqrstuvwxyz")

    def item_to_letter(self, item):
        letter = item.name[0]
        if letter.isdigit():
            return '#'
        else:
            return letter.lower()


class NetworkCoverflow(CoverflowListController, NetworkShareBrowser):

    node_widget = ImageWithReflection

    def node_renderer(self, item, widget):
        widget.clear()
        resource = "elisa.plugins.poblesec.file"
        theme = self.frontend.get_theme()
        file = theme.get_resource(resource)
        widget.set_from_file(file)
        widget.clear()
        fallback = "elisa.plugins.poblesec.file"
        resource = self.itemclass_to_resource.get(item.__class__, fallback)
        theme = self.frontend.get_theme()
        file = theme.get_resource(resource)
        widget.set_from_file(file)

    def set_title_from_item(self, item):
        self.title.label = item.name


class NetworkGrid(GridController, NetworkShareBrowser):

    node_widget = GridItem

    def node_renderer(self, item, widget):
        widget.image.clear()
        fallback = "elisa.plugins.poblesec.file"
        resource = self.itemclass_to_resource.get(item.__class__, fallback)
        self.frontend.load_from_theme(resource, widget.image.quick_image)

    def set_title_from_item(self, item):
        self.title.label = item.name


class NetworkListSwitcher(ListSwitcherController):

    modes = [NetworkVerticalWithFastScroll, NetworkCoverflow, NetworkGrid]


class NetworkListSwitcherVertical(NetworkListSwitcher):

    default_mode = NetworkVerticalWithFastScroll


class NetworkListSwitcherCoverflow(NetworkListSwitcher):

    default_mode = NetworkCoverflow


class NetworkListSwitcherGrid(NetworkListSwitcher):

    default_mode = NetworkGrid

