# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Widget and controller for a popup dialog.
"""

import pgm

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.widgets.list_horizontal import ListHorizontal
from elisa.plugins.pigment.widgets.notifying_list import List
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text

from elisa.plugins.poblesec.sliced_image import SlicedImageHorizontal

from twisted.internet import defer

from elisa.core.input_event import EventValue


class PopupWidget(Widget):

    """
    Popup dialog widget (containing an icon, a title, a body text, and a
    button).
    """

    def __init__(self, icon, title, text, buttons):
        super(PopupWidget, self).__init__()
        self._selected_item_index = -1

        self.background = Image()
        self.add(self.background)
        self.background.x, self.background.y = (0.0, 0.0)
        self.background.width, self.background.height = (1.0, 1.0)
        self.background.bg_color = (0, 0, 0, 200)
        self.background.visible = True

        self.icon = Image()
        self.add(self.icon)
        self.icon.x, self.icon.y = (0.04, 0.4)
        self.icon.width, self.icon.height = (0.2, 0.2)
        theme = Theme.get_default()
        image = theme.get_resource(icon)
        self.icon.set_from_file(image)
        self.icon.bg_color = (0, 0, 0, 0)
        self.icon.visible = True

        self.title = Text()
        self.add(self.title)
        self.title.weight = pgm.TEXT_WEIGHT_BOLD
        self.title.label = title
        self.title.x, self.title.y = (0.26, 0.38)
        self.title.width, self.title.height = (0.65, 0.05)
        self.title.bg_color = (0, 0, 0, 0)
        self.title.visible = True

        self.text = Text()
        self.add(self.text)
        self.text.font_height = 0.32
        self.text.weight = pgm.TEXT_WEIGHT_BOLD
        self.text.label = text
        self.text.x, self.text.y = (0.26, 0.45)
        self.text.width, self.text.height = (0.65, 0.12)
        self.text.bg_color = (0, 0, 0, 0)
        self.text.fg_color = (200, 200, 200, 255)
        self.text.visible = True

        button_size = (0.3, 0.06)
        first_button_x = 0.26
        button_y = 0.63

        self.selector = SlicedImageHorizontal()
        self.add(self.selector)
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        self.selector.left_cap.set_from_file(left_cap)
        self.selector.right_cap.set_from_file(right_cap)
        self.selector.body.set_from_file(body)
        self.selector.x, self.selector.y = (first_button_x, button_y)
        self.selector.width, self.selector.height = button_size
        self.selector.bg_color = (0, 0, 0, 255)
        self.selector.visible = True

        self.buttons = buttons
        self.button_widgets = []
        self.callbacks = {}

        def button_clicked_cb(text, x, y, z, button, time, pressure):
            self.callbacks[text.label]()
            return True

        for text, callback in self.buttons:

            button = Text()
            self.add(button)
            button.font_height = 0.8
            button.weight = pgm.TEXT_WEIGHT_BOLD
            button.alignment = pgm.TEXT_ALIGN_CENTER
            button.label = text
            x = first_button_x + (button_size[0] * len(self.button_widgets))
            button.x, button.y = (x, button_y)
            button.width, button.height = button_size
            button.bg_color = (0, 0, 0, 0)
            button.visible = True

            self.callbacks[text] = callback
            button.connect('clicked', button_clicked_cb)

            self.button_widgets.append(button)

        self._update_style_properties(self._style.get_items())

    def _set_selected_item_index(self, value):
        if value in range(len(self.button_widgets)):
            self._selected_item_index = value
            button = self.button_widgets[value]
            button.focus = True
            self.selector.x = button.x
            for b in self.button_widgets:
                if b != button:
                    b.focus = False

    def _get_selected_item_index(self):
        return self._selected_item_index

    selected_item_index = property(fget=_get_selected_item_index,
                                   fset=_set_selected_item_index)

class PopupController(PigmentController):

    """
    Popup dialog controller.
    """

    def __init__(self):
        super(PopupController, self).__init__()
        self.popup = None

    def initialize(self, icon, title, text, buttons):
        self.icon = icon
        self.title = title
        self.text = text
        self.buttons = buttons
        return defer.succeed(self)

    def _set_text(self, value):
        self._text = value
        if self.popup:
            self.popup.text.label = value

    def _get_text(self):
        return self._text

    text = property(fget=_get_text, fset=_set_text)

    def set_frontend(self, frontend):
        super(PopupController, self).set_frontend(frontend)

        self.popup = PopupWidget(self.icon, self.title, self.text, self.buttons)
        self.widget.add(self.popup)
        self.popup.x, self.popup.y = (0, 0)
        self.popup.z = 0.1
        self.popup.width, self.popup.height = (1.0, 1.0)
        self.popup.visible = True

        # by default the focus is forwarded to the first button of the popup
        self.popup.button_widgets[0].focus = self.widget.focus
        self.widget.connect('focus', self._on_focus)

    def _on_focus(self, widget, focus):
        # always forward the focus to the buttons of the popup
        if focus:
            self.popup.selected_item_index = 0

    def handle_input(self, manager, input_event):
        if input_event.value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
            self.buttons[self.popup.selected_item_index][1]()
        elif input_event.value == EventValue.KEY_GO_LEFT:
            self.popup.selected_item_index -= 1
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            self.popup.selected_item_index += 1
        return True
