# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import common
from elisa.core.common import application
from elisa.core.media_uri import MediaUri
from elisa.plugins.poblesec.list_preview import PreviewListController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image
from pgm.timing import implicit
from elisa.plugins.base.messages.device import NewDeviceDetected, NewUnknownDevice, DeviceRemoved

from twisted.internet import reactor

import pgm
import math


class LoadableMixin(object):
    """Add loading animation feedback functionality to widgets."""

    loading_attribute = None

    def activate(self, previous=None):
        # FIXME: we need the frontend to use cloned images (by
        # "load_from_theme). This a cheap - UGLY - way to get the frontend
        # without changing much client code.
        from elisa.core.common import application
        self.frontend = application.interface_controller.frontends['frontend1']

        if previous:
            if previous != self:
                previous._toggle_loading_animation(False)

        if previous != self:
            self._allow_animation = True
            icon = getattr(self, self.loading_attribute)
            self.animated_icon = implicit.AnimatedObject(icon)
            settings = {'end_callback' : lambda ctrl: self._toggle_loading_animation(True),
                        'duration' : 400}
            self.animated_icon.setup_next_animations(**settings)
            self.animated_icon.opacity = 0

            self._interaction = None
            self._loading = False
            self._rotation_matrix = pgm.mat4x4_new_identity()
            self._rotation_matrix.translate(0.5, 0.5, 0.0)
            self._rotation_matrix.rotate_z(-math.pi / 30.0)
            self._rotation_matrix.translate(-0.5, -0.5, 0.0)

            self._create_interaction()

    def _create_interaction(self):
        self._interaction = Image()
        self.add(self._interaction)

        icon = getattr(self, self.loading_attribute)

        width, height = (icon.width, icon.height)
        x, y = (icon.x, icon.y)
        self._interaction.width, self._interaction.height = (width, height)
        self._interaction.x, self._interaction.y = (x, y)

        # downscaling
        scale = pgm.mat4x4_new_identity()
        scale.translate(0.5, 0.5, 0.0)
        factor = 2.0
        scale.scale(factor, factor, 0.0)
        scale.translate(-0.5, -0.5, 0.0)

        self._interaction.bg_color = (0, 0, 0, 0)
        self._interaction.opacity = 255
        self._interaction.visible = True

    def _toggle_loading_animation(self, enabled):
        if enabled and self._allow_animation:
            if self._loading:
                return
            if not hasattr(self, '_interaction') or not self._interaction:
                self._create_interaction()
            wait_icon = Theme.get_default().get_resource("elisa.plugins.poblesec.waiting_icon")
            self._interaction.visible = True
            self.frontend.load_from_theme("elisa.plugins.poblesec.waiting_icon", self._interaction)
            self._interaction.set_name('wait icon')
            self._loading = True
            reactor.callLater(0.023, self._transform_mapping_matrix_cb)
        else:
            self._allow_animation = False
            self._loading = False
            self._interaction.clear()
            self._interaction.visible = False
            settings = {'end_callback' : lambda ctrl: self._toggle_loading_animation(True),
                        'duration' : 400}
            self.animated_icon.setup_next_animations(**settings)
            self.animated_icon.opacity = 255

    def _transform_mapping_matrix_cb(self):
        if self._loading:
            self._interaction.mapping_matrix *= self._rotation_matrix
            reactor.callLater(0.017, self._transform_mapping_matrix_cb)


class MenuItemWidget(Widget, LoadableMixin):

    loading_attribute = 'icon'

    def __init__(self):
        super(MenuItemWidget, self).__init__()
        self._create_widgets()
        self._update_style_properties(self._style.get_items())

    def _create_widgets(self):
        label = Text()
        self.label = label
        self.add(label)
        label.weight = pgm.TEXT_WEIGHT_BOLD
        label.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        label.bg_a = 0

        icon = Image()
        self.icon = icon
        self.add(icon)
        icon.bg_a = 0

        label.visible = True
        icon.visible = True

    def _update_style_properties(self, props=None):
        super(MenuItemWidget, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'font-family':
                self.label.font_family = value
            elif key == 'label-x':
                self.label.x = value
            elif key == 'label-y':
                self.label.y = value
            elif key == 'label-width':
                self.label.width = value
            elif key == 'label-height':
                self.label.height = value
            elif key == 'icon-x':
                self.icon.x = value
            elif key == 'icon-y':
                self.icon.y = value
            elif key == 'icon-width':
                self.icon.width = value
            elif key == 'icon-height':
                self.icon.height = value

class DoubleLineMenuItemWidget(MenuItemWidget):

    def _create_widgets(self):
        super(DoubleLineMenuItemWidget, self)._create_widgets()
        sublabel = Text()
        self.sublabel = sublabel
        self.add(sublabel)
        sublabel.weight = pgm.TEXT_WEIGHT_NORMAL
        sublabel.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        sublabel.bg_a = 0
        sublabel.fg_color = (195, 194, 194, 255)
        sublabel.visible = True


    def _update_style_properties(self, props=None):
        super(DoubleLineMenuItemWidget, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'font-family':
                self.label.font_family = value
            elif key == 'label-x':
                self.label.x = value
            elif key == 'label-y':
                self.label.y = value
            elif key == 'label-width':
                self.label.width = value
            elif key == 'label-height':
                self.label.height = value
            elif key == 'sublabel-x':
                self.sublabel.x = value
            elif key == 'sublabel-y':
                self.sublabel.y = value
            elif key == 'sbutitle-width':
                self.sublabel.width = value
            elif key == 'sublabel-height':
                self.sublabel.height = value
            elif key == 'icon-x':
                self.icon.x = value
            elif key == 'icon-y':
                self.icon.y = value
            elif key == 'icon-width':
                self.icon.width = value
            elif key == 'icon-height':
                self.icon.height = value

class SectionMenuController(PreviewListController):

    node_widget = MenuItemWidget

    def set_frontend(self, frontend):
        super(SectionMenuController, self).set_frontend(frontend)

    def node_renderer(self, item, widget):
        widget.label.label = item.label
        self.frontend.load_from_theme(item.icon, widget.icon)

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            # we assume there was only one controller created with that path
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            path = item.controller_path
            label = item.label
            args = item.controller_args
            dfr = browser.history.append_controller(path, label, **args)

            self._previous_clicked = selected_item

    def image_from_item(self, item):
        theme = self.frontend.get_theme()
        image_path = theme.get_resource(item.icon)
        return image_path

class HardwareListControllerMixin(SectionMenuController):

    # translate the type of the section into a "media type"
    # (types defined in elisa.plugins.base.models.file.FileModel)
    translation = {'music': 'audio',
                   'video': 'video',
                   'pictures': 'image'}

    def initialize(self):
        dfr = super(HardwareListControllerMixin, self).initialize()
        bus = common.application.bus
        bus.register(self._device_add_cb, NewDeviceDetected)
        bus.register(self._device_remove_cb, DeviceRemoved)
        return dfr

    def set_frontend(self, frontend):
        super(HardwareListControllerMixin, self).set_frontend(frontend)

    def _device_add_cb(self, msg, sender):
        # FIXME: only file protocol supported at this time, no audio cd's nor dvd's
        link = Link()

        if msg.model.protocol == 'file':
            file_media_type = self.translation[self.media_type]
            link.controller_path = "/poblesec/%s/filesystem" % (self.media_type)
            link.controller_args = {'uri': msg.model.mount_point, 'media_type': file_media_type}
            link.icon = "elisa.plugins.poblesec.removable_device"
        elif msg.model.protocol == 'ipod' and self.media_type == 'music':
            link.controller_path = "/poblesec/music/ipod"
            link.controller_args = {'uri': msg.model.mount_point}
            link.icon = "elisa.plugins.poblesec.ipod"
        else:
            # not supported yet
            return

        for existing_link in self.model:
            uri = existing_link.controller_args.get('uri', None)
            if msg.model.mount_point and uri == msg.model.mount_point:
                # already added, can be in some cases ...
                return

        link.label = msg.model.label
        self.model.append(link)

    def _device_remove_cb(self, msg, sender):
        for link in self.model:
            uri = link.controller_args.get('uri', None)
            if None not in (uri, msg.udi) and msg.udi == uri:
                self.model.remove(link)
                return

    def clean(self):
        dfr = super(HardwareListControllerMixin, self).clean()
        bus = common.application.bus
        bus.unregister(self._device_add_cb)
        bus.unregister(self._device_remove_cb)
        return dfr

class MusicHardwareListController(HardwareListControllerMixin):
    media_type = 'music'

class VideoHardwareListController(HardwareListControllerMixin):
    media_type = 'video'

class PicturesHardwareListController(HardwareListControllerMixin):
    media_type = 'pictures'
