# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.utils.cancellable_defer import CancelledError

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.list import List
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image

from elisa.plugins.poblesec.base.list import ListController
from elisa.plugins.poblesec.widgets.image_with_reflection import \
    ImageWithReflection

import pgm
from pgm.utils import maths
from pgm.timing import implicit


class ItemWidget(Widget):
    def __init__(self):
        super(ItemWidget, self).__init__()

        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.layout = pgm.IMAGE_SCALED
        self.image.alignment = pgm.IMAGE_BOTTOM
        self.image.position = (0.0, 0.0, 0.0)
        self.image.size = (1.0, 0.5)
        self.image.visible = True

        # reflection
        self.reflection = Image()
        self.add(self.reflection)
        flip_matrix = pgm.mat4x4_new_predefined(pgm.MAT4X4_FLIP_VERTICAL)
        self.reflection.mapping_matrix = flip_matrix
        self.reflection.bg_a = 0
        self.reflection.layout = pgm.IMAGE_SCALED
        self.reflection.alignment = pgm.IMAGE_TOP
        self.reflection.position = (0.0, 0.5, 0.0)
        self.reflection.size = (1.0, 0.5)
        self.reflection.opacity = 50
        self.reflection.visible = True

class CoverflowList(List):

    def visible_range_size__set(self, visible_range_size):
        self._item_width = 0.3
        self._item_height = 0.6
        super(CoverflowList, self).visible_range_size__set(visible_range_size)

    visible_range_size = property(List.visible_range_size__get,
                                  visible_range_size__set)

    def _piecewise_interpolation(self, x, y, factor):
        factor += 0.5
        t = self._visible_range_size
        x = map(lambda a: t*a, x)

        # clamp after lower and upper limits
        if factor < x[0]:
            return y[0]
        elif factor > x[-1]:
            return y[-1]
        else:
            # general case: looking for the segment where factor belongs
            i = 0
            while factor > x[i+1]:
                i += 1

            # factor must be between 0.0 and 1.0
            new_factor = (factor - x[i]) / (x[i+1] - x[i])
            return maths.lerp(y[i], y[i+1], new_factor)

    def compute_x(self, index):
        w = 1.0-self._item_width
        x = [0.0, 0.4, 0.5, 0.6, 1.0]
        y = [0.0, w*0.1, w*0.5, w*0.9, w]
#        y = [e-self._widget_width/2.0 for e in y]

        return self._piecewise_interpolation(x, y, index)

    def compute_z(self, index):
        min_z = -200.0
        max_z = 200.0

        x = [0.0, 0.4, 0.5, 0.6, 1.0]
        y = [min_z, min_z/2.0, max_z, min_z/2.0, min_z]

        return self._piecewise_interpolation(x, y, index)

    def compute_opacity(self, index):
        x = [0.0, 0.4, 0.47, 0.53, 0.6, 1.0]
        y = [50, 200, 255, 255, 200, 50]

        return self._piecewise_interpolation(x, y, index)

    def _prepare_widget(self, widget):
        # center the items
        # assuming that each item has a reflection of the same height as
        # itself
        widget.y = self._item_height/4.0
        widget.size = (self._item_width, self._item_height)

    def _layout_widget(self, widget, position):
        widget.x = self.compute_x(position)
        widget.z = self.compute_z(position)
        widget.opacity = self.compute_opacity(position)

    def _selected_to_range_start(self, selected_item_index):
        half_size = (self.visible_range_size-1.0)/2.0
        prev_selected = selected_item_index
        visible_range_start = selected_item_index-half_size
        return visible_range_start

    def _create_widgets(self):
        # FIXME: do not destroy and reload everything
        for widget in self._widgets:
            self.remove(widget)
        self._widgets[:] = []

        nb_widgets = self._visible_range_size+1+2*self.preloaded
        for i in xrange(nb_widgets):
            widget = self._widget_class()
            widget.connect('clicked', self._fcl_child_clicked)
            self._widgets.append(widget)
            self.add(widget, forward_signals=False)

    # Signals support methods

    def do_key_press_event(self, viewport, event, widget):
        if event.keyval == pgm.keysyms.Left:
            self.selected_item_index -= 1
        elif event.keyval == pgm.keysyms.Right:
            self.selected_item_index += 1
        elif event.keyval == pgm.keysyms.Home:
            self.selected_item_index = 0
        elif event.keyval == pgm.keysyms.End:
            self.selected_item_index = len(self.model) - 1

    def do_scrolled(self, x, y, z, direction, time):
        if direction == pgm.SCROLL_UP:
            self.selected_item_index -= 1
        else:
            self.selected_item_index += 1

    def _fcl_child_clicked(self, drawable, x, y, z, button, time, pressure):
        if self._dragging and self._drag_accum > self.drag_threshold:
            return True

        index = self._widgets.index(drawable)
        item_index = self._item_index_from_widget_index(index)
        item = self.model[item_index]
        self.emit('item-clicked', item)
        return True

    def do_clicked(self, x, y, z, button, time, pressure):
        # always let the children handle the clicks
        return False

    def do_drag_motion(self, x, y, z, button, time, pressure):
        time_since_last = time - self._last_drag_motion
        if time_since_last > self.drag_motion_resolution:
            self._last_drag_motion = time
        else:
            return True

        relative_item_width = self._item_width
        absolute_item_width = relative_item_width*self.absolute_width
        motion = (x-self._initial[0])/absolute_item_width
        self.visible_range_start -= motion

        time_delta = time - self._initial[2]
        if time_delta != 0:
            self.speed = motion/time_delta*1000.0
            self.speed = maths.clamp(self.speed, -14.0, 14.0)

        self._initial = (x, y, time)
        self._drag_accum += abs(motion) 

        return True

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        from elisa.plugins.pigment.graph.image import Image

        widget = cls(Image)
        widget.visible = True

        model = range(15)
        def renderer(item, widget):
            return
        widget.set_renderer(renderer)
        widget.set_model(model)

        widget.position = (0.0, 0.0, 0.0)
        widget.size = (4.0, 3.0)

        def item_clicked_cb(self, item):
            print "item clicked", item

        widget.connect('item-clicked', item_clicked_cb)

        return widget


class CoverflowController(ListController):

    def nodes_setup(self):
        super(CoverflowController, self).nodes_setup()
        # FIXME: bad naming: looks like a model not a widget
        self.nodes = CoverflowList(self.node_widget, 9)
        self.widget.add(self.nodes)
        self.nodes.size = (1.0, 1.0)
        self.nodes.position = (0.0, 0.1, 0.0)
        self.nodes.visible = True
        self.nodes.focus = True

        self.titles_setup()

    def titles_setup(self):
        self.title = Text()
        self.widget.add(self.title)
        self.title.weight = pgm.TEXT_WEIGHT_BOLD
        self.title.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self.title.alignment = pgm.TEXT_ALIGN_CENTER
        self.title.bg_a = 0
        # FIXME: hardcoded font
        self.title.font_family = "Liberation Sans"
        # FIXME: hardcoded values
        self.title.size = (0.65, 0.04)
        self.title.position = ((1.0-self.title.width)/2.0, 0.82, 0.0)
        self.title.visible = True

        self.animated_title = implicit.AnimatedObject(self.title)

    def node_selected(self, widget, item, previous_item):
        self._fade_out_title()

    def set_title_from_item(self, item):
        pass

    def _fade_out_title(self):
        # do not do anything if it was already fading out
        if self.animated_title.opacity == 0.0:
            return

        self.animated_title.stop_animations()

        def faded_out(timer):
            self._fade_in_title()
            selected_item = self.model[self.nodes.selected_item_index]
            self.set_title_from_item(selected_item)

        # fade out
        self.animated_title.setup_next_animations(duration=200,
                                      transformation=implicit.DECELERATE,
                                      end_callback=faded_out)
        self.animated_title.opacity = 0.0

    def _fade_in_title(self):
        # fade in
        self.animated_title.setup_next_animations(duration=200,
                                        transformation=implicit.DECELERATE)
        self.animated_title.opacity = 255

    def handle_input(self, manager, input_event):
        if self.nothing_to_display_widget.visible:
            return False

        if input_event.value == EventValue.KEY_GO_LEFT:
            self.nodes.selected_item_index -= 1
            return True
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            self.nodes.selected_item_index += 1
            return True
        else:
            return super(CoverflowController, self).handle_input(manager, input_event)


class ImageWithReflectionCoverflowController(CoverflowController):

    """
    Coverflow controller tied to the image with reflection widget.
    """

    node_widget = ImageWithReflection

    def initialize(self):
        def create_view_mode(self):
            self._view_mode = self.view_mode()
            return self

        init_deferred = super(ImageWithReflectionCoverflowController, self).initialize()
        init_deferred.addCallback(create_view_mode)
        return init_deferred

    def node_renderer(self, item, widget):
        """
        Render a node using the common API methods defined by the
        L{elisa.plugins.poblesec.base.list.GenericListViewMode} class.
        """
        # Cancel previous deferred calls for this widget
        self.cancel_deferreds(widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        # Set the default image of the widget
        widget.image.clear()
        default_image = self._view_mode.get_default_image(item)
        self.frontend.load_from_theme(default_image, widget.image)

        # Set the real image of the widget
        theme = self.frontend.get_theme()
        image_deferred = self._view_mode.get_image(item, theme)
        if image_deferred is not None:
            def got_thumbnail(thumbnail_file):
                if thumbnail_file is not None:
                    widget.image.set_from_file(thumbnail_file)

            self.register_deferred(widget, image_deferred)
            image_deferred.addCallbacks(got_thumbnail, _failure)

    def set_title_from_item(self, item):
        """
        Set the title of the coverflow for an item using the common API methods
        defined by the L{elisa.plugins.poblesec.base.list.GenericListViewMode}
        class.
        """
        # Cancel previous deferred calls for this widget
        self.cancel_deferreds(self.title)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        def got_label(text):
            self.title.label = text

        label_deferred = self._view_mode.get_label(item)
        label_deferred.addCallbacks(got_label, _failure)
        self.register_deferred(self.title, label_deferred)


if __name__ == "__main__":
    import logging
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    lst = CoverflowList.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()
