# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.utils.cancellable_defer import CancelledError

from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.grid_horizontal import GridHorizontal

from elisa.plugins.poblesec.widgets.loadable_mixin import LoadableMixin
from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.long_loading_image import LongLoadingImage
from elisa.plugins.poblesec.base.list import ListController

import pgm
from pgm.timing import implicit


class GridItem(Widget, LoadableMixin):

    loading_attribute = 'image'

    def __init__(self):
        super(GridItem, self).__init__()

        self.image = LongLoadingImage()
        self.add(self.image)
        self.image.width, self.image.height = (0.7, 0.7)
        self.image.x = (1.0-self.image.width)/2.0
        self.image.y = (1.0-self.image.height)/2.0
        self.image.border_outer_color = (255, 255, 255, 255)
        self.image.visible = True

        self.image.image.border_width = 0.010
        self.image.image.layout = pgm.IMAGE_ZOOMED


class GridController(ListController):

    def nodes_setup(self):
        super(GridController, self).nodes_setup()
        # FIXME: default number of rows should be configurable
        rows = 3
        # default dummy value
        columns = rows

        # FIXME: bad naming: looks like a model not a widget
        self.nodes = GridHorizontal(self.node_widget, rows, columns)
        self.widget.add(self.nodes)
        # FIXME: hardcoded value
        self.nodes.width, self.nodes.height = (1.0, 0.75)
        self.nodes.x = 0.0
        # FIXME: hardcoded value
        self.nodes.y = 1.0-self.nodes.height-0.10
        self.nodes.visible = True
        self.nodes.focus = True

        # selector setup
        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.grid_selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.grid_selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.grid_selector_body')
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)

        self.nodes.set_selector(selector)

        self.titles_setup()

    def titles_setup(self):
        self.title = Text()
        self.widget.add(self.title)
        self.title.weight = pgm.TEXT_WEIGHT_BOLD
        self.title.ellipsize = pgm.TEXT_ELLIPSIZE_MIDDLE
        self.title.alignment = pgm.TEXT_ALIGN_CENTER
        self.title.bg_a = 0
        # FIXME: hardcoded font
        self.title.font_family = "Liberation Sans"
        # FIXME: hardcoded values
        self.title.width = 0.65
        self.title.height = 0.04
        self.title.x = (1.0-self.title.width)/2.0
        self.title.y = (1.0- (self.nodes.height + 0.05) -self.title.height)/2.0
        self.title.visible = True

        self.animated_title = implicit.AnimatedObject(self.title)

    def node_selected(self, widget, item, previous_item):
        self._fade_out_title()

    def set_title_from_item(self, item):
        pass

    def _fade_out_title(self):
        # do not do anything if it was already fading out
        if self.animated_title.opacity == 0.0:
            return

        self.animated_title.stop_animations()

        def faded_out(timer):
            self._fade_in_title()
            selected_item = self.model[self.nodes.selected_item_index]
            self.set_title_from_item(selected_item)

        # fade out
        self.animated_title.setup_next_animations(duration=200,
                                      transformation=implicit.DECELERATE,
                                      end_callback=faded_out)
        self.animated_title.opacity = 0.0

    def _fade_in_title(self):
        # fade in
        self.animated_title.setup_next_animations(duration=200,
                                        transformation=implicit.DECELERATE)
        self.animated_title.opacity = 255

    def prepare(self):
        aspect_ratio = self.nodes.absolute_width/self.nodes.absolute_height
        columns = self.nodes.rows*aspect_ratio
        self.nodes.visible_range_size = columns

    def handle_input(self, manager, input_event):
        if self.nothing_to_display_widget.visible:
            return False

        if input_event.value == EventValue.KEY_GO_LEFT:
            index = self.nodes.selected_item_index - self.nodes.rows
            if index >= 0:
                self.nodes.selected_item_index = index
            return True
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            index = self.nodes.selected_item_index + self.nodes.rows
            if index < len(self.model):
                self.nodes.selected_item_index = index
            return True
        if input_event.value == EventValue.KEY_GO_UP:
            if self.nodes.selected_item_index % self.nodes.rows == 0:
                # first row of the grid (top)
                return False
            else:
                self.nodes.selected_item_index -= 1
                return True
        elif input_event.value == EventValue.KEY_GO_DOWN:
            if self.nodes.selected_item_index % self.nodes.rows == self.nodes.rows-1:
                # last row of the grid (bottom)
                return False
            else:
                self.nodes.selected_item_index += 1
                return True
        else:
            return super(GridController, self).handle_input(manager, input_event)


class GridItemGridController(GridController):

    """
    Grid controller tied to the grid item widget.
    """

    node_widget = GridItem

    def initialize(self):
        def create_view_mode(self):
            self._view_mode = self.view_mode()
            return self

        init_deferred = super(GridItemGridController, self).initialize()
        init_deferred.addCallback(create_view_mode)
        return init_deferred

    def node_renderer(self, item, widget):
        """
        Render a node using the common API methods defined by the
        L{elisa.plugins.poblesec.base.list.GenericListViewMode} class.
        """
        # Cancel previous deferred calls for this widget
        self.cancel_deferreds(widget)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        # Set the default image of the widget
        widget.image.clear()
        default_image = self._view_mode.get_default_image(item)
        self.frontend.load_from_theme(default_image, widget.image.quick_image)

        # Set the real image of the widget
        theme = self.frontend.get_theme()
        image_deferred = self._view_mode.get_image(item, theme)
        if image_deferred is not None:
            def got_thumbnail(thumbnail_file):
                if thumbnail_file is not None:
                    widget.image.image.set_from_file(thumbnail_file)

            self.register_deferred(widget, image_deferred)
            image_deferred.addCallbacks(got_thumbnail, _failure)

    def set_title_from_item(self, item):
        """
        Set the title of the grid for an item using the common API methods
        defined by the L{elisa.plugins.poblesec.base.list.GenericListViewMode}
        class.
        """
        # Cancel previous deferred calls for this widget
        self.cancel_deferreds(self.title)

        def _failure(failure):
            # Swallow errbacks only when the deferred has been cancelled
            failure.trap(CancelledError)

        def got_label(text):
            self.title.label = text

        label_deferred = self._view_mode.get_label(item)
        label_deferred.addCallbacks(got_label, _failure)
        self.register_deferred(self.title, label_deferred)
