# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core import common
from elisa.core.media_uri import MediaUri

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.base.messages.device import NewDeviceDetected, \
                                               NewUnknownDevice, \
                                               DeviceRemoved

from twisted.internet import defer


class SectionMenuHierarchy(HierarchyController):

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
            path = item.controller_path
            label = item.label
            args = item.controller_args
            dfr = browser.history.append_controller(path, label, **args)


class SectionMenuViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    def get_label(self, item):
        return defer.succeed(item.label)

    def get_default_image(self, item):
        return item.icon

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        return None


class SectionMenuController(SectionMenuHierarchy, MenuItemPreviewListController):
    view_mode = SectionMenuViewMode


class HardwareListControllerMixin(SectionMenuController):

    # translate the type of the section into a "media type"
    # (types defined in elisa.plugins.base.models.file.FileModel)
    translation = {'music': 'audio',
                   'video': 'video',
                   'pictures': 'image'}

    def initialize(self):
        dfr = super(HardwareListControllerMixin, self).initialize()
        bus = common.application.bus
        bus.register(self._device_add_cb, NewDeviceDetected)
        bus.register(self._device_remove_cb, DeviceRemoved)
        return dfr

    def _device_add_cb(self, msg, sender):
        # FIXME: only file protocol supported at this time, no audio cd's nor dvd's
        link = Link()

        if msg.model.protocol == 'file':
            file_media_type = self.translation[self.media_type]
            link.controller_path = "/poblesec/%s/filesystem" % (self.media_type)
            link.controller_args = {'uri': msg.model.mount_point, 'media_type': file_media_type}
            link.icon = "elisa.plugins.poblesec.removable_device"
        elif msg.model.protocol == 'ipod' and self.media_type == 'music':
            link.controller_path = "/poblesec/music/ipod"
            link.controller_args = {'uri': msg.model.mount_point}
            link.icon = "elisa.plugins.poblesec.ipod"
        else:
            # not supported yet
            return

        for existing_link in self.model:
            uri = existing_link.controller_args.get('uri', None)
            if msg.model.mount_point and uri == msg.model.mount_point:
                # already added, can be in some cases ...
                return

        link.label = msg.model.label
        self.model.append(link)

    def _device_remove_cb(self, msg, sender):
        for link in self.model:
            uri = link.controller_args.get('uri', None)
            if None not in (uri, msg.udi) and msg.udi == uri:
                self.model.remove(link)
                return

    def clean(self):
        dfr = super(HardwareListControllerMixin, self).clean()
        bus = common.application.bus
        bus.unregister(self._device_add_cb)
        bus.unregister(self._device_remove_cb)
        return dfr

class MusicHardwareListController(HardwareListControllerMixin):
    media_type = 'music'

class VideoHardwareListController(HardwareListControllerMixin):
    media_type = 'video'

class PicturesHardwareListController(HardwareListControllerMixin):
    media_type = 'pictures'
