# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Module responsible for starting the Application
"""

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

import sys
import os
import datetime
import gobject
import platform


from elisa.core import __version__, version_info
from elisa.core.config import Config, ConfigError
from elisa.core.log import Loggable
from elisa.core.utils import exception_hook, misc
from elisa.core import bus

from elisa.core.components.message import Message

from elisa.core import config_upgrader
from elisa.core.plugin_registry import PluginRegistry
from elisa.core import input_manager, media_manager, service_manager
from elisa.core import metadata_manager, resource_manager
from elisa.core import interface_controller
from elisa.core.media_directory_helper import MediaDirectoryHelper

from twisted.internet import reactor, defer, task, threads
from twisted.web import client

# database
try:
    from elisa.extern.storm_wrapper import store
    from storm.locals import create_database
except ImportError:
    store = None


from distutils.version import LooseVersion

UPDATES_URL = "http://elisa.fluendo.com/updates/update.php?install_date=%(date)s&version=%(version)s&os=%(os)s&user_id=%(user_id)s"
CONFIG_DIR = os.path.join(os.path.expanduser('~'), ".elisa-0.5")

# We add to the configuration file name the version number in order to bypass
# the configuration upgrade system. Direct consequences:
# - custom user configuration is lost between releases
# - prevents configuration upgrade bugs
# - avoid configuration upgrade work
version_string = "_".join([str(n) for n in version_info])
CONFIG_FILE = "elisa_%s.conf" % version_string

TRANS_FILE = "data/translations.lst"
PICTURES_CACHE = os.path.join(CONFIG_DIR, 'pictures_cache')

DEFAULT_CONFIG = """\
[general]
version = '%(version)s'
install_date = '%(install_date)s'
resource_providers = ['amazon.resource_provider:AmazonResourceProvider', 'discogs.discogs_resource:DiscogsResource', 'database.media_scanner:MediaScanner', 'weather.report_provider:WeatherReportProvider', 'base.local_resource:LocalResource', 'youtube.resource_provider:YoutubeResourceProvider', 'shoutcast.shoutcast_resource:ShoutcastResource', 'flickr.resource_provider:FlickrResourceProvider', 'wmd.wmd_resource:WMDResource', 'filtered_shares.filtered_shares_resource:FilteredSharesResource', 'coherence.coherence_resource:CoherenceResource', 'coherence.upnp_resource:UpnpResource', 'smbwin32.smbwin32_resource:SmbWin32Resource', 'elisa_updater.resource_provider:UpdaterResourceProvider', 'ipod.ipod_resource:IpodResource', 'hal.hal_resource:HALResource', 'avahi.avahi_resource_provider:AvahiResourceProvider', 'daap.daap_resource_provider:DaapResourceProvider']
metadata_providers = []
service_providers = ['coherence.coherence_service:CoherenceService', 'osso.osso_service:OssoService', 'gnome.gnome_screensaver_service:GnomeScreensaverService']
input_providers = ['winremote.streamzap_input:StreamzapInput', 'lirc.lirc_input:LircInput']
frontends = ['frontend1']
# database connection string. see https://storm.canonical.com/Manual
database = '%(database_uri)s'

[frontend1]
frontend = 'pigment.pigment_frontend:PigmentFrontend'
theme = 'elisa.plugins.poblesec'
controller_path = '/poblesec'

[directories]
video = ['*default*']
pictures = ['*default*']
music = ['*default*']
"""

def setup_output_redirection(directory_path, log_filename=None):
    # See http://www.py2exe.org/index.cgi/StderrLog
    log_filename = log_filename or "elisa.log"
    elisa_log = os.path.join(directory_path, log_filename)
    sys.stdout = open(elisa_log, "w")
    sys.stderr = open(elisa_log, "w")


class ComponentsLoadedMessage(Message):
    """
    Sent when all components have been instantiated
    """

class NewElisaVersionMessage(Message):
    """
    Sent when Application detects a new Elisa version is available
    online.

    @ivar version: new version in dotted syntax (ex: 10.0.0)
    @type version: unicode
    @ivar installer_url:     URL of the new win32 installer (optional)
    @type installer_url:     unicode
    """

    def __init__(self, version, installer_url=None):
        super(NewElisaVersionMessage, self).__init__()
        self.version = version
        self.installer_url = installer_url

class BaseApplication(Loggable):
    config = None
    plugin_registry = None
    bus = None
    media_manager = None
    metadata_manager = None
    service_manager = None
    input_manager = None
    store = None

    def __init__(self, config, plugin_directories):
        # we do this here and not in initialize() as this needs to be done as
        # soon as possible
        super(BaseApplication, self).__init__()

        self._config_filename = self._get_config_filename(config)

        # if running frozen on win32, redirect stdout/stderr to $CONFIG_DIR/elisa.log
        if hasattr(sys, "frozen"):
            setup_output_redirection(os.path.dirname(self._config_filename))

        self.plugin_registry = self._create_plugin_registry(plugin_directories)
        self.config, self._install_date, self._user_id = self._load_config(config)

    def _create_plugin_registry(self, plugin_directories):
        plugin_registry = PluginRegistry(plugin_directories)
        plugin_registry.load_plugins()
        return plugin_registry

    def _get_config_filename(self, config):
        if config is not None and not isinstance(config, basestring):
            # config is already a config-object
            config_filename = config.get_filename()
        elif config is None:
            config_filename = os.path.join(CONFIG_DIR, CONFIG_FILE)
        else:
            # config is a string
            config_filename = config

        dirname = os.path.dirname(config_filename)
        if dirname and not os.path.exists(dirname):
            try:
                os.makedirs(dirname)
            except OSError, e:
                self.warning("Could not create '%s': %s" % (dirname, e))
                raise
        return config_filename

    def _load_config(self, config):
        today = datetime.date.today().isoformat()
        if config is not None and not isinstance(config, basestring):
            # config is already a config-object
            cfg = config
        else:
            config_filename = self._config_filename
            self.info("Using config file: %r", config_filename)
            default_config = DEFAULT_CONFIG % {'version': __version__,
                    'install_date': today,
                    'database_uri': 'sqlite:%s/elisa.db' % \
                    CONFIG_DIR.replace('\\','/')}

            try:
                cfg = Config(config_filename, default_config=default_config)
            except ConfigError, error:
                self.warning(error)
                raise

            if not cfg.first_load:
                # ok we might have an old config format here
                upgrader = config_upgrader.ConfigUpgrader(cfg, default_config)
                cfg = upgrader.update_for(version_info)

        install_date = cfg.get_option('install_date', section='general',
                                      default=today)
        user_id = cfg.get_option('user_id', section='general', default='')
        return (cfg, install_date, user_id)

class Application(BaseApplication):
    """ Application is the entry point of Elisa. It groups all the necessary
    elements needed for Elisa to run. It is in charge of instantiating a
    Config and a PluginRegistry. Application also provides access to
    input events and data, and holds the user interfaces. It creates
    various managers (InputManager, MediaManager...),
    an InterfaceController and a DBBackend.

    @ivar plugin_registry:      loads and manages the plugins
    @type plugin_registry:      L{elisa.core.plugin_registry.PluginRegistry}
    @ivar config:               Application's configuration file, storing options
    @type config:               L{elisa.core.config.Config}
    @ivar bus:                  DOCME
    @type bus:                  L{elisa.core.bus.Bus}
    @ivar metadata_manager:     DOCME
    @type metadata_manager:     L{elisa.core.metadata_manager.MetadataManager}
    @ivar resource_manager:     DOCME
    @type resource_manager:     L{elisa.core.resource_manager.ResourceManager}
    @ivar service_manager:      DOCME
    @type service_manager:      L{elisa.core.service_manager.ServiceManager}
    @ivar interface_controller: DOCME
    @type interface_controller: L{elisa.core.interface_controller.InterfaceController}
    @ivar input_manager:        DOCME
    @type input_manager:        L{elisa.core.input_manager.InputManager}
    @ivar media_manager:        DOCME
    @type media_manager:        L{elisa.core.media_manager.MediaManager}
    @ivar store:                the access point to the database using storm
    @type store:                L{elisa.extern.twisted_storm.store.DeferredStore}
    """

    log_category = "application"

    def __init__(self, config_filename=None, show_tracebacks=False,
                 splash=None):
        """
        @param config_filename: the config filename to use. Can be absolute
                                or relative path
        @type config_filename:  string or None to use default config file
        """
        default_path = os.path.join(CONFIG_DIR, 'plugins')
        paths = os.environ.get('ELISA_PLUGIN_PATH', default_path).split(os.path.pathsep)
        paths = [p for p in paths if p != '']
        if default_path not in paths:
            paths.insert(0, default_path)

        super(Application, self).__init__(config_filename, paths)

        # load the exception hook asap
        self.show_tracebacks = show_tracebacks
        self._load_exception_hook()

        self.debug("Creating")
        self.running = False
        self._splash = splash

        self.store = None

        self.bus = bus.Bus()
        self.service_manager = service_manager.ServiceManager()
        self.metadata_manager = metadata_manager.MetadataManager()
        self.media_manager = media_manager.MediaManager(self.metadata_manager)
        self.resource_manager = resource_manager.ResourceManager()
        self.input_manager = input_manager.InputManager()
        self.interface_controller = interface_controller.InterfaceController()

    def _check_updates(self):
        distro = misc.get_os_name()
        url = UPDATES_URL % {'date': self._install_date, 'version': __version__,
                             'os': distro, 'user_id': self._user_id}
        dfr = client.getPage(url)

        def got_result(result):
            # epr_url = result
            splitted = result.splitlines()
            if len(splitted) == 1:
                # hey, looks like new elisaweb hasn't been deployed yet.
                user_id = ''
                version = ''
                installer_url = ''
            else:
                user_id, version, installer_url = splitted

            if version and LooseVersion(version) > LooseVersion(__version__):
                msg = NewElisaVersionMessage(version, installer_url)
                self.bus.send_message(msg)

            # store user_id in config
            self.config.set_option('user_id', user_id, section='general')
            self._user_id = user_id

            # re-check for updates one day later
            reactor.callLater(86400, threads.deferToThread, self._check_updates)

        dfr.addCallback(got_result)

    def _load_exception_hook(self):
        """ Override the default system exception hook with our own
        """
        # FIXME: make this configurable
        self.logdir = None
        self.debug_level = int(self.show_tracebacks)

        sys.excepthook = self._excepthook
        # log twisted errors: Deactivated
        # from twisted.python import log
        # log.err = self.log_failure

    def _excepthook(self, *args):
        data = exception_hook.format_traceback(*args)
        path = exception_hook.write_to_logfile(data, self.logdir)
        self.warning("An Traceback occurred and got saved to %s" % path)
        self._after_hook(data)

    def _after_hook(self, data):
        if self.debug_level > 0:
            print data
            if self.debug_level == 2:
                try:
                    import pdb
                except ImportError:
                    print "pdb missing. debug shell not started"
                    return

                print "You are now in a debug shell. Application hold until" \
                      " you press 'c'!"
                pdb.set_trace()

    # Exception handling methods
    def log_traceback(self):
        """
        Log the traceback without stopping the process. This could ususally be
        used in parts, where you want to go on and log the exception.
        Example::

            try:
                component.initialize()
            except:
                # and log all the other exceptions
                path = application.log_traceback()
                self.warning("Initilize Component '%s' failed. Traceback saved at %s" % path)
            self.going_on()

        @return: path to the file, where the traceback got logged
        """
        data = exception_hook.format_traceback()
        path = exception_hook.write_to_logfile(data, self.logdir)
        self._after_hook(data)
        return path

    def log_failure(self, failure):
        """
        Log the twisted failure without re-raising the exception. Example in
        an errback::

            def errback(failure):
                path = application.log_failure(failure)
                self.warning("Connection refused. Full output at %s" % path)
                return

        @param failure: the failure to log
        @type failure:  L{twisted.python.failure.Failure}

        @return: path to the file, where the traceback got logged
        """
        data = exception_hook.format_failure(failure)
        path = exception_hook.write_to_logfile(data, self.logdir)
        self._after_hook(data)
        return path

    def initialize(self):
        """
        Load the providers for the different managers, then initialize the
        interface_controller.
        """
        # If the configuration directory does not exist, create it
        if not os.path.exists(CONFIG_DIR):
            os.makedirs(CONFIG_DIR, 0755)

        # we want to load the providers and then initialize the player and the
        # interface controller
        self.media_directories = MediaDirectoryHelper()

        def loading_providers_done(result):
            return self.interface_controller.initialize()

        managers = ( # manager, option in the config file
                    (self.resource_manager, 'resource_providers'),
                    (self.service_manager, 'service_providers'),
                    (self.metadata_manager, 'metadata_providers'),
                    (self.media_manager, 'media_providers'),
                    (self.input_manager, 'input_providers'),
                   )

        def iter_managers(managers):
            for manager, option in managers:
                self.debug("Loading providers for manager %s" % manager)
                providers = self.config.get_option(option, section='general',
                                                   default=[])
                if not providers:
                    self.debug("No providers to load for manager %s" % manager)
                    continue

                dfr = manager.load_components(providers)
                yield dfr
        
        def load_managers(db, managers):
            return task.coiterate(iter_managers(managers)).addCallback(
                    loading_providers_done)

        def interface_initialized(result):
            self.bus.send_message(ComponentsLoadedMessage())
            self._close_splash_screen()

        dfr = self.initialize_db()
        dfr.addCallback(load_managers, managers)
        dfr.addCallback(interface_initialized)
        return dfr

    def initialize_db(self):
        """
        initialize the database depending on the configuration
        """
        if store == None:
            self.warning("Could not import storm. Database disabled")
            return defer.succeed(None)

        db_string = self.config.get_option('database', section='general',
                                                    default='')

        try:
            db = create_database(db_string)
        except Exception, e:
            return defer.fail(e)

        self.store = store.DeferredStore(db, auto_reload=False)
        return self.store.start()


    def start(self):
        """ Execute the application. Start the Managers and the
        InterfaceController.
        """
        if hasattr(gobject, 'set_prgname') == True:
            gobject.set_prgname('elisa')

        self.running = True
        self.info("Starting")
        self.bus.start()
        self.input_manager.start()
        self.metadata_manager.start()
        self.resource_manager.start()
        self.media_manager.start(seconds=5)
        self.service_manager.start()

        # FIXME: use the http client for this
        threads.deferToThread(self._check_updates)

    def stop(self, stop_reactor=True):
        """Stop the application.

        @param stop_reactor:    stop the reactor after stopping the application
        @type stop_reactor:     bool
        @rtype:                 L{twisted.internet.defer.Deferred}
        """
        self._close_splash_screen()

        def interface_controller_stopped(result):
            self.info("Stopping managers")

            manager_deferreds = []
            for manager in (self.service_manager, self.metadata_manager,
                            self.media_manager, self.input_manager,
                            self.resource_manager):
                manager_deferreds.append(defer.maybeDeferred(manager.stop))
            if self.store:
                # FIXME: store is not closed properly: self.store.stop should be
                # called after the commit
                manager_deferreds.append(self.store.commit())

            dfr = defer.DeferredList(manager_deferreds)
            dfr.addCallback(managers_stopped)
            return dfr

        def managers_stopped(managers):
            self.info("Stopping reactor")
            self.bus.stop()

            self.running = False

            if self.config:
                self.config.write()

            if stop_reactor and reactor.running:
                if platform.system() == 'Windows':
                    # FIXME: extremely ugly hack to terminate Elisa under Windows
                    # fix needed:
                    # - fix the locks in gst_metadata when it is out of process
                    reactor.fireSystemEvent('shutdown')
                    from mswin32 import tools
                    tools.kill_process('elisa_fork')
                    tools.kill_process('elisa')
                    import win32process
                    win32process.ExitProcess(0)
                else:
                    reactor.stop()

        if self.running:
            # stop the interface controller, then the player registry and the
            # managers
            self.info("Stopping interface controller")
            dfr = self.interface_controller.stop()
            dfr.addCallback(interface_controller_stopped)
            return dfr

        return defer.succeed(None)

    def _close_splash_screen(self):
        if self._splash != None:
            self._splash.destroy()


