# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from twisted.trial import unittest
from elisa.core.tests.elisa_test_case import ElisaTestCase

import tempfile
import os
from elisa.core import config
from elisa.core import log

class TestConfig(ElisaTestCase):

    test_conf = "test.conf"

    test_section_dict = {'test_list':['some','list'],
                         'test_int': 0, 'test_bool_t': True,
                         'test_bool_f': False,
                         'test_str': 'foobar'}

    def tearDown(self):
        try:
            os.unlink(self.test_conf)
        except OSError:
            pass

    def load_config(self):
        """ Load the config from test.conf file
        """
        c = config.Config(self.test_conf)
        return c

    def load_default_config(self):
        c = config.Config()
        return c

    def set_test_section(self, c):
        """ Fill a 'tests' section in the config
        """
        c.set_section('tests')
        c.set_option('test_list',['some','list'], section='tests')
        c.set_option('test_int', 0, section='tests')
        c.set_option('test_bool_t', True, section='tests')
        c.set_option('test_bool_f', False, section='tests')
        c.set_option('test_str', 'foobar', section='tests')

    def test_malformed_config(self):
        """ Create a dumb config file containing formatting errors and
        check errors are reported.
        """
        data="""\
[general]

foo=['bar'
"""
        f = open('malformed.conf','w')
        f.write(data)
        f.close()

        self.assertRaises(Exception, config.Config, 'malformed.conf')
        os.unlink('malformed.conf')

    def test_import_export(self):
        """ Load test config, write it, load dumped config and compare
        the two instances.
        """
        c1 = self.load_config()
        self.set_test_section(c1)
        c1.write()

        c2 = self.load_config()
        self.assertEqual(c1.as_dict(), c2.as_dict())

    def test_list_option(self):
        """ Test tuple and list option types
        """
        c = self.load_config()

        self.assertEqual(c.get_option('resolution'), None)

        self.set_test_section(c)
        self.assertEqual(c.get_option('test_list', section='tests'),
                         ['some','list'])
        c.write()

    def test_int_option(self):
        """ Test integer option type
        """
        c = self.load_config()
        self.set_test_section(c)
        self.assertEqual(c.get_option('test_int', section='tests'), 0)

    def test_boolean_option(self):
        c = self.load_config()
        self.set_test_section(c)
        self.assertEqual(c.get_option('test_bool_t', section='tests'), True)
        self.assertEqual(c.get_option('test_bool_f', section='tests'), False)


    def test_string_option(self):
        """ Test string option type
        """
        c = self.load_config()
        self.set_test_section(c)
        self.assertEqual(c.get_option('test_str', section='tests'), 'foobar')

    def test_section(self):
        """ Check the whole tests section dumped as a dictionnary
        """
        c = self.load_config()
        self.set_test_section(c)
        self.assertEqual(c.get_section('tests'),
                         self.test_section_dict)

    def test_del_section(self):
        """ Test section removal
        """
        c = self.load_config()
        self.set_test_section(c)

        c.del_section('tests')
        self.assert_('tests' not in c.as_dict().keys())

    def test_del_option(self):
        """ Test option removal
        """
        c = self.load_config()
        self.set_test_section(c)

        for opt_name in  self.test_section_dict.keys():
            c.del_option(opt_name,'tests')
            self.assertEqual(c.get_option(opt_name,'tests'), None)

    def test_get_config_dir(self):
        c = self.load_config()
        self.assertEqual(c.get_config_dir(), os.getcwd())

        fd, filename = tempfile.mkstemp()
        tmp_config = config.Config(self.test_conf)
        tmp_config.write(filename)

        cfg = config.Config(filename)
        self.assertEqual(cfg.get_config_dir(), tempfile.gettempdir())

        os.unlink(filename)
