# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Test the hal Resource provider
"""

from twisted.trial.unittest import TestCase, SkipTest
from twisted.internet import defer

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.plugins.hal.hal_resource import HALResource

class SimpleDevice(dict):

    def PropertyExists(self, prop):
        return prop in self

    def QueryCapability(self, cap):
        return cap in self

    def GetProperty(self, prop):
        return self[prop]

    def GetAllProperties(self):
        return self

class HalMixin(object):
    """
    Base class that sets up the resource provider and patches the application
    and the bus
    """

    def setUp(self):
        def created(resource):
            self.resource = resource

        # patch the application:
        class Dummy(object):pass

        class DummyBus(object):
            def __init__(self):
                self.messages = []
            def register(*args):
                pass
            def send_message(self, msg, sender):
                self.messages.append( msg )
        
        self.bus = DummyBus()

        self.app = common.application
        common.application = Dummy()
        common.application.bus = self.bus

        return HALResource.create({}).addCallback(created)

    def tearDown(self):
        common.application = self.app

class TestCapabilityCheck(HalMixin, TestCase):

    def test_added_volume(self):
        """
        test whether it is calling the new_volume method when a new volume is
        added
        """
        def get_device(*args):
            return SimpleDevice({'volume': True, u'info.parent' : None,
                                 'volume.disc.is_videodvd': True})

        def new_volume_cb(*args):
            new_volume_cb.values = args

        self.resource._get_device_with_udi = get_device

        self.resource._device_added_callback('jop')

        model = self.bus.messages[0].model
        self.assertEquals(model.protocol, 'dvd')

    def test_added_videodevice(self):
        raise SkipTest("Not yet implemented")
            
class TestVolumeParsing(HalMixin, TestCase):
    """
    Test the volume regonition system
    """

    def setUp(self):
        self.devices = {}
        def overwrite_property_getting(result):

            def get_property(device, prop_name, default=None):
                return device.get(prop_name, default)

            def get_device_with_udi(udi):
                return self.devices[udi]
                
            self.resource._get_property = get_property
            self.resource._get_device_with_udi = get_device_with_udi

        return super(TestVolumeParsing,
                self).setUp().addCallback(overwrite_property_getting)
    
    def test_dvd(self):
        """
        test that it regonizes dvds properly
        """
        self.devices = {'parent' : SimpleDevice()}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.disc.is_videodvd'] = True
        this_device['block.device'] = '/dev/dvd'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('testdvd', this_device)

        self.assertEquals(model.device, '/dev/dvd')
        self.assertEquals(model.protocol, 'dvd')
        self.assertEquals(model.udi, 'testdvd')

    def test_audiocd(self):
        """
        test that it regonizes audiocds properly
        """
        self.devices = {'parent' : SimpleDevice()}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.disc.has_audio'] = True
        this_device['block.device'] = '/dev/audiocd'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('testaudio', this_device)

        self.assertEquals(model.device, '/dev/audiocd')
        self.assertEquals(model.protocol, 'cdda')
        self.assertEquals(model.udi, 'testaudio')

    def test_disc_partition(self):
        """
        test that it regonizes partions properly
        """
        self.devices = {'parent' : SimpleDevice()}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.disc.is_partition'] = True
        this_device['block.device'] = 'sda1'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('harddrive', this_device)

        self.assertEquals(model.device, 'sda1')
        self.assertEquals(model.protocol, 'file')
        self.assertEquals(model.udi, 'harddrive')
        self.assertEquals(model.label, '')
        self.assertEquals(model.mount_point, None)

    def test_volume_partition(self):
        """
        test that it regonizes volume partitions 
        """
        self.devices = {'parent' : SimpleDevice()}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.is_partition'] = True
        this_device['volume.is_mounted'] = False
        this_device['block.device'] = 'sda2'
        this_device['volume.label'] = 'Wingo'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('harddrive', this_device)

        self.assertEquals(model.device, 'sda2')
        self.assertEquals(model.protocol, 'file')
        self.assertEquals(model.udi, 'harddrive')
        self.assertEquals(model.label, 'Wingo')
        self.assertEquals(model.mount_point, None)

    def test_has_data(self):
        """
        test that it regonizes data on cds or dvds
        """
        self.devices = {'parent' : SimpleDevice()}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.disc.has_data'] = True
        this_device['volume.is_mounted'] = True
        this_device['block.device'] = 'sda3'
        this_device['volume.mount_point'] = '/mnt'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('harddrive', this_device)

        self.assertEquals(model.device, 'sda3')
        self.assertEquals(model.protocol, 'file')
        self.assertEquals(model.udi, 'harddrive')
        self.assertEquals(model.label, '')
        self.assertEquals(model.mount_point, MediaUri('file:///mnt/'))

    def test_removable(self):
        """
        test that it regonizes even only removable is set
        """
        self.devices = {'parent' : SimpleDevice({'storage.removable':True})}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['block.device'] = 'sda4'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('hdd', this_device)

        self.assertEquals(model.device, 'sda4')
        self.assertEquals(model.protocol, 'file')
        self.assertEquals(model.udi, 'hdd')
        self.assertEquals(model.label, '')
        self.assertEquals(model.mount_point, None)

    def test_ignore(self):
        """
        test that ignore works properly
        """
        self.devices = {'parent' : SimpleDevice({'storage.removable':True})}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['volume.disc.has_data'] = True
        this_device['volume.ignore'] = True
        this_device['block.device'] = 'sda4'
        this_device['info.parent'] = 'parent'

        model = self.resource._create_model_for_device('hdd', this_device)

        self.assertEquals(model, None)

    def test_general_portable_player(self):
        """
        test that regonizing a portable player works properly
        """
        self.devices = {'parent' :
                SimpleDevice({'portable_audio_player.type':'iAudio',
                              'portable_audio_player.access_method.protocols': ['file',]})}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['block.device'] = 'sda5'
        this_device['info.parent'] = 'parent'
        this_device['volume.is_mounted'] = True
        this_device['volume.label'] = 'M3'
        this_device['volume.mount_point'] = '/mnt/3'

        model = self.resource._create_model_for_device('iaudio', this_device)

        self.assertEquals(model.device, 'sda5')
        self.assertEquals(model.protocol, 'portable_player')
        self.assertEquals(model.udi, 'iaudio')
        self.assertEquals(model.label, 'M3')
        self.assertEquals(model.mount_point, MediaUri('file:///mnt/3/'))

    def test_ipod(self):
        """
        test that regonizing a portable player works properly
        """
        self.devices = {'parent' :
                SimpleDevice({'portable_audio_player.type':'IPod',
                              'portable_audio_player.access_method.protocols': ['file','ipod']})}

        this_device = SimpleDevice()
        this_device['volume'] = True
        this_device['block.device'] = '/dev/ip'
        this_device['info.parent'] = 'parent'
        this_device['volume.label'] = 'iPhone'

        model = self.resource._create_model_for_device('udi_ipod', this_device)

        self.assertEquals(model.device, '/dev/ip')
        self.assertEquals(model.protocol, 'ipod')
        self.assertEquals(model.udi, 'udi_ipod')



class TestGet(HalMixin, TestCase):

    def test_hal_get(self):
        """
        test the get methods
        """
        raise SkipTest("Not yet implemented")

