/************************************************************************
 *
 *  UCBWrapper.java
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright: 2002-2007 by Henrik Just
 *
 *  All Rights Reserved.
 * 
 *  Version 0.4.1e (2007-02-20)
 *
 */
 
package writer2latex.filter;

import com.sun.star.io.XActiveDataSink; 
import com.sun.star.io.XInputStream;

import com.sun.star.lang.XMultiServiceFactory;

import com.sun.star.ucb.Command; 
import com.sun.star.ucb.InsertCommandArgument; 
import com.sun.star.ucb.OpenCommandArgument2; 
import com.sun.star.ucb.OpenMode; 
import com.sun.star.ucb.XCommandProcessor; 
import com.sun.star.ucb.XContent;
import com.sun.star.ucb.XContentIdentifier;
import com.sun.star.ucb.XContentIdentifierFactory;
import com.sun.star.ucb.XContentProvider;

import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XInterface;

class UCBActiveDataSink implements XActiveDataSink {
    XInputStream m_aStream = null;
    public UCBActiveDataSink() {
        super();
    }

    public void setInputStream(XInputStream aStream) {
        m_aStream = aStream;                
    }

    public XInputStream getInputStream() {
        return m_aStream;        
    }
} 

/** <p>This class wraps some uses of the Universal Content Broker into user
 * friendly methods.
 */
public class UCBWrapper {
    // The methods in this class are based on the examples in the dev guide to OOo
	
    private static XInterface xUCB;
	
    // Helper method to execute and ucb command
    Object executeCommand(XContent xContent, String aCommandName, Object aArgument)
        throws com.sun.star.ucb.CommandAbortedException, com.sun.star.uno.Exception {
        // Obtain command processor interface from given content.
        XCommandProcessor xCmdProcessor = (XCommandProcessor)UnoRuntime.queryInterface( 
            XCommandProcessor.class, xContent);
        // Assemble command to execute.
        Command aCommand  = new Command();
        aCommand.Name     = aCommandName;
        aCommand.Handle   = -1; // not available
        aCommand.Argument = aArgument;
        // Note: throws CommandAbortedException and Exception since
        // we pass null for the XCommandEnvironment parameter
        return xCmdProcessor.execute(aCommand, 0, null);
    } 
	
    /** <p>The constructor initializes the UCB using a supplied MultiServiceFactory</p>
     *  @param xMSF a multiservicefactory supporting the UniversalContentBroker service
     */
    public UCBWrapper(XMultiServiceFactory xMSF) { 
        try {
            // Supply configuration to use for this UCB instance
            String[] keys = new String[2];
            keys[0] = "Local";
            keys[1] = "Office";
            xUCB = (XInterface) xMSF.createInstanceWithArguments(
               "com.sun.star.ucb.UniversalContentBroker", keys );
        } catch (com.sun.star.uno.Exception e) {
            System.err.println("Failed to get UCB object");
            xUCB = null;
        }
    }
	
    /** <p>Use the ucb to write some data to an url. 
     * Fail silently if anything goes wrong.
     * @param xData the XInputStream to get the data from
     * @param sUrl the url to write the data to
     */
    public void write(XInputStream xData, String sUrl) {
        // Obtain the required ucb interfaces
        XContentIdentifierFactory xIdFactory = (XContentIdentifierFactory)UnoRuntime.queryInterface( 
            XContentIdentifierFactory.class, xUCB);
        XContentProvider xProvider = (XContentProvider)UnoRuntime.queryInterface(
            XContentProvider.class, xUCB);
			
        // Get Content object for the given url
        XContentIdentifier xId = xIdFactory.createContentIdentifier(sUrl);
        XContent xContent = null;
        try {
            xContent = xProvider.queryContent(xId);
        } catch (com.sun.star.ucb.IllegalIdentifierException iiex) {
            // fail silently
        }
		
        // Fill argument structure
        InsertCommandArgument aArg = new InsertCommandArgument();
        aArg.Data = xData;
        aArg.ReplaceExisting = true;

        // Execute insert command using a Command Processor
        try {
            executeCommand(xContent,"insert",aArg);
        }
        catch (com.sun.star.ucb.CommandAbortedException e) {
            // do nothing
        }
        catch (com.sun.star.uno.Exception e) {
            // do nothing
        }
        catch (Throwable e) {
            // do noting
        }
		
    }
	
    /** <p>Use the ucb to read some data from an url. 
     * Fail silently if anything goes wrong.
     * @param sUrl the url to read the data from
     * @return an XInputStream from which the data can be read after reading
     */
    public XInputStream read(String sUrl) {
        // Obtain the required ucb interfaces
        XContentIdentifierFactory xIdFactory = (XContentIdentifierFactory)UnoRuntime.queryInterface( 
            XContentIdentifierFactory.class, xUCB);
        XContentProvider xProvider = (XContentProvider)UnoRuntime.queryInterface(
            XContentProvider.class, xUCB);
			
        // Get Content object for the given url
        XContentIdentifier xId = xIdFactory.createContentIdentifier(sUrl);
        XContent xContent = null;
        try {
            xContent = xProvider.queryContent(xId);
        } catch (com.sun.star.ucb.IllegalIdentifierException iiex) {
            // fail silently
        }
		
        // Fill argument structure
        OpenCommandArgument2 aArg = new OpenCommandArgument2();
        aArg.Mode = OpenMode.DOCUMENT;
        aArg.Priority = 32768;        // Ignored by most implementations
        XActiveDataSink xDataSink = new UCBActiveDataSink();
        aArg.Sink = xDataSink;
        try {
            // Execute command "open". The implementation of the command will
            // supply an XInputStream implementation to the data sink,
            executeCommand(xContent, "open", aArg);
        }
        catch (com.sun.star.ucb.CommandAbortedException e) {
            // do nothing
        }
        catch (com.sun.star.uno.Exception e) {
            // do nothing
        }

        // Return the input stream supplied by the open command implementation.
        return xDataSink.getInputStream();
    }
	
    

}



