# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# validators.py - Validation functions

import re

class ValidationError(Exception): pass

def nospaces(val):
    """Check for whitespace"""
    if re.search(r'\s', val):
        raise ValidationError(_("Found whitespace in '%s'") % val)

def hex(val):
    """Check that the value is a correctly formatted hexadecimal number
       without prefix"""
    if not re.match(r'^[a-fA-F0-9]*$', val):
        raise ValidationError(_("'%s' is not a hexadecimal number") % val)

def atleastone(val):
    """Check for existence of data"""
    if not len(val):
        raise ValidationError(_("At least one item required"))

url_re = re.compile('^https?://[a-z][-a-z0-9.+/%&?]*$', re.I)
def url(val):
    """Basic URL sanity check"""
    if not url_re.match(val):
        raise ValidationError(_("'%s' is not a valid url") % val)

mail_re = re.compile(r'^[-a-z0-9+%._]+@[-a-z0-9.]+$', re.I)
def mailaddress(val):
    """Basic mail address validity check"""
    if not mail_re.match(val):
        raise ValidationError(_("'%s' is not a valid mailaddress") % val)

def version(val):
    """Version number (formatted like a.b.c.d...) check"""
    if not re.match(r'^\d+(\.\d+)*$', val):
        raise ValidationError(_("'%s' is not a valid version number")  % val)

shellsafe_re = re.compile('^[-a-z0-9_+=.^%@:/ ]*$', re.I)
def shellsafe(val):
    """Check for shell metacharacters"""
    if not shellsafe_re.match(val):
        raise ValidationError(_("'%s' contains characters that can be interpreted by the shell"))

def commaseparated(func):
    """Check for commaseparated data"""
    def inner(val):
        for v in val.split(','):
            func(v)
    return inner

def blank_ok(func):
    def inner(val):
        if val:
            func(val)
    return inner
