
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: environment.c 1538 2006-11-13 15:19:30Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <mntent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "download.h"
#include "desktop_entry.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "oxine.h"

extern oxine_t *oxine;

static l_list_t *disc_list;

static char *dir_home = NULL;
static char *dir_oxine = NULL;
static char *dir_oxine_cache = NULL;
static char *dir_oxine_tmp = NULL;
static char *dir_oxine_playlists = NULL;

static char *file_config = NULL;
static char *file_mediamarks_music = NULL;
static char *file_mediamarks_video = NULL;
static char *file_mediamarks_media = NULL;
static char *file_lirc_config = NULL;
static char *file_jsw_config = NULL;
static char *file_dvb_channels = NULL;
static char *file_favorites = NULL;
static char *file_playlist_rw = NULL;

/*
 * This converts a relative path to an absolute one.
 */
static char *
relative_to_absolute (const char *filename)
{
    l_list_t *list = l_list_new ();

    /* We add all parts of our current directory to the list. */
    if (filename[0] != '/') {
        char _directory[1024];
        getcwd (_directory, 1024);

        char *saveptr;
        char *d = strtok_r (_directory, "/", &saveptr);
        while (d) {
            l_list_append (list, d);
            d = strtok_r (NULL, "/", &saveptr);
        }
    }

    /* We add all parts of the filename to the list. */
    {
        char _filename[1024];
        strncpy (_filename, filename, 1023);

        char *saveptr;
        char *f = strtok_r (_filename, "/", &saveptr);
        while (f) {
            if (strcmp (".", f) == 0) {
                // do nothing
            }

            else if (strcmp ("..", f) == 0) {
                l_list_remove (list, l_list_last (list));
            }

            else {
                l_list_append (list, f);
            }

            f = strtok_r (NULL, "/", &saveptr);
        }
    }

    char _result[1024];
    char *p = _result;
    char *s = (char *) l_list_first (list);
    while (s) {
        *p++ = '/';
        char *q = s;
        while (*q != '\0')
            *p++ = *q++;
        s = (char *) l_list_next (list, s);
    }
    *p = '\0';

    l_list_free (list, NULL);

    return ho_strdup (_result);
}

char *
resolve_softlink (const char *mrl)
{
    struct stat filestat;

    /* We want to make sure we're handling an absolute path without any ups
     * and downs in between. */
    char *abs_mrl = relative_to_absolute (mrl);

    if (lstat (abs_mrl, &filestat) == -1) {
        error (_("Unable to get status for file '%s': %s."),
               abs_mrl, strerror (errno));
        return abs_mrl;
    }

    if (!S_ISLNK (filestat.st_mode)) {
        return abs_mrl;
    }

    char link_target[1024];
    int len = readlink (abs_mrl, link_target, 1024);
    if (len == -1) {
        error (_("Unable to read link '%s' file: %s."),
               abs_mrl, strerror (errno));
        return abs_mrl;
    }

    if (link_target[len - 1] == '/')
        link_target[len - 1] = 0;
    else
        link_target[len] = 0;

    if (link_target[0] != '/') {
        char *dev = ho_strdup (abs_mrl);
        char *link = ho_strdup (link_target);
        snprintf (link_target, 1024, "%s/%s", dirname (dev), link);
        ho_free (dev);
        ho_free (link);
    }
#ifdef DEBUG
    char *abs_target = relative_to_absolute (link_target);
    debug ("found symbolic link: %s -> %s", abs_mrl, abs_target);
    ho_free (abs_target);
#endif
    ho_free (abs_mrl);

    return resolve_softlink (link_target);
}

static int
disc_list_contains (const char *device)
{
    device_entry_t *entry = l_list_first (disc_list);
    while (entry) {
        if (strcmp (entry->device, device) == 0)
            return 1;
        entry = l_list_next (disc_list, entry);
    }
    return 0;
}

static device_entry_t *
disc_new (const char *device, const char *mountpoint)
{
    assert (device);

    device_entry_t *disc = ho_new (device_entry_t);

    disc->device = ho_strdup (device);
    if (mountpoint)
        disc->mountpoint = ho_strdup (mountpoint);
    else
        disc->mountpoint = NULL;
    disc->is_type_cd = false;
    disc->is_disc_in = false;

    return disc;
}

/*
 * Looks for the device in /etc/fstab.
 * Returns true if an entry for the device could be
 * found.
 */
static bool
scan_fstab_for_mountpoint (const char *device, bool is_type_cd)
{
    FILE *fstab = fopen ("/etc/fstab", "r");
    if (!fstab)
        return false;

    /* We want to make sure we're handling an absolute path without any ups
     * and downs in between. */
    char *abs_device = relative_to_absolute (device);

    struct mntent *entry = getmntent (fstab);
    while (entry) {
        if ((strcmp (abs_device, entry->mnt_fsname) == 0)
            && !disc_list_contains (abs_device)) {
            info (_("Found a device we are interested in:"));
            info (_("             device: %s"), entry->mnt_fsname);
            info (_("         mountpoint: %s"), entry->mnt_dir);
            info (_("         filesystem: %s"), entry->mnt_type);

            device_entry_t *new_entry = disc_new (entry->mnt_fsname,
                                                  entry->mnt_dir);

            new_entry->is_type_cd = is_type_cd;
            if (strstr (entry->mnt_type, "iso9660")
                || strstr (entry->mnt_type, "udf")) {
                new_entry->is_type_cd = true;
            }

            debug ("         is type CD: %s",
                   new_entry->is_type_cd ? "true" : "false");

            l_list_append (disc_list, new_entry);

            endmntent (fstab);

            ho_free (abs_device);
            return true;
        }
        entry = getmntent (fstab);
    }
    endmntent (fstab);

    ho_free (abs_device);
    return false;
}

static void
evaluate_device (const char *device, bool is_type_cd)
{
    /* We want to make sure we're handling an absolute path without any ups
     * and downs in between. */
    char *abs_device = relative_to_absolute (device);
    debug ("evaluating %s", abs_device);

    /* If the disc list already contains this device, we can leave. */
    if (disc_list_contains (abs_device))
        goto out_free;

    /* We try to guess if this device is a CDROM drive. */
    if (strstr (abs_device, "cdrom")
        || strstr (abs_device, "cdrw")
        || strstr (abs_device, "dvd")
        || strstr (abs_device, "dvdrw"))
        is_type_cd = true;

    /* If the device does not exist, we only add it if it's in fstab */
    if (access (abs_device, F_OK) != 0) {
        scan_fstab_for_mountpoint (abs_device, is_type_cd);
        goto out_free;
    }

    struct stat filestat;
    if (lstat (abs_device, &filestat) == -1) {
        error (_("Unable to get status for file '%s': %s."),
               abs_device, strerror (errno));
        goto out_free;
    }

    if (S_ISLNK (filestat.st_mode)) {
        if (!scan_fstab_for_mountpoint (abs_device, is_type_cd)) {
            char *link_target = resolve_softlink (abs_device);
            evaluate_device (link_target, is_type_cd);
            ho_free (link_target);
        }
    }

    else if (S_ISBLK (filestat.st_mode)) {
        debug (" found block device: %s", abs_device);

        if (!scan_fstab_for_mountpoint (abs_device, is_type_cd)) {
            device_entry_t *new_entry = disc_new (abs_device, NULL);
            new_entry->is_type_cd = is_type_cd;
            l_list_append (disc_list, new_entry);
        }
    }

  out_free:
    ho_free (abs_device);
}


static void
autofind_discs (void)
{
    static const char *confignames[] = {
        "media.dvd.device",
        "media.vcd.device",
        "media.audio_cd.device",
        NULL
    };

#ifdef HAVE_DISC_POLLING
    static const char *devices[] = {
        "/dev/cdrom", "/dev/cdrom1", "/dev/cdrom2",
        "/dev/cdrw", "/dev/cdrw1", "/dev/cdrw2",
        "/dev/dvd", "/dev/dvd1", "/dev/dvd2",
        "/dev/dvdrw", "/dev/dvdrw1", "/dev/dvdrw2",
        NULL
    };
#endif

    disc_list = l_list_new ();

    /* If polling is disabled we add all the devices we can find in the
     * configuration file. */
    int i = 0;
    while (confignames[i]) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, confignames[i++], &centry);

        if (centry.str_value && (strlen (centry.str_value) > 0)) {
            evaluate_device (centry.str_value, true);
        }
    }

#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
        return;

    debug ("Start looking for devices (e.g. discs, cdrom)...");

    /* First we look for our standard list of devices. */
    i = 0;
    while (devices[i]) {
        evaluate_device (devices[i++], true);
    }

    /* Next we look for extra devices supplied by the user. */
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "misc.polling.devices", &centry);
    if (centry.str_value && strlen (centry.str_value) > 0) {
        char *saveptr;
        char *devices = ho_strdup (centry.str_value);
        char *device = strtok_r (devices, ";", &saveptr);
        while (device) {
            evaluate_device (device, false);
            device = strtok_r (NULL, ";", &saveptr);
        }
        ho_free (devices);
    }
#endif
}


device_entry_t *
devicelist_first (void)
{
    if (!disc_list) {
        autofind_discs ();

        debug ("Found the following devices (e.g. discs, cdrom)...");
        device_entry_t *entry = devicelist_first ();
        while (entry) {
            debug ("device: %s, mountpoint: %s, is_cd: %s",
                   entry->device,
                   entry->mountpoint ? entry->mountpoint : "unknown",
                   entry->is_type_cd ? "true" : "false");
            entry = devicelist_next (entry);
        }
    }

    return (device_entry_t *) l_list_first (disc_list);
}


device_entry_t *
devicelist_next (device_entry_t * device_entry)
{
    assert (disc_list);
    assert (device_entry);

    return (device_entry_t *) l_list_next (disc_list, device_entry);
}


const char *
get_dir_home (void)
{
    if (!dir_home) {
        char *tmp = getenv ("HOME");
        if (tmp && (strlen (tmp) > 0)
            && (strcmp (tmp, "/") != 0)
            && (access (tmp, X_OK | W_OK | R_OK) == 0)) {
            dir_home = ho_strdup (tmp);
            if (dir_home[strlen (tmp) - 1] == '/')
                dir_home[strlen (tmp) - 1] = '\0';
        } else {
            error (_("Unable to find usable home directory, "
                     "using '/tmp'."));
            dir_home = ho_strdup ("/tmp");
        }
    }

    return (const char *) dir_home;
}


const char *
get_dir_oxine (void)
{
    if (!dir_oxine) {
        dir_oxine = ho_strdup_printf ("%s/.oxine", get_dir_home ());

        mkdir (dir_oxine, 0700);
    }

    return (const char *) dir_oxine;
}


const char *
get_dir_oxine_playlists (void)
{
    if (!dir_oxine_playlists) {
        dir_oxine_playlists =
            ho_strdup_printf ("%s/playlists", get_dir_oxine ());

        mkdir (dir_oxine_playlists, 0700);
    }

    return (const char *) dir_oxine_playlists;
}


const char *
get_dir_oxine_cache (void)
{
    if (!dir_oxine_cache) {
        dir_oxine_cache = ho_strdup_printf ("%s/cache", get_dir_oxine ());

        mkdir (dir_oxine_cache, 0700);
    }

    return (const char *) dir_oxine_cache;
}


const char *
get_dir_oxine_tmp (void)
{
    if (!dir_oxine_tmp) {
        dir_oxine_tmp = ho_strdup ("/tmp/.oxine");

        mkdir (dir_oxine_tmp, 0700);
    }

    return (const char *) dir_oxine_tmp;
}


const char *
get_file_config (void)
{
    if (!file_config) {
        file_config = ho_strdup_printf ("%s/config", get_dir_oxine ());
    }

    return (const char *) file_config;
}


const char *
get_file_mediamarks_music (void)
{
    if (!file_mediamarks_music) {
        file_mediamarks_music =
            ho_strdup_printf ("%s/mediamarks_music.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_music;
}


const char *
get_file_mediamarks_video (void)
{
    if (!file_mediamarks_video) {
        file_mediamarks_video =
            ho_strdup_printf ("%s/mediamarks_video.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_video;
}


const char *
get_file_mediamarks_media (void)
{
    if (!file_mediamarks_media) {
        file_mediamarks_media =
            ho_strdup_printf ("%s/mediamarks_media.xml", get_dir_oxine ());
    }

    return (const char *) file_mediamarks_media;
}


const char *
get_file_jsw_config (void)
{
    if (!file_jsw_config) {
        file_jsw_config = ho_strdup_printf ("%s/joystick", get_dir_oxine ());
    }

    return (const char *) file_jsw_config;
}


const char *
get_file_lirc_config (void)
{
    if (!file_lirc_config) {
        file_lirc_config = ho_strdup_printf ("%s/lircrc", get_dir_oxine ());
    }

    return (const char *) file_lirc_config;
}


const char *
get_file_dvb_channels (void)
{
    if (!file_dvb_channels) {
        file_dvb_channels =
            ho_strdup_printf ("%s/.xine/channels.conf", get_dir_home ());
    }

    return (const char *) file_dvb_channels;
}


const char *
get_file_favorites (void)
{
    if (!file_favorites) {
        file_favorites =
            ho_strdup_printf ("%s/favorites.xml", get_dir_oxine ());
    }

    return (const char *) file_favorites;
}


const char *
get_file_playlist_rw (void)
{
    if (!file_playlist_rw) {
        file_playlist_rw =
            ho_strdup_printf ("%s/playlist_rw.xml", get_dir_oxine ());
    }

    return (const char *) file_playlist_rw;
}


int
file_is_newer (const char *mrl1, const char *mrl2)
{
    struct stat stat1;
    if (stat (mrl1, &stat1) == -1) {
        error (_("Unable to get status for file '%s': %s."), mrl1,
               strerror (errno));
        return 0;
    }

    struct stat stat2;
    if (stat (mrl2, &stat2) == -1) {
        error (_("Unable to get status for file '%s': %s."), mrl2,
               strerror (errno));
        return 1;
    }

    return (stat1.st_mtime > stat2.st_mtime);
}


int
mkdir_recursive (const char *mrl, mode_t mode)
{
    if (access (mrl, R_OK) != 0) {
        if (mkdir (mrl, mode) == -1) {
            char *dirn = ho_strdup (mrl);
            mkdir_recursive (dirname (dirn), mode);
            ho_free (dirn);

            if (mkdir (mrl, mode) == -1) {
                error (_("Could not create '%s': %s!"), mrl,
                       strerror (errno));
                return 0;
            }
        }
    }
    return 1;
}


/*
 * This first converts a relative path to an absolute 
 * one and then prepends 'file://' to it.
 */
static char *
filename_to_uri (const char *filename)
{
    char *_filename = relative_to_absolute (filename);

    char *_result = ho_malloc (strlen (_filename) + 8);
    sprintf (_result, "%s%s", "file://", _filename);

    ho_free (_filename);

    return _result;
}


typedef enum {
    UNSAFE_ALL = 0x1,           /* Escape all unsafe characters   */
    UNSAFE_ALLOW_PLUS = 0x2,    /* Allows '+'  */
    UNSAFE_PATH = 0x8,          /* Allows '/', '&', '=', ':', '@', '+', '$' and ',' */
    UNSAFE_HOST = 0x10,         /* Allows '/' and ':' and '@' */
    UNSAFE_SLASHES = 0x20       /* Allows all characters except for '/' and '%' */
} UnsafeCharacterSet;

static const char acceptable[96] = {
    /* A table of the ASCII chars from space (32) to DEL (127) */
    /*      !    "    #    $    %    &    '    (    )    *    +    ,    -    .    / */
    0x00, 0x3F, 0x20, 0x20, 0x28, 0x00, 0x2C, 0x3F,
    0x3F, 0x3F, 0x3F, 0x2A, 0x28, 0x3F, 0x3F, 0x1C,
    /* 0    1    2    3    4    5    6    7    8    9    :    ;    <    =    >    ? */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x38, 0x20, 0x20, 0x2C, 0x20, 0x20,
    /* @    A    B    C    D    E    F    G    H    I    J    K    L    M    N    O */
    0x38, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* P    Q    R    S    T    U    V    W    X    Y    Z    [    \    ]    ^    _ */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x20, 0x3F,
    /* `    a    b    c    d    e    f    g    h    i    j    k    l    m    n    o */
    0x20, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* p    q    r    s    t    u    v    w    x    y    z    {    |    }    ~  DEL */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x3F, 0x20
};

static const char hex[16] = "0123456789ABCDEF";

/* FIXME: The original also escapes the ? as this can be part of a valid http URL I
 *        decided to leave it. I'm not sure if this is a good idea though. */
//#define ACCEPTABLE_URI(a) ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask))
#define ACCEPTABLE_URI(a) ((a) == '?' || ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask)))

char *
filename_escape_to_uri (const char *filename)
{
    if (!filename)
        return NULL;

    // We convert a normal filename to an URI, everything else we leave as it
    // is. This could be done a lot better as this only converts an existing
    // file to an URI (but we dont care about nonexistant files).
    char *_filename;
    if (access (filename, F_OK) == 0)
        _filename = filename_to_uri (filename);
    else
        _filename = ho_strdup (filename);

    // Watchit:
    // The following code was mostly taken from glib (g_filename_to_uri).
    UnsafeCharacterSet use_mask = UNSAFE_PATH;
    int unacceptable = 0;

    const char *p;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c))
            unacceptable++;
    }

    char *result = ho_malloc (strlen (_filename) + 2 * unacceptable + 1);

    char *q = result;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c)) {
            *q++ = '%';
            *q++ = hex[c >> 4];
            *q++ = hex[c & 15];
        } else {
            *q++ = *p;
        }
    }
    *q = '\0';

    ho_free (_filename);

    //debug ("%s -> %s", filename, result);

    return result;
}

#define UNACCEPTABLE_SHELL(a) ((a == ' ') || (a == '&') \
                              || (a == '(') || (a == ')'))

char *
filename_escape_for_shell (const char *filename)
{
    if (!filename)
        return NULL;

    int unacceptable = 0;

    const char *p;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c))
            unacceptable++;
    }

    char *result = ho_malloc (strlen (filename) + unacceptable + 1);

    char *q = result;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c)) {
            *q++ = '\\';
        }
        *q++ = *p;
    }
    *q = '\0';

    //debug ("%s -> %s", filename, result);

    return result;
}


static void
device_entry_free (void *entry_p)
{
    device_entry_t *entry = (device_entry_t *) entry_p;

    if (!entry)
        return;
    ho_free (entry->device);
    ho_free (entry->mountpoint);
    ho_free (entry->current_mrl);
    ho_free (entry);
}

void
environment_free (void)
{
    if (disc_list)
        l_list_free (disc_list, device_entry_free);
    ho_free (dir_home);
    ho_free (dir_oxine);
    ho_free (dir_oxine_cache);
    ho_free (dir_oxine_tmp);
    ho_free (dir_oxine_playlists);
    ho_free (file_config);
    ho_free (file_mediamarks_music);
    ho_free (file_mediamarks_video);
    ho_free (file_mediamarks_media);
    ho_free (file_jsw_config);
    ho_free (file_lirc_config);
    ho_free (file_dvb_channels);
    ho_free (file_favorites);
    ho_free (file_playlist_rw);
}

char *
read_entire_file (const char *mrl, int *size)
{
    char *buffer = NULL;

    if (is_downloadable (mrl)) {
        download_t *download = ho_new (download_t);
        download->buf = NULL;
        download->error = NULL;
        download->size = 0;
        download->status = 0;

        if (network_download (mrl, download)) {
            buffer = ho_malloc (download->size);
            memcpy (buffer, download->buf, download->size);
            *size = download->size;
        }

        ho_free (download->buf);
        ho_free (download->error);
        ho_free (download);
    } else {
        if (access (mrl, R_OK) != 0) {
            error (_("Could not open '%s': %s!"), mrl, strerror (errno));
            return NULL;
        }

        struct stat st;
        if (stat (mrl, &st) < 0) {
            error (_("Unable to get status for file '%s': %s."), mrl,
                   strerror (errno));
            return NULL;
        }

        if ((*size = st.st_size) == 0) {
            error (_("File '%s' is empty!"), mrl);
            return NULL;
        }

        int fd;
        if ((fd = open (mrl, O_RDONLY)) == -1) {
            error (_("Could not open '%s': %s!"), mrl, strerror (errno));
            return NULL;
        }

        if ((buffer = (char *) ho_malloc (*size + 1)) == NULL) {
            close (fd);
            return NULL;
        }

        int bytes_read;
        if ((bytes_read = read (fd, buffer, *size)) != *size) {
            error (_("Could not read '%s': %s!"), mrl, strerror (errno));
            *size = bytes_read;
        }

        close (fd);

        buffer[*size] = '\0';
    }

    return buffer;
}


char *
get_basename (const char *mrl)
{
    char *mrl_copy = NULL;
    if (mrl)
        mrl_copy = ho_strdup (mrl);
    char *mrl_base = ho_strdup (basename (mrl_copy));
    if (mrl_copy)
        ho_free (mrl_copy);
    return mrl_base;
}


char *
get_dirname (const char *mrl)
{
    char *mrl_copy = NULL;
    if (mrl)
        mrl_copy = ho_strdup (mrl);
    char *mrl_dirn = ho_strdup (dirname (mrl_copy));
    if (mrl_copy)
        ho_free (mrl_copy);
    return mrl_dirn;
}


char *
get_thumbnail (const char *directory_mrl)
{
    char tmp_mrl[1024];
    char *thumbnail_mrl = NULL;

    snprintf (tmp_mrl, 1024, "%s/title.jpg", directory_mrl);
    if (!thumbnail_mrl && (access (tmp_mrl, R_OK) == 0))
        thumbnail_mrl = ho_strdup (tmp_mrl);

    snprintf (tmp_mrl, 1024, "%s/title.jpeg", directory_mrl);
    if (!thumbnail_mrl && (access (tmp_mrl, R_OK) == 0))
        thumbnail_mrl = ho_strdup (tmp_mrl);

    snprintf (tmp_mrl, 1024, "%s/title.png", directory_mrl);
    if (!thumbnail_mrl && (access (tmp_mrl, R_OK) == 0))
        thumbnail_mrl = ho_strdup (tmp_mrl);

    snprintf (tmp_mrl, 1024, "%s/title.gif", directory_mrl);
    if (!thumbnail_mrl && (access (tmp_mrl, R_OK) == 0))
        thumbnail_mrl = ho_strdup (tmp_mrl);

    snprintf (tmp_mrl, 1024, "%s/.directory", directory_mrl);
    if (!thumbnail_mrl && (access (tmp_mrl, R_OK) == 0)) {
        char *value = get_localestring_from_desktop_entry (tmp_mrl,
                                                           "Desktop Entry",
                                                           "Icon");
        if (value && (access (value, R_OK) == 0))
            thumbnail_mrl = value;
        else
            ho_free (value);
    }

    return thumbnail_mrl;
}


void
environment_config_register_all (void)
{
#ifdef HAVE_DISC_POLLING
    xine_config_register_string (oxine->xine, "misc.polling.devices",
                                 "/dev/cdrom;/dev/dvd/;/dev/usbstick",
                                 _("devices to poll (separate using ';')"),
                                 _("devices to poll (separate using ';')"),
                                 10, NULL, NULL);
#endif
}
