
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_osd.c 1574 2006-11-19 20:48:44Z mschwerin $
 *
 * This file is part of the odk system.
 *
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <unistd.h>

#ifdef HAVE_IMAGEMAGICK
#include <sys/types.h>
#include <magick/api.h>
#include <magick/ImageMagick.h>
#endif

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_private.h"


#ifdef DEBUG
#define VALIDATE_POINT(x,y)    {        \
    assert (x >= 0);                    \
    assert (y >= 0);                    \
    assert (x < odk->osd.width);        \
    assert (y < odk->osd.height);       \
}
#else
#define VALIDATE_POINT(x,y)    {        \
    if (x < 0)                          \
        return;                         \
    if (y < 0)                          \
        return;                         \
    if (x >= odk->osd.width)            \
        return;                         \
    if (y >= odk->osd.height)           \
        return;                         \
}
#endif

#define MIN(a,b)    (a < b ? a : b)
#define MAX(a,b)    (a > b ? a : b)

/* 
 * ***************************************************************************
 * Name:            odk_osd_show_osd
 * Access:          public
 *
 * Description:     Shows the passed OSD.
 * ***************************************************************************
 */
static void
odk_osd_show_osd (odk_t * odk, xine_osd_t * osd)
{
    if (odk->is_unscaled_osd_available)
        xine_osd_show_unscaled (osd, 0);
    else
        xine_osd_show (osd, 0);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_show
 * Access:          public
 *
 * Description:     Shows the OSD.
 * ***************************************************************************
 */
void
odk_osd_show (odk_t * odk)
{
    odk_osd_show_osd (odk, odk->osd.x_osd);

    odk->is_osd_visible = true;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_hide
 * Access:          public
 *
 * Description:     Hides the OSD.
 * ***************************************************************************
 */
void
odk_osd_hide (odk_t * odk)
{
    if (odk->osd.x_osd)
        xine_osd_hide (odk->osd.x_osd, 0);

#ifdef HAVE_OSD_IMAGE
    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        odk_osd_image_t *cur = odk->image_cache.entries[i].image;
        if (cur->x_osd && cur->is_visible) {
            xine_osd_hide (cur->x_osd, 0);
            cur->is_visible = false;
        }
    }
#endif

    odk->is_osd_visible = false;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_clear
 * Access:          public
 *
 * Description:     Clears the OSD.
 * ***************************************************************************
 */
void
odk_osd_clear (odk_t * odk)
{
    xine_osd_clear (odk->osd.x_osd);

    odk->is_osd_visible = false;
}

/* 
 * ***************************************************************************
 * Name:            rgb2yuv
 * Access:          private
 *
 * Description:     Converts RGB color to YUV color.
 * ***************************************************************************
 */
static uint32_t
rgb2yuv (uint32_t rgb)
{
    uint8_t r = (rgb >> 16) & 0xff;
    uint8_t g = (rgb >> 8) & 0xff;
    uint8_t b = (rgb >> 0) & 0xff;

    //    printf("\nRGB 0x%06X r:%3X g:%3X b:%3X\n", rgb, r, g, b);
    //    printf("RGB 0x%06X r:%3d g:%3d b:%3d\n", rgb, r, g, b);

    uint8_t y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
    uint8_t u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
    uint8_t v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;

    uint32_t yuv = ((y << 16) | (v << 8) | u);

    //    printf("YUV 0x%06X y:%3X u:%3X v:%3X\n", yuv, y, u, v);
    //    printf("YUV 0x%06X y:%3d u:%3d v:%3d\n", yuv, y, u, v);

    return yuv;
}


/* 
 * ***************************************************************************
 * Name:            get_color
 * Access:          private
 *
 * Description:     Returns the index of the color in the palette. If the
 *                  color is not in the palette yet, it is added. Input colors
 *                  should be YUV.
 * ***************************************************************************
 */
static int
get_color (odk_palette_t * palette, uint32_t color, uint8_t transparency)
{
    /* The first entry of the palette should always be transparent */
    if (palette->num_colors == 0) {
        palette->colors[0] = 0x00;
        palette->transparency[0] = 0x00;
        palette->num_colors++;
    }

    /* We ignore all entries that have no color */
    if (!color)
        return 0;

    int i = 0;

    int best_index = 0;
    uint8_t best_error = 0xff;

    /* What we do here is look for the color already in the palette, that best
     * matches the color we wish to add. The color must match. What is allowed
     * to be off a bit is the transparency. */
    for (; i < palette->num_colors; i++) {
        if (palette->colors[i] == color) {
            uint8_t current_error =
                abs (palette->transparency[i] - transparency);

            if (current_error == 0)
                return i;
            if (current_error < best_error) {
                best_error = current_error;
                best_index = i;
            }
        }
    }

    /* If the error in transparency is small enough we return the index of the
     * color we found. */
    if (best_error < 10)
        return best_index;

    /* If the error is to large and the palette is already full we return
     * index 0 which is completely transparent so that this color will not
     * appear in the output. */
    if (palette->num_colors == NUM_COLORS)
        return 0;

    /* If the error is to large and the palette still has space in it, we add
     * the new color. */
    palette->colors[palette->num_colors] = color;
    palette->transparency[palette->num_colors] = transparency;
    palette->num_colors++;

    return (palette->num_colors - 1);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_get_color
 * Access:          public
 *
 * Description:     Returns index to given color. Input colors should be RGB.
 * ***************************************************************************
 */
int
odk_osd_get_color (odk_t * odk, uint32_t color, uint8_t transparency)
{
    int color_index =
        get_color (&odk->osd.palette, rgb2yuv (color), transparency);

    xine_osd_set_palette (odk->osd.x_osd, odk->osd.palette.colors,
                          odk->osd.palette.transparency);

    return color_index;
}


/* 
 * ***************************************************************************
 * Name:            palette_transition
 * Access:          private
 *
 * Description:     Generates a gradient on palette out of given values. Input
 *                  colors should be YUV.
 * ***************************************************************************
 */
static void
palette_transition (odk_palette_t * palette, int num_colors,
                    uint32_t from_color, uint8_t from_transparency,
                    uint32_t to_color, uint8_t to_transparency)
{
    double transparency_step = to_transparency - from_transparency;
    transparency_step /= num_colors;

    uint8_t cb_from = from_color & 0xff;
    uint8_t cr_from = (from_color >> 8) & 0xff;
    uint8_t y_from = (from_color >> 16) & 0xff;

    uint8_t cb_to = to_color & 0xff;
    uint8_t cr_to = (to_color >> 8) & 0xff;
    uint8_t y_to = (to_color >> 16) & 0xff;

    double cb_step = (double) (cb_to - cb_from) / num_colors;
    double cr_step = (double) (cr_to - cr_from) / num_colors;
    double y_step = (double) (y_to - y_from) / num_colors;

    int first_index = palette->num_colors;
    int last_index = first_index + num_colors;

    int i = first_index + 1;
    for (; i < last_index; i++) {
        uint8_t cb_cur = cb_from + (int8_t) (cb_step * (i - first_index));
        uint8_t cr_cur = cr_from + (int8_t) (cr_step * (i - first_index));
        uint8_t y_cur = y_from + (int8_t) (y_step * (i - first_index));

        palette->colors[i] = cb_cur + (cr_cur << 8) + (y_cur << 16);
        palette->transparency[i] = from_transparency;
        palette->transparency[i] += transparency_step * (i - first_index);
    }

    palette->colors[first_index] = from_color;
    palette->transparency[first_index] = from_transparency;

    palette->colors[last_index] = to_color;
    palette->transparency[last_index] = to_transparency;

    palette->num_colors += num_colors;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_alloc_text_palette
 * Access:          public
 *
 * Description:     Allocates 10 colors in the palette and returns the index
 *                  to the first color. Input colors should be RGB.
 * ***************************************************************************
 */
int
odk_osd_alloc_text_palette (odk_t * odk,
                            uint32_t fg_color, uint8_t fg_transparency,
                            uint32_t bg_color, uint8_t bg_transparency,
                            uint32_t bo_color, uint8_t bo_transparency)
{
    int first_color = odk->osd.palette.num_colors;

    /* A xine text palette always consists of 11 colors of which color 0 is
     * transparent, color 1 is the background and color 10 is the foreground. */
    odk->osd.palette.colors[first_color] = 0;
    odk->osd.palette.transparency[first_color] = 0;
    odk->osd.palette.num_colors += 1;

    palette_transition (&odk->osd.palette, 10,
                        rgb2yuv (bg_color), bg_transparency,
                        rgb2yuv (fg_color), fg_transparency);

    xine_osd_set_palette (odk->osd.x_osd, odk->osd.palette.colors,
                          odk->osd.palette.transparency);

    return first_color;
}

#ifdef HAVE_OSD_IMAGE

/* 
 * ***************************************************************************
 * Name:            odk_osd_image_new
 * Access:          private
 *
 * Description:     Creates and initializes an OSD image.
 * ***************************************************************************
 */
static odk_osd_image_t *
odk_osd_image_new (odk_t * odk, const char *mrl,
                   int x, int y, int w, int h, int alignment,
                   int border_draw, uint32_t border_color,
                   int use_separate_osd)
{
    odk_osd_image_t *image = ho_new (odk_osd_image_t);

    image->mrl = ho_strdup (mrl);

    image->x = x;
    image->y = y;
    image->w = w;
    image->h = h;

    image->alignment = alignment;

    image->border_draw = border_draw;
    image->border_color = border_color;

    image->is_visible = false;

    /* This is set when loading the image. */
    image->x_osd = NULL;
    image->data = NULL;

    return image;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_image_free
 * Access:          private
 *
 * Description:     Frees an OSD image.
 * ***************************************************************************
 */
static void
odk_osd_image_free (odk_osd_image_t * image)
{
    if (image) {
        if (image->x_osd)
            xine_osd_free (image->x_osd);
        if (image->data)
            ho_free (image->data);
        if (image->mrl)
            ho_free (image->mrl);
        ho_free (image);
    }
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_load_image
 * Access:          private
 *
 * Description:     Loads an OSD image. If use_separate_osd is true the image is
 *                  drawn into a separate OSD object that is saved inside the
 *                  image object. Else only the image data is loaded.
 * ***************************************************************************
 */
static odk_osd_image_t *
odk_osd_load_image (odk_t * odk, const char *mrl,
                    int x, int y, int width, int height, int alignment,
                    bool border_draw, uint32_t border_color,
                    bool use_separate_osd)
{
    assert (width > 0);
    assert (height > 0);

    if (!(odk->osd.hscale && odk->osd.vscale))
        return NULL;

    /* Initialize ImageMagick and load the image. */
    InitializeMagick (NULL);

    ExceptionInfo exception;
    GetExceptionInfo (&exception);

    ImageInfo *image_info = CloneImageInfo ((ImageInfo *) NULL);
    strcpy (image_info->filename, mrl);

    Image *image = ReadImage (image_info, &exception);
    CatchException (&exception);
    if (!image) {
        error (_("Could not read image '%s'!"), mrl);

        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        return NULL;
    }

    /* Create OSD image data structure. */
    odk_osd_image_t *osd_image = odk_osd_image_new (odk, mrl, x, y,
                                                    width, height, alignment,
                                                    border_draw, border_color,
                                                    use_separate_osd);

    double des_ratio = (double) height / (double) width;
    double img_ratio = (double) image->rows / (double) image->columns;

    if (des_ratio < img_ratio) {
        width = ((double) image->columns / (double) image->rows)
            * (double) height;
    } else {
        height = ((double) image->rows / (double) image->columns)
            * (double) width;
    }

    if (alignment & ODK_ALIGN_RIGHT)
        x -= width;
    if (alignment & ODK_ALIGN_CENTER)
        x -= width / 2;
    if (alignment & ODK_ALIGN_VCENTER)
        y -= height / 2;
    if (alignment & ODK_ALIGN_BOTTOM)
        y -= height;

    /* Scale width and height to the current output size. */
    width *= odk->osd.hscale;
    height *= odk->osd.vscale;

    /* Scale position */
    x *= odk->osd.hscale;
    y *= odk->osd.vscale;

    if (x < 0)
        x += OSD_WIDTH * odk->osd.hscale - width;
    if (y < 0)
        y += OSD_HEIGHT * odk->osd.vscale - height;

    /* If necessary we scale the image. We have to use SampleImage 
     * here so we don't introduce new colors to the image. */
    if ((image->columns != width) || (image->rows != height)) {
        image = SampleImage (image, width, height, &exception);
        CatchException (&exception);

        if (!image) {
            error (_("Could not scale image '%s'!"), mrl);

            /* Destroy ImageMagick stuff */
            DestroyImage (image);
            DestroyImageInfo (image_info);
            DestroyExceptionInfo (&exception);
            DestroyMagick ();

            /* Free OSD image */
            odk_osd_image_free (osd_image);

            return NULL;
        }
    }

    /* We fetch the number of unique colors of the image. */
    int num_colors = GetNumberColors (image, NULL, &exception);
    CatchException (&exception);

    /* The OSD palette has NUM_COLORS colors. The first entry 
     * of our palette is always transparent. */
    int max_colors = NUM_COLORS - 1;
    /* If necessary we leave space for the border color. */
    if (border_draw)
        max_colors -= 1;

    /* If necessary, we quantize the image. */
    if (num_colors > max_colors) {
        QuantizeInfo quantize_info;
        GetQuantizeInfo (&quantize_info);
        quantize_info.number_colors = max_colors;
#ifdef USE_YUV
        quantize_info.colorspace = YUVColorspace;
#endif
        QuantizeImage (&quantize_info, image);
        CatchException (&exception);
    }

    /* Get image pixels. */
    const PixelPacket *pp = AcquireImagePixels (image, 0, 0, width, height,
                                                &exception);
    CatchException (&exception);
    if (!pp) {
        error (_("Could not get pixel data!"));

        /* Destroy ImageMagick stuff */
        DestroyImage (image);
        DestroyImageInfo (image_info);
        DestroyExceptionInfo (&exception);
        DestroyMagick ();

        /* Free OSD image */
        odk_osd_image_free (osd_image);

        return NULL;
    }

    uint8_t *image_data = ho_malloc (width * height);

    odk_palette_t *palette;
    if (use_separate_osd) {
        palette = ho_new (odk_palette_t);
        palette->num_colors = 0;
    } else {
        palette = &odk->osd.palette;
    }

    int p = 0;
    uint8_t palette_map[NUM_COLORS];
    for (; p < NUM_COLORS; p++)
        palette_map[p] = p;

    int col;
    int row;
    for (row = 0; row < height; row++) {
        for (col = 0; col < width; col++) {
#ifdef USE_YUV
            uint8_t y = ScaleQuantumToChar (pp->red);
            uint8_t u = ScaleQuantumToChar (pp->green);
            uint8_t v = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = ((y << 16) | (v << 8) | u);
#else
            uint8_t r = ScaleQuantumToChar (pp->red);
            uint8_t g = ScaleQuantumToChar (pp->green);
            uint8_t b = ScaleQuantumToChar (pp->blue);
            uint32_t yuv = rgb2yuv ((r << 16) | (g << 8) | b);
#endif
            uint8_t t = 0xff - ScaleQuantumToChar (pp->opacity);
            image_data[row * width + col] = get_color (palette, yuv, t);
            pp++;
        }
    }

    /* Draw a border around the image. */
    if (border_draw) {
        uint32_t color = get_color (palette, rgb2yuv (border_color), 0xff);

        for (col = 0; col < width; col++) {
            image_data[col] = color;
            image_data[col + (height - 1) * width] = color;
        }

        for (row = 0; row < height; row++) {
            image_data[row * width] = color;
            image_data[(row + 1) * width - 1] = color;
        }
    }

    if (use_separate_osd) {
        /* Create a new OSD object. */
        osd_image->x_osd = xine_osd_new (odk->win->stream,
                                         odk->osd.x_off, odk->osd.y_off,
                                         odk->osd.width, odk->osd.height);
        /* Set the palette. */
        xine_osd_set_palette (osd_image->x_osd, palette->colors,
                              palette->transparency);
        /* Draw the image to that OSD object. */
        xine_osd_draw_bitmap (osd_image->x_osd, image_data,
                              x, y, width, height, palette_map);
        /* We don't need the image data and the palette any more. */
        ho_free (image_data);
        ho_free (palette);
    }

    else {
        /* We keep the image data inside our OSD image. */
        osd_image->data = image_data;
        /* Set the palette. */
        xine_osd_set_palette (odk->osd.x_osd, odk->osd.palette.colors,
                              odk->osd.palette.transparency);
    }

    /* Destroy the ImageMagick stuff. */
    DestroyImage (image);
    DestroyImageInfo (image_info);
    DestroyExceptionInfo (&exception);
    DestroyMagick ();

    return osd_image;
}

/* 
 * ***************************************************************************
 * Name:            odk_osd_image_cache_init
 * Access:          private
 *
 * Description:     Inits the internal image cache.
 * ***************************************************************************
 */
static void
odk_osd_image_cache_init (odk_t * odk)
{
    int i = 0;
    for (; i < NUM_IMAGE_CACHE_ENTRIES; i++) {
        odk->image_cache.entries[i].image = NULL;
        odk->image_cache.entries[i].timestamp = 0;
    }
    odk->image_cache.fill = 0;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_image_cache_free
 * Access:          private
 *
 * Description:     Frees the internal image cache.
 * ***************************************************************************
 */
static void
odk_osd_image_cache_free (odk_t * odk)
{
    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        odk_osd_image_free (odk->image_cache.entries[i].image);
        odk->image_cache.entries[i].image = NULL;
        odk->image_cache.entries[i].timestamp = 0;
    }
    odk->image_cache.fill = 0;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_image_cache_get
 * Access:          private
 *
 * Description:     Retrieves an image from the internal image cache.
 * ***************************************************************************
 */
static odk_osd_image_t *
odk_osd_image_cache_get (odk_t * odk, const char *mrl,
                         int x, int y, int w, int h, int alignment,
                         bool border_draw, uint32_t border_color,
                         bool use_separate_osd)
{
    int i = 0;
    for (; i < odk->image_cache.fill; i++) {
        image_cache_entry_t entry = odk->image_cache.entries[i];
        if ((strcmp (mrl, entry.image->mrl) == 0)
            && (entry.image->x == x)
            && (entry.image->y == y)
            && (entry.image->w == w)
            && (entry.image->h == h)
            && (entry.image->alignment == alignment)
            && ((entry.image->border_draw == border_draw))
            && ((entry.image->border_color == border_color))
            && ((use_separate_osd && entry.image->x_osd)
                || (!use_separate_osd && entry.image->data))) {
            entry.timestamp = time (NULL);
            // debug ("image cache: HIT (index=%d)", i);
            return entry.image;
        }
    }

    // debug ("image cache: MISS");

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_image_cache_add
 * Access:          private
 *
 * Description:     Adds an image to the internal image cache.
 * ***************************************************************************
 */
static int
odk_osd_image_cache_add (odk_t * odk, odk_osd_image_t * image)
{
    assert (odk);
    assert (image);

    if (odk->image_cache.fill < NUM_IMAGE_CACHE_ENTRIES) {
        // debug ("image cache: ADD (index=%d)", odk->image_cache.fill);
        odk->image_cache.entries[odk->image_cache.fill].image = image;
        odk->image_cache.entries[odk->image_cache.fill].timestamp =
            time (NULL);
        odk->image_cache.fill += 1;
    }

    else {
        /* Find the cache entry that has been used the least recently. */
        int i = 0;
        int j = 0;
        time_t ts = time (NULL);
        for (; i < NUM_IMAGE_CACHE_ENTRIES; i++) {
            image_cache_entry_t entry = odk->image_cache.entries[i];
            if (!entry.image->is_visible && (entry.timestamp < ts)) {
                j = i;
                ts = entry.timestamp;
            }
        }

        /* Remove and free that entry and add the new image in that place. */
        // debug ("image cache: REPLACE (index=%d)", j);
        if (!odk->image_cache.entries[j].image->is_visible) {
            odk_osd_image_free (odk->image_cache.entries[j].image);
            odk->image_cache.entries[j].image = image;
            odk->image_cache.entries[j].timestamp = time (NULL);
        } else {
            error (_("Could not free a cache entry for '%s'."), image->mrl);
            return 0;
        }
    }
    return 1;
}
#endif


/* 
 * ***************************************************************************
 * Name:            odk_osd_draw_image
 * Access:          public
 *
 * Description:     Draws the image into the standard OSD.
 * ***************************************************************************
 */
odk_osd_image_t *
odk_osd_draw_image (odk_t * odk, const char *mrl,
                    int x, int y, int width, int height, int alignment,
                    bool border_draw, uint32_t border_color)
{
#ifdef HAVE_OSD_IMAGE
    odk_osd_image_t *image = NULL;

    assert (odk);
    assert (mrl);

    if (access (mrl, R_OK) == 0) {
        image = odk_osd_image_cache_get (odk, mrl, x, y, width, height,
                                         alignment,
                                         border_draw, border_color, false);

        if (!image) {
            image = odk_osd_load_image (odk, mrl, x, y, width, height,
                                        alignment,
                                        border_draw, border_color, false);
            if (image)
                odk_osd_image_cache_add (odk, image);
        }

        if (image) {
            int p = 0;
            uint8_t palette_map[NUM_COLORS];
            for (; p < NUM_COLORS; p++)
                palette_map[p] = p;

            /* We only have to scale the location and size. The image data has
             * already been scaled to the correct size either in
             * odk_osd_adapt_size or on loading. */
            int x = (int) ((double) image->x * odk->osd.hscale);
            int y = (int) ((double) image->y * odk->osd.vscale);
            int w = (int) ((double) image->w * odk->osd.hscale);
            int h = (int) ((double) image->h * odk->osd.vscale);

            xine_osd_draw_bitmap (odk->osd.x_osd, image->data,
                                  x, y, w, h, palette_map);
        }
    }

    else {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
    }

    return image;
#else
    return NULL;
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_show_image
 * Access:          public
 *
 * Description:     Draws the image into a separate OSD and shows that.
 * ***************************************************************************
 */
odk_osd_image_t *
odk_osd_show_image (odk_t * odk, const char *mrl,
                    int x, int y, int width, int height, int alignment,
                    bool border_draw, uint32_t border_color)
{
#ifdef HAVE_OSD_IMAGE
    odk_osd_image_t *image = NULL;

    assert (odk);
    assert (mrl);

    if (access (mrl, R_OK) == 0) {
        image = odk_osd_image_cache_get (odk, mrl, x, y, width, height,
                                         alignment,
                                         border_draw, border_color, true);

        if (!image) {
            image = odk_osd_load_image (odk, mrl, x, y, width, height,
                                        alignment,
                                        border_draw, border_color, true);
            if (image)
                odk_osd_image_cache_add (odk, image);
        }

        if (image && !image->is_visible) {
            odk_osd_show_osd (odk, image->x_osd);
            image->is_visible = true;
        }
    }

    else {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
    }

    return image;
#else
    return NULL;
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_hide_image
 * Access:          public
 *
 * Description:     This hides the image OSD. It is important to note, that
 *                  pointers to the passed image may become invalid after
 *                  hiding the image, as non-visible entries in the internal
 *                  image cache may be freed.
 * ***************************************************************************
 */
void
odk_osd_hide_image (odk_osd_image_t * image)
{
#ifdef HAVE_OSD_IMAGE
    if (image && image->x_osd && image->is_visible) {
        xine_osd_hide (image->x_osd, 0);
        image->is_visible = false;
    }
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_get_bitmap
 * Access:          public
 *
 * Description:     Returns predefined bitmaps
 * ***************************************************************************
 */
uint8_t *
odk_get_bitmap (int type)
{
    uint8_t playlist[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };
    uint8_t directory[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0
    };
    uint8_t home[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0
    };
    uint8_t poweroff[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0,
        0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
        0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0,
        0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0
    };

    uint8_t arrowup[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
    };

    uint8_t simplearrowup[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
    };

    uint8_t mute[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0,
        0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0
    };

    uint8_t volume[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
        0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0
    };

    uint8_t plus[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    };

    uint8_t minus[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };

    uint8_t check[] = {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0,
        0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
        0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };

    uint8_t *p = ho_malloc (sizeof (uint8_t) * 400);

    switch (type) {
    case BITMAP_PLAYLIST:
        memcpy (p, playlist, 400);
        break;
    case BITMAP_FOLDER:
        memcpy (p, directory, 400);
        break;
    case BITMAP_HOME:
        memcpy (p, home, 400);
        break;
    case BITMAP_POWEROFF:
        memcpy (p, poweroff, 400);
        break;
    case BITMAP_CHECK:
        memcpy (p, check, 400);
        break;
    case BITMAP_ARROW_UP:
        memcpy (p, arrowup, 400);
        break;
    case BITMAP_ARROW_DOWN:
        {
            int row = 19;
            for (; row >= 0; row--) {
                memcpy (p + (19 - row) * 20, arrowup + row * 20, 20);
            }
        }
        break;
    case BITMAP_ARROW_LEFT:
        {
            int i = 0;
            for (; i < 20; i++) {
                int j = 0;
                for (; j < 20; j++) {
                    p[i * 20 + j] = arrowup[j * 20 + i];
                }
            }
        }
        break;
    case BITMAP_ARROW_RIGHT:
        {
            int i = 0;
            for (; i < 20; i++) {
                int j = 0;
                for (; j < 20; j++)
                    p[i * 20 + (19 - j)] = arrowup[j * 20 + i];
            }
        }
        break;
    case BITMAP_SIMPLE_ARROW_UP:
        memcpy (p, simplearrowup, 400);
        break;
    case BITMAP_SIMPLE_ARROW_DOWN:
        {
            int row = 19;
            for (; row >= 0; row--) {
                memcpy (p + (19 - row) * 20, simplearrowup + row * 20, 20);
            }
        }
        break;
    case BITMAP_PLUS:
        memcpy (p, plus, 400);
        break;
    case BITMAP_MINUS:
        memcpy (p, minus, 400);
        break;
    case BITMAP_MUTE:
        memcpy (p, mute, 400);
        break;
    case BITMAP_VOLUME:
        memcpy (p, volume, 400);
        break;
    }

    return p;
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_bitmap
 * Access:          public
 *
 * Description:     Draw a bitmap. x and y define the center for the bitmap!
 * ***************************************************************************
 */
void
odk_draw_bitmap (odk_t * odk,
                 uint8_t * bitmap, int x, int y, int w, int h,
                 uint8_t * palette_map)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    //int pw = (int) ((double) w * odk->hscale);
    //int ph = (int) ((double) h * odk->osd.vscale);
    int px = round ((double) x * odk->osd.hscale) - w / 2;
    int py = round ((double) y * odk->osd.vscale) - h / 2;

    VALIDATE_POINT (px, py);
    VALIDATE_POINT (px - w / 2, py - h / 2);
    VALIDATE_POINT (px + w / 2, py + h / 2);

    // TODO: scale bitmap
    /*
       uint8_t pbmp[pw * ph];
       printf ("%d %d %d %d\n", w, h, pw, ph);
       for (y = 0; y < ph; y++) {
       for (x = 0; x < pw; x++) {
       int ox = x / odk->hscale;
       int oy = y / odk->osd.vscale;
       pbmp[x + (y * pw)] = bitmap[ox + (oy * w)];
       }
       }
     */

    xine_osd_draw_bitmap (odk->osd.x_osd, bitmap, px, py, w, h, palette_map);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_line
 * Access:          public
 *
 * Description:     Draw a line.
 * ***************************************************************************
 */
void
odk_draw_line (odk_t * odk, int x1, int y1, int x2, int y2, int color)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px1 = round ((double) x1 * odk->osd.hscale);
    int py1 = round ((double) y1 * odk->osd.vscale);
    int px2 = round ((double) x2 * odk->osd.hscale);
    int py2 = round ((double) y2 * odk->osd.vscale);

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);

    xine_osd_draw_line (odk->osd.x_osd, px1, py1, px2, py2,
                        XINE_OSD_TEXT1 + color);

    xine_osd_draw_point (odk->osd.x_osd, px1, py1, XINE_OSD_TEXT1 + color);
    xine_osd_draw_point (odk->osd.x_osd, px2, py2, XINE_OSD_TEXT1 + color);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_rect
 * Access:          public
 *
 * Description:     Draw a rectangle.
 * ***************************************************************************
 */
void
odk_draw_rect (odk_t * odk, int x1, int y1, int x2, int y2,
               int color, bool filled)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int px1 = round ((double) x1 * odk->osd.hscale);
    int py1 = round ((double) y1 * odk->osd.vscale);
    int px2 = round ((double) x2 * odk->osd.hscale);
    int py2 = round ((double) y2 * odk->osd.vscale);

    VALIDATE_POINT (px1, py1);
    VALIDATE_POINT (px2, py2);

    // FIXME: This is a hack! We draw a point to the bottom righthand corner.
    // This is necessary, because the bottom line of a non-filled rectangle is
    // not displayed, if nothing else has a y-value that is greater. This is a
    // xine-lib bug (I think).
    if (!filled) {
        xine_osd_draw_point (odk->osd.x_osd, odk->osd.width, odk->osd.height,
                             XINE_OSD_TEXT1 + color);
    }
    // FIXME: xine-lib does not draw the point at the lower right hand corner
    // of an unfilled rectangle.
    xine_osd_draw_point (odk->osd.x_osd, px2, py2, XINE_OSD_TEXT1 + color);
    xine_osd_draw_rect (odk->osd.x_osd, px1, py1, px2, py2,
                        XINE_OSD_TEXT1 + color, false);

    // FIXME: xine-lib does not draw the correct width of a filled rectangle.
    xine_osd_draw_rect (odk->osd.x_osd, px1, py1, px2, py2,
                        XINE_OSD_TEXT1 + color, filled);
}


/* 
 * ***************************************************************************
 * Name:            odk_draw_text
 * Access:          public
 *
 * Description:     Draw a text.
 * ***************************************************************************
 */
void
odk_draw_text (odk_t * odk, int x, int y, const char *text,
               int alignment, int color_base)
{
    if (!(odk->osd.hscale && odk->osd.vscale))
        return;

    int pw = 0;
    int pd = 0;
    int ph = 0;
    {
        int w;
        int h1;
        int h2;
        // We always want the same baseline for text.
        // It should not matter if the text contains 
        // characters like g or f.
        odk_get_text_size (odk, "DVB", &w, &h1);
        odk_get_text_size (odk, "DVg", &w, &h2);

        int h = h1 - (h2 - h1);
        // This moves the text up a few pixels, so that 
        // the top is really at the top of the charaters.
        y -= (h2 - h1);

        if (odk->is_current_font_freetype) {
            if (alignment & ODK_ALIGN_VCENTER)
                y += h / 2;
            if (alignment & ODK_ALIGN_TOP)
                y += h;
        } else {
            if (alignment & ODK_ALIGN_VCENTER)
                y -= h / 2;
            if (alignment & ODK_ALIGN_BOTTOM)
                y -= h;
        }

        pd = round ((double) (h2 - h1) * odk->osd.vscale);
        ph = round ((double) h * odk->osd.vscale);
    }
    {
        int w;
        int h;
        // The width has to be calculated using 
        // the real text.
        odk_get_text_size (odk, text, &w, &h);

        if (alignment & ODK_ALIGN_CENTER)
            x -= w / 2;
        if (alignment & ODK_ALIGN_RIGHT)
            x -= w;

        pw = round ((double) w * odk->osd.hscale);
    }

    int px = round ((double) x * odk->osd.hscale);
    int py = round ((double) y * odk->osd.vscale);

    VALIDATE_POINT (px, py);
    VALIDATE_POINT (px + pw, py + ph);

    xine_osd_draw_text (odk->osd.x_osd, px, py, text, color_base);

#ifdef DEBUG_TEXT_DRAWING
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd, px + pw, py + pd,
                        color_base + 10);
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd + ph / 2, px + pw,
                        py + pd + ph / 2, color_base + 10);
    xine_osd_draw_line (odk->osd.x_osd, px, py + pd + ph, px + pw,
                        py + pd + ph, color_base + 10);
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_get_text_size
 * Access:          public
 *
 * Description:     Returns width and height of the text.
 * ***************************************************************************
 */
void
odk_get_text_size (odk_t * odk, const char *text, int *width, int *height)
{
    int w;
    int h;

    if (!(odk->osd.hscale && odk->osd.vscale)) {
        *width = 0;
        *height = 0;
        return;
    }

    xine_osd_get_text_size (odk->osd.x_osd, text, &w, &h);
    *width = (int) ((double) w / odk->osd.hscale);
    *height = (int) ((double) h / odk->osd.vscale);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_set_font
 * Access:          public
 *
 * Description:     Sets the font and font-size xine-lib uses to paint text.
 * ***************************************************************************
 */
void
odk_osd_set_font (odk_t * odk, const char *font, int font_size)
{
    int psize;

    if (odk->osd.hscale < odk->osd.vscale)
        psize = (int) ((double) font_size * odk->osd.hscale);
    else
        psize = (int) ((double) font_size * odk->osd.vscale);

    /* smallest text size possible */
    if (psize < 16)
        psize = 16;

    if (strchr (font, '.') || strchr (font, '/'))
        odk->is_current_font_freetype = 1;
    else
        odk->is_current_font_freetype = 0;

    xine_osd_set_font (odk->osd.x_osd, font, psize);
}


/* 
 * ***************************************************************************
 * Name:            odk_get_frame_size
 * Access:          private
 *
 * Description:     Returns width and height of the current frame.
 * ***************************************************************************
 */
void
odk_get_frame_size (odk_t * odk, int *w, int *h)
{
    int r;
    int f;

    if (!xine_get_current_frame (odk->win->stream, w, h, &r, &f, NULL)) {
        *w = xine_get_stream_info (odk->win->stream,
                                   XINE_STREAM_INFO_VIDEO_WIDTH);
        *h = xine_get_stream_info (odk->win->stream,
                                   XINE_STREAM_INFO_VIDEO_HEIGHT);
    }
}


/* 
 * ***************************************************************************
 * Name:            adapt_osd_size
 * Access:          private
 *
 * Description:     Adapt the OSD to the current output size. Whenever the
 *                  size of the current video is equal to that of the OSD we
 *                  do not use unscaled OSD so that we can use transparency.
 * ***************************************************************************
 */
void
odk_osd_adapt_size (odk_t * odk, xine_format_change_data_t * fcd)
{
    int x_off = 0;
    int y_off = 0;
    int width = 0;
    int height = 0;

    /* If the size of the current frame is not equal to that of our OSD and if
     * unscaled OSD is available, we use unscaled OSD. */
    if (odk->is_unscaled_osd_available) {
        width = odk->win->output_width;
        height = odk->win->output_height;
    }
    /* If we got XINE_EVENT_FRAME_FORMAT_CHANGE we can use the data provided
     * by this event. */
    else if (fcd) {
        width = fcd->width;
        height = fcd->height;
    }
    /* Else we've got to get the data all by ourselves. */
    else {
        odk_get_frame_size (odk, &width, &height);
    }

    /* Adjust the OSD size to the current zoom level. */
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (odk->xine, "gui.video.zoom", &centry);
    if (centry.num_value < 100) {
        int w = width;
        int h = height;
        width = (width * centry.num_value) / 100;
        height = (height * centry.num_value) / 100;
        x_off = (w - width) / 2;
        y_off = (h - height) / 2;
    }

    /* Correct height and width to the OSD aspect ratio. */
    double osd_aspect = (double) OSD_WIDTH / (double) OSD_HEIGHT;
    double output_aspect = (double) width / (double) height;
    if (osd_aspect > output_aspect) {
        int real_height = height;
        height = width * OSD_HEIGHT / OSD_WIDTH;
        y_off += (real_height - height) / 2;
    } else if (osd_aspect < output_aspect) {
        int real_width = width;
        width = height * OSD_WIDTH / OSD_HEIGHT;
        x_off += (real_width - width) / 2;
    }

    if (!odk->osd.x_osd || (odk->osd.x_off != x_off)
        || (odk->osd.y_off != y_off)
        || (odk->osd.width != width)
        || (odk->osd.height != height)) {

        /* We hide all images so they don't get redrawn in the old size before
         * we have resized them to the new OSD size. */
#ifdef HAVE_OSD_IMAGE
        int i;
        for (i = 0; i < odk->image_cache.fill; i++) {
            odk_osd_image_t *cur = odk->image_cache.entries[i].image;
            if (cur->x_osd && cur->is_visible) {
                xine_osd_hide (cur->x_osd, 0);
            }
        }
#endif

        odk->osd.x_off = x_off;
        odk->osd.y_off = y_off;
        odk->osd.width = width;
        odk->osd.height = height;

        odk->osd.vscale = (double) height / (double) OSD_HEIGHT;
        odk->osd.hscale = (double) width / (double) OSD_WIDTH;

        /* Free the old OSD, and create a new one. */
        if (odk->osd.x_osd)
            xine_osd_free (odk->osd.x_osd);
        odk->osd.x_osd = xine_osd_new (odk->win->stream,
                                       odk->osd.x_off, odk->osd.y_off,
                                       odk->osd.width, odk->osd.height);

        xine_osd_set_palette (odk->osd.x_osd,
                              odk->osd.palette.colors,
                              odk->osd.palette.transparency);

        /* Adapt the sizes of the images in the cache and show any image, that
         * is marked as visible. */
#ifdef HAVE_OSD_IMAGE
        for (i = 0; i < odk->image_cache.fill; i++) {
            odk_osd_image_t *cur = odk->image_cache.entries[i].image;

            if (cur->x_osd) {
                odk_osd_image_t *tmp = odk_osd_load_image (odk, cur->mrl,
                                                           cur->x, cur->y,
                                                           cur->w, cur->h,
                                                           cur->alignment,
                                                           cur->border_draw,
                                                           cur->border_color,
                                                           true);

                xine_osd_free (cur->x_osd);
                cur->x_osd = tmp->x_osd;
                ho_free (tmp->mrl);
                ho_free (tmp);

                if (cur->is_visible)
                    odk_osd_show_osd (odk, cur->x_osd);
            }

            else if (cur->data) {
                odk_osd_image_t *tmp = odk_osd_load_image (odk, cur->mrl,
                                                           cur->x, cur->y,
                                                           cur->w, cur->h,
                                                           cur->alignment,
                                                           cur->border_draw,
                                                           cur->border_color,
                                                           false);

                ho_free (cur->data);
                cur->data = tmp->data;
                ho_free (tmp->mrl);
                ho_free (tmp);
            }
        }
#endif
    }

    /* We send an event to notify any listeners that the format of the OSD has
     * changed. */
    oxine_event_t ev;
    ev.type = OXINE_EVENT_OSD_FORMAT_CHANGED;
    odk_oxine_event_send (odk, &ev);
}


/* 
 * ***************************************************************************
 * Name:            odk_is_osd_visible
 * Access:          public
 *
 * Description:     Is the OSD visible?
 * ***************************************************************************
 */
int
odk_is_osd_visible (odk_t * odk)
{
    return odk->is_osd_visible;
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_init
 * Access:          private
 *
 * Description:     Init OSD stuff.
 * ***************************************************************************
 */
void
odk_osd_init (odk_t * odk)
{
    odk->is_osd_visible = false;
    odk->is_current_font_freetype = false;
    odk->is_unscaled_osd_available = false;

    odk->osd.x_osd = NULL;
    odk->osd.hscale = 0;
    odk->osd.vscale = 0;
    odk->osd.x_off = 0;
    odk->osd.y_off = 0;
    odk->osd.width = 0;
    odk->osd.height = 0;
    odk->osd.palette.num_colors = 0;

    /* Test support of unscaled OSD. */
    xine_osd_t *x_osd = xine_osd_new (odk->win->stream, 0, 0, 10, 10);
    int capabilities = xine_osd_get_capabilities (x_osd);
    xine_osd_free (x_osd);
    odk->is_unscaled_osd_available = (capabilities & XINE_OSD_CAP_UNSCALED);
    if (odk->is_unscaled_osd_available) {
        info (_("The current video driver supports unscaled OSD!"));
    } else {
        warn (_("The current video driver does not support unscaled OSD!"));
    }

    /* Initialize the image cache. */
#ifdef HAVE_OSD_IMAGE
    odk_osd_image_cache_init (odk);
#endif

    /* Adapt size of OSD to current window/ output size. */
    odk_osd_adapt_size (odk, NULL);
}


/* 
 * ***************************************************************************
 * Name:            odk_osd_free
 * Access:          private
 *
 * Description:     Free OSD stuff.
 * ***************************************************************************
 */
void
odk_osd_free (odk_t * odk)
{
    /* Free the image cache. */
#ifdef HAVE_OSD_IMAGE
    odk_osd_image_cache_free (odk);
#endif

    /* Free the OSD drawing area. */
    if (odk->osd.x_osd)
        xine_osd_free (odk->osd.x_osd);
    odk->osd.x_osd = NULL;
}
