
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 * The scrollbar code was backported from the xine-ui CVS and was originally
 * written by Miguel Freitas.
 *
 * We really have to worry about threads in this module! We have three threads:
 * the update-thread, the user-interaction-thread (motion, buttons, keys) and
 * the xine-lib-thread.
 *
 * The update-thread only interacts with otk in otk_update_job. The
 * user-interaction-thread always passes through otk_event_handler. So it
 * should be enough to enclose those two methods in mutexes.
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/X.h>
#include <X11/keysym.h>

#include "assert.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "mutex.h"
#include "odk.h"
#include "otk.h"
#include "oxine.h"
#include "scheduler.h"
#include "xmlparser.h"

#define DIRECTION_UP                    (1)
#define DIRECTION_DOWN                  (2)
#define DIRECTION_LEFT                  (3)
#define DIRECTION_RIGHT                 (4)

#define OTK_WIDGET_WINDOW               (1)
#define OTK_WIDGET_BUTTON               (2)
#define OTK_WIDGET_LABEL                (3)
#define OTK_WIDGET_LIST                 (4)
#define OTK_WIDGET_LISTENTRY            (5)
#define OTK_WIDGET_SLIDER               (6)
#define OTK_WIDGET_SCROLLBAR            (8)
#define OTK_WIDGET_CHECKBOX             (9)
#define OTK_WIDGET_BORDER               (10)
#define OTK_WIDGET_EDITBOX              (11)

#define OSD_TEXT_PALETTE_TRANSPARENT    (0)
#define OSD_TEXT_PALETTE_BACKGROUND     (1)
#define OSD_TEXT_PALETTE_FOREGROUND     (10)

#define ABS(x) ((x)<0?-(x):(x))

typedef struct otk_button_s otk_button_t;
typedef struct otk_label_s otk_label_t;
typedef struct otk_window_s otk_window_t;
typedef struct otk_list_s otk_list_t;
typedef struct otk_listentry_s otk_listentry_t;
typedef struct otk_scrollbar_s otk_scrollbar_t;
typedef struct otk_slider_s otk_slider_t;
typedef struct otk_checkbox_s otk_checkbox_t;
typedef struct otk_border_s otk_border_t;
typedef struct otk_editbox_s otk_editbox_t;

struct otk_widget_s {
    int type;

    otk_t *otk;
    odk_t *odk;

    int x;
    int y;
    int w;
    int h;

    int selectable;

    bool is_focused;
    bool is_enabled;
    bool is_visible;

    bool need_update;
    bool need_repaint;

    char *font;
    int fontsize;
    int alignment;

    void (*draw) (otk_widget_t * this);
    void (*destroy) (otk_widget_t * this);

    // we call this callback when this item gets the focus
    void *focus_enter_cb_data;
    void (*focus_enter_cb) (void *user_data);

    // we call this callback when this item looses the focus
    void *focus_leave_cb_data;
    void (*focus_leave_cb) (void *user_data);
};

struct otk_slider_s {
    otk_widget_t widget;

    // horizontal or vertical?
    int direction;
    // just a backup
    int last_value;
    // how to draw the value
    char *value_format;

    int marker_w;
    int marker_h;

    // use a simple marker
    bool simple_marker;

    int default_value;

    int min_value;
    int max_value;
    int min_step;

    // we call this callback to when the slider is clicked
    void *set_value_cb_data;
    otk_int_set_cb_t set_value_cb;

    // we call this callback to get the current value
    void *get_value_cb_data;
    otk_int_get_cb_t get_value_cb;
};

struct otk_border_s {
    otk_widget_t widget;
};

struct otk_editbox_s {
    otk_widget_t widget;

    char *text;
    char *display_text;

    int max_length;
    int cur_length;

    int cur_position;

    int first_visible;
    int num_visible;

    int carret_x;

    // we call this callback when the editbox is changed
    void *cb_data;
    otk_editbox_cb_t cb;
};

struct otk_checkbox_s {
    otk_widget_t widget;

    int is_checked;
    uint8_t *bitmap;

    // we call this callback when the checkbox is clicked
    void *cb_data;
    otk_cb_t cb;
};

struct otk_button_s {
    otk_widget_t widget;

    /* The text for a text only button. */
    char *text;

    /* The bitmap for a bitmap button. */
    uint8_t *bitmap;

    /* The images for image buttons. */
    char *image_mrl;
    char *image_focus_mrl;
    odk_osd_image_t *image;

    // we call this callback when the button is clicked
    void *cb_data;
    otk_cb_t cb;
};

struct otk_listentry_s {
    otk_widget_t widget;

    otk_list_t *list;

    int pos_visible_list;
    int pos_complete_list;

    bool is_first;
    bool is_last;
    bool is_selected;
    bool is_visible;

    // text to show
    char *text;

    // time this item was last clicked
    time_t doubleclicktime;

    // we call this callback when this item is activated (ENTER, doubleclick)
    void *activate_cb_data;
    otk_cb_t activate_cb;

    // we call this callback when this item is selected or deselected (SPACE, singleclick)
    void *select_cb_data;
    otk_cb_t select_cb;

    // we call this callback when the remove key is pressed
    void *remove_cb_data;
    otk_cb_t remove_cb;
};

struct otk_list_s {
    otk_widget_t widget;
    otk_widget_t *scrollbar;

    // the entries
    l_list_t *entries;
    // number of entries
    int num_entries;
    // number of selected entries
    int num_selected;
    // number of visible entries
    int num_visible;
    // the number of the entry at the top of the visible list
    int first_visible;

    // may the user select entries
    int allow_select;

    // size and position of the entries
    int entry_x;
    int entry_y;
    int entry_height;
    int entry_width;

    int entry_spacing;


    // data that is sent in every callback
    void *cb_data;

    // we call this callback when the visible entry changes 
    // (only used for selectors)
    void *entry_changed_cb_data;
    otk_int_set_cb_t entry_changed_cb;
};

struct otk_label_s {
    otk_widget_t widget;

    char *text;

    // upcall to get the new text
    otk_label_uc_t update_cb;
    void *update_cb_data;
};

struct otk_scrollbar_s {
    otk_widget_t widget;

    int position;
    int length;

    otk_cb_t cb_up;
    otk_cb_t cb_down;

    otk_int_set_cb_t cb_click;

    void *cb_data;
};

struct otk_window_s {
    otk_widget_t widget;

    bool keep;
    bool fill;
    bool show_border;

    otk_widget_t *focus_ptr;

    l_list_t *subs;
};

struct otk_s {
    odk_t *odk;

    otk_window_t *window;
    l_list_t *windows;

    int textpalette_label;
    int textpalette_button[3];  // 1 is focused, 2 disabled

    int update_job;

    pthread_mutexattr_t draw_mutex_attr;
    pthread_mutex_t draw_mutex;
};

extern oxine_t *oxine;

/* 
 * ***************************************************************************
 * Name:            trunc_text_to_width
 * Access:          private
 *
 * Description:     Checks the width of text and truncates the text if
 *                  necessary.
 * ***************************************************************************
 */
static char *
trunc_text_to_width (odk_t * odk, const char *orig, int max_width)
{
    int textwidth;
    int textheight;
    odk_get_text_size (odk, orig, &textwidth, &textheight);

    char *text = ho_strdup (orig);
    if (textwidth > max_width) {
        int i = strlen (text) - 4;

        if (i < 0)
            return text;

        text[i] = text[i + 1] = text[i + 2] = '.';
        text[i + 3] = 0;

        odk_get_text_size (odk, text, &textwidth, &textheight);
        while ((textwidth > max_width) && (i > 0)) {
            i--;
            text[i] = '.';
            text[i + 3] = 0;
            odk_get_text_size (odk, text, &textwidth, &textheight);
        }
    }

    return text;
}


/* 
 * ***************************************************************************
 * Name:            is_correct_widget
 * Access:          private
 *
 * Description:     Checks if the expected type and the real type of a widget
 *                  are the same.
 * ***************************************************************************
 */
static int
is_correct_widget (otk_widget_t * widget, int expected)
{
    if (!widget)
        return 0;

    if (widget->type != expected)
        return 0;

    return 1;
}


/* 
 * ***************************************************************************
 * Name:            widget_remove
 * Access:          private
 *
 * Description:     Removes a widget from the window.
 * ***************************************************************************
 */
static void
widget_remove (otk_t * otk, otk_widget_t * widget)
{
    if (otk->window->focus_ptr == widget)
        otk->window->focus_ptr = NULL;
    l_list_remove (otk->window->subs, widget);
}


/* 
 * ***************************************************************************
 * Name:            widget_destroy
 * Access:          private
 *
 * Description:     Destroys a widget.
 * ***************************************************************************
 */
static void
widget_destroy (void *data)
{
    otk_widget_t *widget = (otk_widget_t *) data;
    if (widget->font)
        ho_free (widget->font);
    widget->destroy (widget);
}


/* 
 * ***************************************************************************
 * Name:            widget_init
 * Access:          private
 *
 * Description:     Sets standard values for a widget.
 * ***************************************************************************
 */
static void
widget_init (otk_widget_t * widget, otk_t * otk,
             int type, int x, int y, int w, int h)
{
    widget->type = type;

    widget->otk = otk;
    widget->odk = otk->odk;

    widget->x = otk->window->widget.x + x;
    widget->y = otk->window->widget.y + y;
    widget->w = w;
    widget->h = h;

    widget->selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;

    widget->is_focused = false;
    widget->is_enabled = true;
    widget->is_visible = true;

    widget->need_update = false;
    widget->need_repaint = false;

    widget->font = ho_strdup ("sans");
    widget->fontsize = 20;
    widget->alignment = ODK_ALIGN_LEFT | ODK_ALIGN_VCENTER;

    widget->draw = NULL;
    widget->destroy = NULL;

    widget->focus_enter_cb = NULL;
    widget->focus_enter_cb_data = NULL;

    widget->focus_leave_cb = NULL;
    widget->focus_leave_cb_data = NULL;

    l_list_append (widget->otk->window->subs, widget);
}


/* 
 * ***************************************************************************
 * Name:            widget_find_xy
 * Access:          private
 *
 * Description:     Returns the widget at (x/y).
 * ***************************************************************************
 */
static otk_widget_t *
widget_find_xy (otk_t * otk, int x, int y)
{
    if (!otk->window)
        return NULL;

    otk_widget_t *found = NULL;
    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget->x <= x) && (widget->y <= y)
            && ((widget->x + widget->w) >= x)
            && ((widget->y + widget->h) >= y)
            && (widget->selectable & OTK_SELECTABLE_MOUSE))
            found = widget;

        widget = l_list_next (otk->window->subs, widget);
    }

    return found;
}

/* 
 * ***************************************************************************
 * Name:            widget_get_first_focus
 * Access:          private
 *
 * Description:     Searches for the first widget to set the focus to.
 * ***************************************************************************
 */
static otk_widget_t *
widget_get_first_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget->selectable & OTK_SELECTABLE_KEY)
            && (widget->is_visible)
            && (widget->is_enabled)) {
            return widget;
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            widget_get_prev_focus
 * Access:          private
 *
 * Description:     Searches for the prev widget to pass the focus to.
 * ***************************************************************************
 */
static otk_widget_t *
widget_get_prev_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *prev_widget = cur_widget;

    assert (cur_widget);

    do {
        /* get the prev widget */
        prev_widget = l_list_prev (otk->window->subs, prev_widget);
        /* check if we're at the end of the list */
        if (!prev_widget)
            prev_widget = l_list_last (otk->window->subs);
        /* check if we've run around once */
        if (prev_widget == cur_widget)
            return NULL;
    } while (!(prev_widget->is_visible)
             || !(prev_widget->is_enabled)
             || !(prev_widget->selectable & OTK_SELECTABLE_KEY)
             || (cur_widget->type == OTK_WIDGET_LISTENTRY
                 && prev_widget->type == OTK_WIDGET_LISTENTRY
                 && (((otk_listentry_t *) cur_widget)->list ==
                     ((otk_listentry_t *) prev_widget)->list)));

    return prev_widget;
}


/* 
 * ***************************************************************************
 * Name:            widget_get_next_focus
 * Access:          private
 *
 * Description:     Searches for the next widget to pass the focus to.
 * ***************************************************************************
 */
static otk_widget_t *
widget_get_next_focus (otk_t * otk)
{
    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *next_widget = cur_widget;

    assert (cur_widget);

    do {
        /* get the next widget */
        next_widget = l_list_next (otk->window->subs, next_widget);
        /* check if we're at the end of the list */
        if (!next_widget)
            next_widget = l_list_first (otk->window->subs);
        /* check if we've run around once */
        if (next_widget == cur_widget)
            return NULL;
    } while (!(next_widget->is_visible)
             || !(next_widget->is_enabled)
             || !(next_widget->selectable & OTK_SELECTABLE_KEY)
             || (cur_widget->type == OTK_WIDGET_LISTENTRY
                 && next_widget->type == OTK_WIDGET_LISTENTRY
                 && (((otk_listentry_t *) cur_widget)->list ==
                     ((otk_listentry_t *) next_widget)->list)));

    return next_widget;
}


/* 
 * ***************************************************************************
 * Name:            get_distance
 * Access:          private
 *
 * Description:     Returns the distance between the widgets base and target.
 * ***************************************************************************
 */
static int
get_distance (otk_widget_t * base, otk_widget_t * target)
{
    int x1 = (base->x + base->w / 2) / 10;
    int y1 = (base->y + base->h / 2) / 10;
    int x2 = (target->x + target->w / 2) / 10;
    int y2 = (target->y + target->h / 2) / 10;
    int dist = (int) sqrt (pow ((x1 - x2), 2) + pow ((y1 - y2), 2));

    return dist;
}


/* 
 * ***************************************************************************
 * Name:            get_vertical_angle
 * Access:          private
 *
 * Description:     Returns the vertical angle between the widgets base and
 *                  target.
 * ***************************************************************************
 */
static double
get_vertical_angle (otk_widget_t * base, otk_widget_t * target)
{
    double a = 0;

    int x1 = base->x + base->w / 2;
    int y1 = base->y + base->h / 2;
    int x2 = target->x + target->w / 2;
    int y2 = target->y + target->h / 2;
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);

    if (x != 0)
        a = atan ((double) y / (double) x) + 1;

    return a;
}


/* 
 * ***************************************************************************
 * Name:            get_horizontal_angle
 * Access:          private
 *
 * Description:     Returns the horizonal angle between the widgets base and
 *                  target.
 * ***************************************************************************
 */
static double
get_horizontal_angle (otk_widget_t * base, otk_widget_t * target)
{
    double a = 0;

    int x1 = base->x + base->w / 2;
    int y1 = base->y + base->h / 2;
    int x2 = target->x + target->w / 2;
    int y2 = target->y + target->h / 2;
    int x = ABS (x1 - x2);
    int y = ABS (y1 - y2);

    if (y != 0)
        a = atan ((double) x / (double) y) + 1;

    return a;
}


/* 
 * ***************************************************************************
 * Name:            widget_find_neighbour
 * Access:          private
 *
 * Description:     Searches for the nearest neighbour in the given direction.
 * ***************************************************************************
 */
static otk_widget_t *
widget_find_neighbour (otk_t * otk, int direction)
{
    // FIXME: This is not really implemented well. This does not always select
    // the correct widget.

    if (!otk->window)
        return NULL;

    if (l_list_length (otk->window->subs) == 0)
        return NULL;

    if (!otk->window->focus_ptr)
        return widget_get_first_focus (otk);

    float best_angle = 0;
    float best_ratio = 0;

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *new_widget = NULL;

    int x = cur_widget->x + cur_widget->w / 2;
    int y = cur_widget->y + cur_widget->h / 2;

    otk_widget_t *widget = l_list_first (otk->window->subs);
    while (widget) {
        if ((widget != cur_widget)
            && (widget->is_visible)
            && (widget->is_enabled)
            && (widget->selectable & OTK_SELECTABLE_KEY)) {

            int nx = widget->x + widget->w / 2;
            int ny = widget->y + widget->h / 2;

            float angle = 0;
            float ratio = 0;

            switch (direction) {
            case DIRECTION_UP:
                if ((y - ny) < 0)
                    break;
                angle = get_horizontal_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case DIRECTION_DOWN:
                if ((ny - y) < 0)
                    break;
                angle = get_horizontal_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case DIRECTION_LEFT:
                if ((x - nx) < 0)
                    break;
                angle = get_vertical_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            case DIRECTION_RIGHT:
                if ((nx - x) < 0)
                    break;
                angle = get_vertical_angle (cur_widget, widget);
                ratio = angle * get_distance (cur_widget, widget);
                break;
            }

            if (((angle == 1.00)
                 && ((best_angle == 0)
                     || (best_angle > angle)))
                || ((ratio > 0)
                    && ((best_ratio == 0)
                        || (ratio < best_ratio)))) {
                best_angle = angle;
                best_ratio = ratio;
                new_widget = widget;
            }
        }
        widget = l_list_next (otk->window->subs, widget);
    }

    return new_widget;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_font
 * Access:          public
 *
 * Description:     Set font and fontsize of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_font (otk_widget_t * widget, const char *font, int fontsize)
{
    ho_free (widget->font);
    widget->font = ho_strdup (font);
    widget->fontsize = fontsize;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_alignment
 * Access:          public
 *
 * Description:     Sets the text alignment of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_alignment (otk_widget_t * widget, int alignment)
{
    widget->alignment = alignment;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_enabled
 * Access:          public
 *
 * Description:     Sets the enabled status of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_enabled (otk_widget_t * widget, bool enabled)
{
    widget->is_enabled = enabled;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_visible
 * Access:          public
 *
 * Description:     Sets the visibility status of a widget.
 * ***************************************************************************
 */
void
otk_widget_set_visible (otk_widget_t * widget, bool visible)
{
    widget->is_visible = visible;
}


/* 
 * ***************************************************************************
 * Name:            otk_widget_set_selectable
 * Access:          public
 *
 * Description:     Sets how the widget is selectable
 * ***************************************************************************
 */
void
otk_widget_set_selectable (otk_widget_t * widget, int selectable)
{
    widget->selectable = selectable;
}


/* 
 * ***************************************************************************
 * Name:            otk_send_event
 * Access:          public
 *
 * Description:     Sends an event.
 * ***************************************************************************
 */
void
otk_send_event (otk_t * otk, oxine_event_t * ev)
{
    odk_oxine_event_send (otk->odk, ev);
}

/*
 * ***********************************************************************************
 * Button-Widget 
 * ***********************************************************************************
 */
static void
button_destroy (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    widget_remove (this->otk, this);

    if (button->text)
        ho_free (button->text);
    if (button->bitmap)
        ho_free (button->bitmap);
    if (button->image_mrl)
        ho_free (button->image_mrl);
    if (button->image_focus_mrl)
        ho_free (button->image_focus_mrl);

    ho_free (button);
}

static void
button_draw (otk_widget_t * this)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    if (button->image_mrl && button->image_focus_mrl) {
        odk_osd_image_t *new = NULL;

        if (this->is_focused) {
            new = odk_osd_draw_image (oxine->odk,
                                      button->image_focus_mrl,
                                      this->x, this->y, this->w, this->h,
                                      ODK_ALIGN_LEFT | ODK_ALIGN_TOP, 0, 0);
        } else {
            new = odk_osd_draw_image (oxine->odk,
                                      button->image_mrl,
                                      this->x, this->y, this->w, this->h,
                                      ODK_ALIGN_LEFT | ODK_ALIGN_TOP, 0, 0);
        }

        /* If the image could not be drawn, we free the MRL, so that we
         * don't try to draw it again. */
        if (!new) {
            ho_free (button->image_mrl);
            button->image_mrl = NULL;
            ho_free (button->image_focus_mrl);
            button->image_focus_mrl = NULL;
        } else {
            return;
        }
    }

    int palette = this->is_enabled ? this->is_focused : 2;
    int color_base = this->otk->textpalette_button[palette];
    int background_color = color_base + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = color_base + OSD_TEXT_PALETTE_FOREGROUND;

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h, background_color, true);

    if (button->text) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);
        char *txt = trunc_text_to_width (this->odk, button->text,
                                         button->widget.w - 10);

        int x = this->x + this->w / 2;
        if (this->alignment & OTK_ALIGN_RIGHT) {
            x = this->x + this->w - 5;
        } else if (this->alignment & OTK_ALIGN_LEFT) {
            x = this->x + 5;
        }
        // we ignore any other alignment than VCENTER
        this->alignment &= ~OTK_ALIGN_TOP;
        this->alignment &= ~OTK_ALIGN_BOTTOM;
        this->alignment |= OTK_ALIGN_VCENTER;

        odk_draw_text (this->odk, x,
                       this->y + this->h / 2,
                       txt, this->alignment, color_base);

        ho_free (txt);
    }

    else if (button->bitmap) {
        uint8_t palette_map[2];
        palette_map[0] = background_color;
        palette_map[1] = foreground_color;

        odk_draw_bitmap (this->odk, button->bitmap,
                         this->x + this->w / 2,
                         this->y + this->h / 2, 20, 20, palette_map);
    }

    this->need_repaint = false;
}

static otk_button_t *
button_new_basic (otk_t * otk, int x, int y, int w, int h,
                  otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = ho_new (otk_button_t);

    widget_init ((otk_widget_t *) button, otk, OTK_WIDGET_BUTTON, x, y, w, h);
    button->widget.draw = button_draw;
    button->widget.destroy = button_destroy;

    button->cb = cb;
    button->cb_data = cb_data;

    return button;
}

otk_widget_t *
otk_text_button_new (otk_t * otk, int x, int y, int w, int h,
                     const char *text, otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);

    if (text)
        button->text = ho_strdup (text);
    else
        button->text = NULL;

    return (otk_widget_t *) button;
}

otk_widget_t *
otk_bitmap_button_new (otk_t * otk, int x, int y, int w, int h,
                       uint8_t * bitmap, otk_cb_t cb, void *cb_data)
{
    otk_button_t *button = button_new_basic (otk, x, y, w, h, cb, cb_data);

    button->bitmap = bitmap;

    return (otk_widget_t *) button;
}

otk_widget_t *
otk_image_button_new (otk_t * otk, int x, int y, int w, int h,
                      const char *mrl, const char *focus_mrl,
                      const char *text_replace, const char *text_over,
                      otk_cb_t cb, void *cb_data)
{
    assert (otk);
    assert (mrl);
    assert (focus_mrl);

    otk_button_t *button = NULL;
    if ((access (mrl, R_OK) == 0) && (access (focus_mrl, R_OK) == 0)) {
        button = button_new_basic (otk, x, y, w, h, cb, cb_data);

        assert (mrl);
        assert (focus_mrl);

        button->image = NULL;
        button->image_mrl = ho_strdup (mrl);
        button->image_focus_mrl = ho_strdup (focus_mrl);
    }

    else if (text_replace) {
        button = (otk_button_t *) otk_text_button_new (otk, x, y, w, h,
                                                       text_replace, cb,
                                                       cb_data);
    }

    return (otk_widget_t *) button;
}

void
otk_button_set_text (otk_widget_t * this, const char *text)
{
    otk_button_t *button = (otk_button_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BUTTON))
        return;

    if (button->text)
        ho_free (button->text);

    if (text)
        button->text = ho_strdup (text);
    else
        button->text = NULL;

    this->need_repaint = true;
}


/*
 * ***********************************************************************************
 * Slider-Widget 
 * ***********************************************************************************
 */
static void
slider_destroy (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SLIDER))
        return;

    widget_remove (this->otk, this);

    if (slider->value_format)
        ho_free (slider->value_format);

    ho_free (slider);
}

static void
slider_draw (otk_widget_t * this)
{
    otk_slider_t *slider = (otk_slider_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SLIDER))
        return;

    int value = 0;
    if (slider->get_value_cb) {
        value = slider->get_value_cb (slider->get_value_cb_data);
    } else {
        value = slider->last_value;
    }

    /* We may get wrong values here sometimes. This is why we always save the
     * last valid value to reuse in such a case. */
    if (value < slider->min_value) {
        debug ("value is outside of valid range (%d-%d): %d",
               slider->min_value, slider->max_value, value);
        value = slider->last_value;
    } else if (value > slider->max_value) {
        debug ("value is outside of valid range (%d-%d): %d",
               slider->min_value, slider->max_value, value);
        value = slider->last_value;
    }
    slider->last_value = value;

    int max = slider->max_value - slider->min_value;
    int val = value - slider->min_value;
    int perc_value = round ((double) val / ((double) max / 100.0));

    /* Just to make sure we don't try to draw anything outside the displayable
     * range. */
    if (perc_value < 0)
        perc_value = 0;
    else if (perc_value > 100)
        perc_value = 100;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->is_focused] +
                   OSD_TEXT_PALETTE_BACKGROUND, 1);

    if (slider->value_format) {
        char str_val[10];
        snprintf (str_val, 10, slider->value_format, perc_value);
        odk_osd_set_font (this->odk, this->font, this->fontsize);

        char *txt = trunc_text_to_width (this->odk, str_val, this->w);
        odk_draw_text (this->odk, this->x + (this->w / 2),
                       this->y + (this->h / 2), txt,
                       ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER,
                       this->otk->textpalette_label);
        ho_free (txt);
    }

    switch (slider->direction) {
    case OTK_SLIDER_HORIZONTAL:
        {
            int w = slider->marker_w;
            int h = slider->marker_h;

            int b = (this->h - h) / 2;
            int m = this->x + b + (w / 2) +
                (perc_value * (this->w - (2 * b) - w)) / 100;

            int y0 = this->y + b;
            int y1 = y0 + h;

            int x0 = m - w / 2;
            int x1 = m + w / 2;

            if (slider->simple_marker) {
                odk_draw_rect (this->odk, x0, y0, x1, y1,
                               this->otk->textpalette_button[this->
                                                             is_focused] +
                               OSD_TEXT_PALETTE_FOREGROUND, 1);
            } else {
                int x = this->x + b;
                for (; x < m; x += 10) {
                    odk_draw_rect (this->odk, x, y0, x + 8, y1,
                                   this->otk->textpalette_button[this->
                                                                 is_focused] +
                                   OSD_TEXT_PALETTE_FOREGROUND, 1);
                }
                for (; x < this->x + this->w - b; x += 10) {
                    odk_draw_rect (this->odk, x, y0, x + 8, y1,
                                   this->otk->textpalette_button[!this->
                                                                 is_focused] +
                                   OSD_TEXT_PALETTE_BACKGROUND, 1);
                }
            }
        }
        break;
    case OTK_SLIDER_VERTICAL:
        {
            int w = slider->marker_w;
            int h = slider->marker_h;

            int b = (this->w - w) / 2;
            int m = this->y + this->h - b - (h / 2) -
                (perc_value * (this->h - (2 * b) - h)) / 100;

            int x0 = this->x + b;
            int x1 = x0 + w;

            int y0 = m - h / 2;
            int y1 = m + h / 2;

            if (y0 < this->y + b)
                y0 = this->y + b;
            if (y1 > this->y + this->h - b)
                y1 = this->y + this->h - b;

            if (slider->simple_marker) {
                odk_draw_rect (this->odk, x0, y0, x1, y1,
                               this->otk->textpalette_button[this->
                                                             is_focused] +
                               OSD_TEXT_PALETTE_FOREGROUND, 1);
            } else {
                int y = this->y + b;
                for (; y < m; y += 10) {
                    odk_draw_rect (this->odk, x0, y, x1, y + 8,
                                   this->otk->textpalette_button[this->
                                                                 is_focused] +
                                   OSD_TEXT_PALETTE_FOREGROUND, 1);
                }
                for (; y < m; y += 10) {
                    odk_draw_rect (this->odk, x0, y, x1, y + 8,
                                   this->otk->textpalette_button[!this->
                                                                 is_focused] +
                                   OSD_TEXT_PALETTE_BACKGROUND, 1);
                }
            }
        }
    }
}

static void
slider_minus_cb (void *slider_p)
{
    otk_slider_t *slider = (otk_slider_t *) slider_p;
    if (slider->get_value_cb && slider->set_value_cb) {
        int value =
            slider->get_value_cb (slider->get_value_cb_data) -
            slider->min_step;
        if (value < slider->min_value)
            value = slider->min_value;
        slider->set_value_cb (slider->set_value_cb_data, value);
    }
}

static void
slider_plus_cb (void *slider_p)
{
    otk_slider_t *slider = (otk_slider_t *) slider_p;
    if (slider->get_value_cb && slider->set_value_cb) {
        int value =
            slider->get_value_cb (slider->get_value_cb_data) +
            slider->min_step;
        if (value > slider->max_value)
            value = slider->max_value;
        slider->set_value_cb (slider->set_value_cb_data, value);
    }
}

static void
slider_reset_cb (void *slider_p)
{
    otk_slider_t *slider = (otk_slider_t *) slider_p;
    if (slider->set_value_cb) {
        slider->set_value_cb (slider->set_value_cb_data,
                              slider->default_value);
    }
}

otk_widget_t *
otk_slider_new (otk_t * otk, int x, int y, int w, int h,
                int marker_w, int marker_h, bool simple_marker,
                int direction, const char *value_format,
                bool with_buttons, bool with_reset_button,
                int default_value, int min_value, int max_value, int min_step,
                otk_int_get_cb_t get_value_cb, void *get_value_cb_data,
                otk_int_set_cb_t set_value_cb, void *set_value_cb_data)
{
    otk_slider_t *slider = ho_new (otk_slider_t);

    assert ((direction == OTK_SLIDER_HORIZONTAL)
            || (direction == OTK_SLIDER_VERTICAL));

    widget_init ((otk_widget_t *) slider, otk, OTK_WIDGET_SLIDER, x, y, w, h);
    slider->widget.selectable = OTK_SELECTABLE_MOUSE;
    slider->widget.draw = slider_draw;
    slider->widget.destroy = slider_destroy;

    slider->direction = direction;
    if (value_format)
        slider->value_format = ho_strdup (value_format);
    else
        slider->value_format = NULL;

    slider->marker_w = marker_w;
    slider->marker_h = marker_h;
    slider->simple_marker = simple_marker;

    slider->default_value = default_value;
    slider->min_value = min_value;
    slider->max_value = max_value;
    slider->min_step = min_step;

    slider->get_value_cb = get_value_cb;
    slider->get_value_cb_data = get_value_cb_data;

    slider->set_value_cb = set_value_cb;
    slider->set_value_cb_data = set_value_cb_data;

    if (slider->direction == OTK_SLIDER_HORIZONTAL) {
        int x1 = x;
        int x2 = x + w - 30;
        int x3 = x + w - 30;

        if (with_buttons) {
            slider->widget.x += 32;
            slider->widget.w -= 64;
        }
        if (with_reset_button) {
            x2 -= 32;
            slider->widget.w -= 32;
        }

        if (with_buttons) {
            otk_bitmap_button_new (oxine->otk, x1, y, 30, h,
                                   odk_get_bitmap (BITMAP_MINUS),
                                   slider_minus_cb, slider);
            otk_bitmap_button_new (oxine->otk, x2, y, 30, h,
                                   odk_get_bitmap (BITMAP_PLUS),
                                   slider_plus_cb, slider);
        }
        if (with_reset_button) {
            otk_bitmap_button_new (oxine->otk, x3, y, 30, h,
                                   odk_get_bitmap (BITMAP_HOME),
                                   slider_reset_cb, slider);
        }
    }

    else if (slider->direction == OTK_SLIDER_VERTICAL) {
        int y1 = y;
        int y2 = y + h - 30;
        int y3 = y + h - 30;

        if (with_buttons) {
            slider->widget.y += 32;
            slider->widget.h -= 64;
        }
        if (with_reset_button) {
            y2 -= 32;
            slider->widget.h -= 32;
        }

        if (with_buttons) {
            otk_bitmap_button_new (oxine->otk, x, y1, w, 30,
                                   odk_get_bitmap (BITMAP_PLUS),
                                   slider_plus_cb, slider);
            otk_bitmap_button_new (oxine->otk, x, y2, w, 30,
                                   odk_get_bitmap (BITMAP_MINUS),
                                   slider_minus_cb, slider);
        }
        if (with_reset_button) {
            otk_bitmap_button_new (oxine->otk, x, y3, w, 30,
                                   odk_get_bitmap (BITMAP_HOME),
                                   slider_reset_cb, slider);
        }
    }

    return (otk_widget_t *) slider;
}

/*
 * ***********************************************************************************
 * Checkbox-Widget 
 * ***********************************************************************************
 */
static void
checkbox_destroy (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_CHECKBOX))
        return;

    widget_remove (this->otk, this);

    if (checkbox->bitmap)
        ho_free (checkbox->bitmap);

    ho_free (checkbox);
}

static void
checkbox_draw (otk_widget_t * this)
{
    otk_checkbox_t *checkbox = (otk_checkbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_CHECKBOX))
        return;

    if (this->is_focused) {
        odk_draw_rect (this->odk, this->x, this->y,
                       this->x + this->w, this->y + this->h,
                       this->otk->textpalette_button[this->is_focused] +
                       OSD_TEXT_PALETTE_BACKGROUND, true);
    }
    {
        odk_draw_rect (this->odk, this->x, this->y,
                       this->x + this->w, this->y + this->h,
                       this->otk->textpalette_button[this->is_focused] +
                       OSD_TEXT_PALETTE_FOREGROUND, false);
    }

    if (checkbox->is_checked) {
        uint8_t palette_map[2];
        palette_map[0] = 0;
        palette_map[1] = 0;

        if (this->is_focused) {
            palette_map[0] = this->otk->textpalette_button[this->is_focused] +
                OSD_TEXT_PALETTE_BACKGROUND;
        }
        {
            palette_map[1] = this->otk->textpalette_button[this->is_focused] +
                OSD_TEXT_PALETTE_FOREGROUND;
        }

        odk_draw_bitmap (this->odk, checkbox->bitmap,
                         this->x + this->w / 2,
                         this->y + this->h / 2, 20, 20, palette_map);
    }
}

otk_widget_t *
otk_checkbox_new (otk_t * otk, int x, int y, int w, bool is_checked,
                  otk_cb_t cb, void *cb_data)
{
    otk_checkbox_t *checkbox = ho_new (otk_checkbox_t);

    widget_init ((otk_widget_t *) checkbox, otk,
                 OTK_WIDGET_CHECKBOX, x, y, w, w);
    checkbox->widget.draw = checkbox_draw;
    checkbox->widget.destroy = checkbox_destroy;

    checkbox->is_checked = is_checked;
    checkbox->bitmap = odk_get_bitmap (BITMAP_CHECK);

    checkbox->cb = cb;
    checkbox->cb_data = cb_data;

    return (otk_widget_t *) checkbox;
}

/*
 * ***********************************************************************************
 * Scrollbar-Widget
 * ***********************************************************************************
 */
static void
scrollbar_destroy (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    widget_remove (this->otk, this);

    ho_free (scrollbar);
}

static void
scrollbar_draw (otk_widget_t * this)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    odk_draw_rect (this->odk, this->x, this->y,
                   this->x + this->w, this->y + this->h,
                   this->otk->textpalette_button[this->is_focused] +
                   OSD_TEXT_PALETTE_BACKGROUND, true);

    int top_y = this->y + 2;
    top_y += (scrollbar->position * (this->h - 4)) / 100;
    int bot_y = this->y + 2;
    bot_y += ((scrollbar->position + scrollbar->length)
              * (this->h - 4)) / 100;

    odk_draw_rect (this->odk, this->x + 5, top_y,
                   this->x + this->w - 5, bot_y,
                   this->otk->textpalette_button[this->is_focused] +
                   OSD_TEXT_PALETTE_FOREGROUND, true);
}

otk_widget_t *
otk_scrollbar_new (otk_t * otk, int x, int y, int w, int h,
                   otk_int_set_cb_t cb_click, otk_cb_t cb_up,
                   otk_cb_t cb_down, void *cb_data)
{
    otk_scrollbar_t *scrollbar = ho_new (otk_scrollbar_t);

    widget_init ((otk_widget_t *) scrollbar, otk,
                 OTK_WIDGET_SCROLLBAR, x, y + w, w, h - (2 * w));
    scrollbar->widget.selectable = OTK_SELECTABLE_MOUSE;
    scrollbar->widget.draw = scrollbar_draw;
    scrollbar->widget.destroy = scrollbar_destroy;

    scrollbar->cb_click = cb_click;
    scrollbar->cb_up = cb_up;
    scrollbar->cb_down = cb_down;
    scrollbar->cb_data = cb_data;
    scrollbar->position = 0;
    scrollbar->length = 100;

    otk_widget_t *b;
    b = otk_bitmap_button_new (otk, x, y, w, w,
                               odk_get_bitmap (BITMAP_SIMPLE_ARROW_UP), cb_up,
                               cb_data);
    b->selectable = OTK_SELECTABLE_MOUSE;
    b = otk_bitmap_button_new (otk, x, y + h - w, w, w,
                               odk_get_bitmap (BITMAP_SIMPLE_ARROW_DOWN),
                               cb_down, cb_data);
    b->selectable = OTK_SELECTABLE_MOUSE;

    return (otk_widget_t *) scrollbar;
}

void
otk_scrollbar_set (otk_widget_t * this, int position, int length)
{
    otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_SCROLLBAR))
        return;

    if (position < 0)
        position = 0;
    if (position > 100)
        position = 100;
    scrollbar->position = position;

    if (length < 0)
        length = 0;
    if (length > 100)
        length = 100;
    scrollbar->length = length;
}


/*
 * ***********************************************************************************
 * Border-Widget
 * ***********************************************************************************
 */
static void
border_draw (otk_widget_t * this)
{
    if (!is_correct_widget (this, OTK_WIDGET_BORDER)
        && !is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h,
                   this->otk->textpalette_label + OSD_TEXT_PALETTE_FOREGROUND,
                   false);
}


static void
border_destroy (otk_widget_t * this)
{
    otk_border_t *border = (otk_border_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_BORDER))
        return;

    widget_remove (this->otk, this);

    ho_free (border);
}


otk_widget_t *
otk_border_new (otk_t * otk, int x, int y, int w, int h)
{
    otk_border_t *border = ho_new (otk_border_t);

    widget_init ((otk_widget_t *) border, otk, OTK_WIDGET_BORDER, x, y, w, h);
    border->widget.selectable = OTK_SELECTABLE_NONE;
    border->widget.draw = border_draw;
    border->widget.destroy = border_destroy;

    return (otk_widget_t *) border;
}


/*
 * ***********************************************************************************
 * List-Widget Callback Stuff
 * ***********************************************************************************
 */
static void listentries_adapt (otk_list_t * list);

static void
list_scroll_down (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible++;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb)
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
}

static void
list_scroll_up (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible--;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb)
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
}

static int
list_set_position (void *data, int position)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible = position * list->num_entries / 100;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    return position;
}

/*
 * ***********************************************************************************
 * ListEntry-Widget
 * ***********************************************************************************
 */
static void
listentry_destroy (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LISTENTRY))
        return;

    widget_remove (this->otk, this);

    if (listentry->text)
        ho_free (listentry->text);

    ho_free (listentry);
}

static void
listentry_draw (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LISTENTRY))
        return;

    if (!listentry->is_visible)
        return;

    int palette = this->is_enabled ? this->is_focused : 2;
    int color_base = this->otk->textpalette_button[palette];
    int background_color = color_base + OSD_TEXT_PALETTE_BACKGROUND;

    if (!this->is_focused && this->is_enabled && !listentry->is_selected)
        background_color = 0;

    if (background_color) {
        odk_draw_rect (this->odk, this->x, this->y,
                       this->x + this->w, this->y + this->h,
                       background_color, true);
    }

    if (listentry->text) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);
        char *txt = trunc_text_to_width (this->odk, listentry->text,
                                         listentry->widget.w - 10);

        int x = this->x + this->w / 2;
        if (this->alignment & OTK_ALIGN_RIGHT) {
            x = this->x + this->w - 5;
        } else if (this->alignment & OTK_ALIGN_LEFT) {
            x = this->x + 5;
        }
        // we ignore any other alignment than VCENTER
        this->alignment &= ~OTK_ALIGN_TOP;
        this->alignment &= ~OTK_ALIGN_BOTTOM;
        this->alignment |= OTK_ALIGN_VCENTER;

        odk_draw_text (this->odk, x, this->y + this->h / 2,
                       txt, this->alignment, color_base);

        ho_free (txt);
    }
}


otk_widget_t *
otk_listentry_new (otk_widget_t * this, const char *text,
                   otk_cb_t activate_cb, void *activate_cb_data,
                   otk_cb_t select_cb, void *select_cb_data,
                   otk_cb_t remove_cb, void *remove_cb_data)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    otk_listentry_t *entry = ho_new (otk_listentry_t);

    widget_init ((otk_widget_t *) entry, this->otk,
                 OTK_WIDGET_LISTENTRY, list->entry_x, 0,
                 list->entry_width, list->entry_height);
    entry->widget.selectable = OTK_SELECTABLE_NONE;
    entry->widget.draw = listentry_draw;
    entry->widget.destroy = listentry_destroy;
    entry->widget.fontsize = list->widget.fontsize;

    entry->list = list;
    entry->activate_cb_data = activate_cb_data;
    entry->activate_cb = activate_cb;
    entry->select_cb_data = select_cb_data;
    entry->select_cb = select_cb;
    entry->remove_cb_data = remove_cb_data;
    entry->remove_cb = remove_cb;

    if (text)
        entry->text = ho_strdup (text);
    else
        entry->text = NULL;

    l_list_append (list->entries, entry);

    list->num_entries++;
    listentries_adapt (list);

    return (otk_widget_t *) entry;
}


void
otk_list_clear_selection (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->is_selected = false;
        entry = l_list_next (list->entries, entry);
    }

    list->num_selected = 0;
}


void
otk_list_select_all (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->is_selected = true;
        entry = l_list_next (list->entries, entry);
    }

    list->num_selected = list->num_entries;
}


int
otk_list_get_selected_count (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return 0;

    return list->num_selected;
}

int *
otk_list_get_selected_pos (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        int *selected_entries = ho_new (list->num_selected * sizeof (int));

        otk_listentry_t *entry = l_list_first (list->entries);
        int pos = 0;
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = pos;
            }
            entry = l_list_next (list->entries, entry);
            pos++;
        }
        return selected_entries;
    }
    return NULL;
}

void **
otk_list_get_selected (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    *num_selected = 0;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        void **selected_entries =
            ho_malloc (list->num_selected * sizeof (void *));

        otk_listentry_t *entry = l_list_first (list->entries);
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = entry->select_cb_data;
            }
            entry = l_list_next (list->entries, entry);
        }
        return selected_entries;
    }
    return NULL;
}


static void
otk_listentry_set_pos (otk_listentry_t * entry, int pos)
{
    if ((pos <= 0) || (pos > entry->list->num_visible)) {
        entry->is_visible = 0;
        entry->widget.selectable = OTK_SELECTABLE_NONE;
        entry->is_first = 0;
        entry->is_last = 0;
        return;
    }
    entry->pos_visible_list = pos;
    entry->is_first = 0;
    entry->is_last = 0;

    entry->widget.y = entry->list->entry_y
        + entry->list->entry_spacing * (pos - 1);
    entry->is_visible = 1;
    entry->widget.selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;

    /* we grab focus if nothing is selected */
    if (pos == 1) {
        entry->is_first = 1;
        if (!entry->widget.otk->window->focus_ptr) {
            otk_widget_set_focus ((otk_widget_t *) entry);
        }
    }
    if (pos == entry->list->num_visible || pos == entry->list->num_entries)
        entry->is_last = 1;
}


static void
listentries_adapt (otk_list_t * list)
{
    if ((list->num_entries - list->first_visible) < list->num_visible)
        list->first_visible = list->num_entries - list->num_visible;
    if (list->first_visible < 0)
        list->first_visible = 0;

    int i = 1;
    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->pos_complete_list = i;
        otk_listentry_set_pos (entry, i - list->first_visible);
        entry = l_list_next (list->entries, entry);
        i++;
    }

    if (list->num_entries && list->scrollbar) {
        otk_scrollbar_set (list->scrollbar,
                           list->first_visible * 100 / list->num_entries,
                           list->num_visible * 100 / list->num_entries);
    }
}

/*
 * ***********************************************************************************
 * List-Widget
 * ***********************************************************************************
 */
static void
list_destroy (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    widget_remove (this->otk, this);

    l_list_free (list->entries, widget_destroy);

    ho_free (list);
}


otk_widget_t *
otk_list_new (otk_t * otk, int x, int y, int w, int h,
              int entry_height, int entry_spacing,
              bool show_border, bool show_scrollbar,
              int allow_select, void *list_cb_data)
{
    otk_list_t *list = ho_new (otk_list_t);

    widget_init ((otk_widget_t *) list, otk, OTK_WIDGET_LIST, x, y, w, h);
    list->widget.selectable = OTK_SELECTABLE_NONE;
    if (show_border)
        list->widget.draw = border_draw;
    else
        list->widget.draw = NULL;
    list->widget.destroy = list_destroy;

    list->allow_select = allow_select;

    list->entry_x = list->widget.x;
    list->entry_y = list->widget.y;
    list->entry_width = list->widget.w;
    list->entry_height = entry_height;
    list->entry_spacing = entry_spacing;

    list->entries = l_list_new ();

    list->num_entries = 0;
    list->num_selected = 0;
    list->num_visible = (list->widget.h - 10) / (list->entry_spacing);

    list->first_visible = 0;

    list->cb_data = list_cb_data;
    list->entry_changed_cb = NULL;
    list->entry_changed_cb_data = NULL;

    list->scrollbar = NULL;

    if (show_scrollbar && (list->num_visible > 1)) {
        list->entry_width -= 35;
        list->scrollbar = otk_scrollbar_new (otk, x + w - 35, y + 5, 30,
                                             h - 10, list_set_position,
                                             list_scroll_up, list_scroll_down,
                                             list);
    } else if (show_scrollbar && (list->num_visible == 1)) {
        list->entry_width -= 70;
        otk_widget_t *b;
        b = otk_bitmap_button_new (otk, x + w - 35, y + 5, 30, 30,
                                   odk_get_bitmap (BITMAP_SIMPLE_ARROW_UP),
                                   list_scroll_up, list);
        b->selectable = OTK_SELECTABLE_MOUSE;
        b = otk_bitmap_button_new (otk, x + w - 70, y + 5, 30, 30,
                                   odk_get_bitmap (BITMAP_SIMPLE_ARROW_DOWN),
                                   list_scroll_down, list);
        b->selectable = OTK_SELECTABLE_MOUSE;
    }

    if (show_border) {
        list->entry_width -= 10;
        list->entry_x += 5;
        list->entry_y += 5;
    }

    return (otk_widget_t *) list;
}


int
otk_list_get_length (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    return l_list_length (list->entries);
}


int
otk_list_get_focus (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;
    otk_t *otk = this->otk;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return 0;

    if (!otk->window)
        return 0;

    if (!otk->window->focus_ptr)
        return 0;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if (((otk_widget_t *) current) == otk->window->focus_ptr)
            return i;
        current = l_list_next (list->entries, current);
        i++;
    }

    return 0;
}

void
otk_list_set_focus (otk_widget_t * this, int pos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    if (pos > list->num_entries - 1)
        pos = list->num_entries - 1;

    int i = 0;
    otk_widget_t *current = l_list_first (list->entries);
    while (current) {
        if (i == pos) {
            while (current && !current->is_enabled) {
                current = l_list_next (list->entries, current);
            }
            if (current) {
                otk_widget_set_focus (current);
            }
            break;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}

void
otk_list_set_pos (otk_widget_t * this, int newpos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    list->first_visible = newpos;
    listentries_adapt (list);
}

int
otk_list_get_pos (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return -1;

    return list->first_visible;
}

void
otk_clear_list (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    list->num_entries = 0;
    list->num_selected = 0;
    l_list_clear (list->entries, widget_destroy);
}

void
otk_list_set_selected (otk_widget_t * this, int pos, bool selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LIST))
        return;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if ((i == pos)
            && (((otk_widget_t *) current)->is_enabled)
            && (((otk_widget_t *) current)->is_visible)
            && (((otk_widget_t *) current)->selectable)) {
            if (!current->is_selected && selected)
                list->num_selected++;

            if (current->is_selected && !selected)
                list->num_selected--;

            current->is_selected = selected;
            return;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}

/*
 * ***********************************************************************************
 * Selector-Widget
 * ***********************************************************************************
 */
otk_widget_t *
otk_selector_new (otk_t * otk, int x, int y, int w,
                  otk_int_set_cb_t cb, void *cb_data)
{
    otk_list_t *list = (otk_list_t *) otk_list_new (otk, x, y, w, 40,
                                                    30, 30, true, true,
                                                    OTK_LIST_NO_SELECTION,
                                                    NULL);

    list->entry_changed_cb = cb;
    list->entry_changed_cb_data = cb_data;

    return (otk_widget_t *) list;
}

/*
 * ***********************************************************************************
 * Editbox-Widget
 * ***********************************************************************************
 */

/*
 * Returns the width of the current display text.
 */
static int
editbox_display_text_width (otk_editbox_t * editbox)
{
    int tw;
    int th;
    odk_osd_set_font (editbox->widget.odk, editbox->widget.font,
                      editbox->widget.fontsize);
    odk_get_text_size (editbox->widget.odk, editbox->display_text, &tw, &th);

    return tw;
}

static void
editbox_display_text_create (otk_editbox_t * editbox)
{
    if (editbox->display_text)
        ho_free (editbox->display_text);

    assert (editbox->first_visible + editbox->num_visible <=
            strlen (editbox->text));

    editbox->display_text = ho_malloc (editbox->num_visible + 1);
    strncpy (editbox->display_text, editbox->text + editbox->first_visible,
             editbox->num_visible);
    editbox->display_text[editbox->num_visible] = 0;
}

/* 
 * This does NOT change the first visible character. It only adjusts the
 * number of visible characters.
 */
static void
editbox_display_text_correct_length (otk_editbox_t * editbox)
{
    int mw = editbox->widget.w - 10;
    editbox->num_visible = strlen (editbox->text) - editbox->first_visible;
    editbox_display_text_create (editbox);
    while (editbox_display_text_width (editbox) > mw) {
        editbox->num_visible--;
        editbox_display_text_create (editbox);
    }
}

/*
 * This moves the visible area to the right by 5 characters if the current
 * position reaches the last visible character.
 */
static void
editbox_display_text_jump_right (otk_editbox_t * editbox)
{
    int mw = editbox->widget.w - 10;
    int last_visible = editbox->first_visible + editbox->num_visible;
    if (editbox->cur_position >= last_visible) {
        last_visible += 5;
        if (last_visible > editbox->cur_length)
            last_visible = editbox->cur_length;

        editbox->first_visible = last_visible - editbox->num_visible;
        editbox_display_text_create (editbox);
        while (editbox_display_text_width (editbox) > mw) {
            editbox->num_visible--;
            editbox->first_visible = last_visible - editbox->num_visible;
            editbox_display_text_create (editbox);
        }
    }
}

/*
 * This moves the visible area to the left by 5 characters if the current
 * position reaches the first visible character.
 */
static void
editbox_display_text_jump_left (otk_editbox_t * editbox)
{
    if (editbox->cur_position <= editbox->first_visible) {
        editbox->first_visible -= 5;
        if (editbox->first_visible < 0)
            editbox->first_visible = 0;
        editbox_display_text_correct_length (editbox);
    }
}


static void
editbox_calculate_carret_position (otk_editbox_t * editbox)
{
    /* Calculate the x position of the caret. */
    int len = editbox->cur_position - editbox->first_visible;
    char txt[len + 1];
    strncpy (txt, editbox->display_text, len);
    txt[len] = 0;
    int tw;
    int th;
    odk_osd_set_font (editbox->widget.otk->odk, editbox->widget.font,
                      editbox->widget.fontsize);
    odk_get_text_size (editbox->widget.otk->odk, txt, &tw, &th);
    editbox->carret_x = editbox->widget.x + 5 + tw;
}

static void
editbox_key_handler (otk_editbox_t * editbox, oxine_event_t * event)
{
    int mw = editbox->widget.w - 10;

    int key = event->data.key.key;
    int modifier = event->data.key.modifier;

    switch (key) {
    case XK_Return:
    case XK_Escape:
    case XK_Tab:
        return;
    default:
        break;
    }

    if ((key == XK_Delete)
        || ((modifier & ControlMask)
            && ((key == XK_d) || (key == XK_D)))) {
        if (editbox->cur_position < editbox->cur_length) {
            int i = editbox->cur_position;
            for (; i < editbox->cur_length; i++)
                editbox->text[i] = editbox->text[i + 1];
            editbox->cur_length--;
            editbox_display_text_correct_length (editbox);

            if (editbox->cb)
                editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
        }
    } else if (key == XK_BackSpace) {
        if (editbox->cur_position > 0) {

            if (editbox->cur_position < editbox->cur_length) {
                int len = strlen (editbox->text);
                int i = editbox->cur_position - 1;
                for (; i < editbox->cur_length; i++)
                    editbox->text[i] = editbox->text[i + 1];
                assert (len - 1 == strlen (editbox->text));
            } else {
                editbox->text[editbox->cur_position - 1] = 0;
            }
            editbox->cur_position--;
            editbox->cur_length--;

            editbox->num_visible--;
            editbox_display_text_create (editbox);
            editbox_display_text_jump_left (editbox);

            if (editbox->cb)
                editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
        }
    } else if ((modifier & ControlMask)
               && ((key == XK_k) || (key == XK_K))) {
        /* This deletes everything from the current 
         * position to the end of the line. */
        editbox->cur_length = editbox->cur_position;
        editbox->text[editbox->cur_length] = 0;

        editbox->num_visible = editbox->cur_position - editbox->first_visible;
        editbox_display_text_correct_length (editbox);

        if (editbox->cb)
            editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
    } else if ((modifier & ControlMask)
               && ((key == XK_u) || (key == XK_U))) {
        /* This deletes everything from the current 
         * position backwards to beginning of the line. */
        if (editbox->cur_position > 0) {
            int i = editbox->cur_position;
            for (; i <= editbox->cur_length; i++)
                editbox->text[i - editbox->cur_position] = editbox->text[i];
            editbox->cur_length -= editbox->cur_position;
            editbox->cur_position = 0;

            editbox->first_visible = 0;
            editbox_display_text_correct_length (editbox);

            if (editbox->cb)
                editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
        }
    } else if ((modifier & ControlMask)
               && ((key == XK_w) || (key == XK_W))) {
        /* This deletes everything from the current 
         * position backwards to the next space. */
        if (editbox->cur_position > 0) {
            int s = editbox->cur_position - 1;
            while ((s >= 0) && (editbox->text[s] == ' '))
                s--;
            while ((s >= 0) && (editbox->text[s] != ' '))
                s--;
            s++;

            int i = editbox->cur_position;
            int j = s;
            for (; i <= editbox->cur_length; i++, j++)
                editbox->text[j] = editbox->text[i];
            editbox->cur_length -= editbox->cur_position - s;
            editbox->cur_position = s;

            if (editbox->cur_position < editbox->first_visible)
                editbox->first_visible = editbox->cur_position;
            editbox_display_text_correct_length (editbox);
            editbox_display_text_jump_left (editbox);

            if (editbox->cb)
                editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
        }
    } else if ((key == XK_Left)
               || ((modifier & ControlMask)
                   && ((key == XK_b) || (key == XK_B)))) {
        if (editbox->cur_position > 0) {
            editbox->cur_position--;
            editbox_display_text_jump_left (editbox);
        }
    } else if ((key == XK_Right)
               || ((modifier & ControlMask)
                   && ((key == XK_f) || (key == XK_F)))) {
        if (editbox->cur_position < editbox->cur_length) {
            editbox->cur_position++;
            editbox_display_text_jump_right (editbox);
        }
    } else if ((key == XK_Home)
               || ((modifier & ControlMask)
                   && ((key == XK_a) || (key == XK_A)))) {
        editbox->cur_position = 0;

        editbox->first_visible = 0;
        editbox_display_text_correct_length (editbox);
    } else if ((key == XK_End)
               || ((modifier & ControlMask)
                   && ((key == XK_e) || (key == XK_E)))) {
        editbox->cur_position = editbox->cur_length;

        editbox->first_visible = 0;
        editbox->num_visible = strlen (editbox->text);
        editbox_display_text_create (editbox);
        while (editbox_display_text_width (editbox) > mw) {
            editbox->num_visible--;
            editbox->first_visible =
                editbox->cur_length - editbox->num_visible;
            editbox_display_text_create (editbox);
        }
    } else {
        if (editbox->cur_length < editbox->max_length) {

            int append = (editbox->cur_position == editbox->cur_length);
            if (append) {
                editbox->text[editbox->cur_length] = key;
            } else {
                int i = editbox->cur_length;
                for (; i > editbox->cur_position; i--)
                    editbox->text[i] = editbox->text[i - 1];
                editbox->text[editbox->cur_position] = key;
            }
            editbox->cur_length++;
            editbox->cur_position++;
            editbox->text[editbox->cur_length] = 0;
            if (append) {
                editbox->num_visible++;
                editbox_display_text_create (editbox);
                while (editbox_display_text_width (editbox) > mw) {
                    editbox->first_visible++;
                    editbox->num_visible--;
                    editbox_display_text_create (editbox);
                }
            } else {
                editbox_display_text_correct_length (editbox);
                editbox_display_text_jump_right (editbox);
            }

            if (editbox->cb)
                editbox->cb (editbox->cb_data, ho_strdup (editbox->text));
        }
    }

#ifdef DEBUG
    assert (editbox->cur_position >= editbox->first_visible);
    assert (editbox->cur_position <=
            editbox->first_visible + editbox->num_visible);
    assert (editbox->num_visible == strlen (editbox->display_text));

    assert (editbox->cur_position >= 0);
    assert (editbox->cur_position <= editbox->cur_length);
    assert (editbox->cur_length == strlen (editbox->text));
#endif

    editbox_calculate_carret_position (editbox);

    otk_draw (editbox->widget.otk);
}

void
otk_editbox_set_text (otk_widget_t * this, const char *text)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_EDITBOX))
        return;

    strncpy (editbox->text, text, editbox->max_length);

    if (strlen (text) < editbox->max_length)
        editbox->cur_length = strlen (text);
    else
        editbox->cur_length = editbox->max_length;

    editbox->cur_position = 0;
    editbox->carret_x = editbox->widget.x + 5;
    editbox->first_visible = 0;
    editbox_display_text_correct_length (editbox);

    this->need_repaint = true;
}

char *
otk_editbox_get_text (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_EDITBOX))
        return NULL;

    return ho_strdup (editbox->text);
}

static void
editbox_draw (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_EDITBOX))
        return;

    int palette = this->is_enabled ? this->is_focused : 2;
    int color_base = this->otk->textpalette_button[palette];
    int background_color = color_base + OSD_TEXT_PALETTE_BACKGROUND;
    int foreground_color = color_base + OSD_TEXT_PALETTE_FOREGROUND;

    if (this->is_focused)
        odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                       this->y + this->h, background_color, true);

    odk_draw_rect (this->odk, this->x, this->y, this->x + this->w,
                   this->y + this->h, foreground_color, false);

    odk_osd_set_font (this->odk, this->font, this->fontsize);
    odk_draw_text (this->odk, this->x + 5, this->y + (this->h / 2),
                   editbox->display_text, this->alignment, color_base);

    if (this->is_focused)
        odk_draw_line (this->odk, editbox->carret_x, this->y + 5,
                       editbox->carret_x, this->y + this->h - 5,
                       foreground_color);

    this->need_repaint = false;
}

static void
editbox_destroy (otk_widget_t * this)
{
    otk_editbox_t *editbox = (otk_editbox_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_EDITBOX))
        return;

    widget_remove (this->otk, this);

    if (editbox->text)
        ho_free (editbox->text);
    if (editbox->display_text)
        ho_free (editbox->display_text);

    ho_free (editbox);
}

otk_widget_t *
otk_editbox_new (otk_t * otk, int x, int y, int w, int h,
                 int max_length, otk_editbox_cb_t cb, void *cb_data)
{
    otk_editbox_t *editbox = ho_new (otk_editbox_t);

    widget_init ((otk_widget_t *) editbox, otk,
                 OTK_WIDGET_EDITBOX, x, y, w, h);
    editbox->widget.draw = editbox_draw;
    editbox->widget.destroy = editbox_destroy;

    editbox->text = ho_malloc (max_length + 1);
    editbox->display_text = ho_strdup ("");

    editbox->max_length = max_length;
    editbox->cur_length = 0;

    editbox->cur_position = 0;

    editbox->first_visible = 0;
    editbox->num_visible = 0;
    editbox->carret_x = editbox->widget.x + 5;

    editbox->cb = cb;
    editbox->cb_data = cb_data;

    return (otk_widget_t *) editbox;
}

/*
 * ***********************************************************************************
 * Label-Widget
 * ***********************************************************************************
 */
void
otk_label_set_upcall (otk_widget_t * this, otk_label_uc_t update_cb,
                      void *update_cb_data)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    label->update_cb = update_cb;
    label->update_cb_data = update_cb_data;
    label->widget.need_update = true;
}

void
otk_label_set_text (otk_widget_t * this, const char *text)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    if (strcmp (text, label->text) == 0)
        return;

    if (!text) {
        if (label->text)
            ho_free (label->text);
        label->text = NULL;
    } else {
        if (label->text && (strlen (text) == strlen (label->text))) {
            strcpy (label->text, text);
        } else {
            ho_free (label->text);
            label->text = ho_strdup (text);
        }
    }

    this->need_repaint = true;
}

static void
label_draw (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    if (!label->text)
        return;

    odk_osd_set_font (this->odk, this->font, this->fontsize);

    char *txt = trunc_text_to_width (this->odk, label->text, this->w);
    odk_draw_text (this->odk, this->x, this->y, txt,
                   this->alignment, this->otk->textpalette_label);
    ho_free (txt);

    this->need_repaint = false;
}

static void
label_destroy (otk_widget_t * this)
{
    otk_label_t *label = (otk_label_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_LABEL))
        return;

    widget_remove (this->otk, this);

    if (label->text)
        ho_free (label->text);

    ho_free (label);
}

otk_widget_t *
otk_label_new (otk_t * otk, int x, int y, int w,
               int alignment, const char *text)
{
    otk_label_t *label = ho_new (otk_label_t);

    widget_init ((otk_widget_t *) label, otk, OTK_WIDGET_LABEL, x, y, w, 0);
    label->widget.selectable = OTK_SELECTABLE_NONE;
    label->widget.draw = label_draw;
    label->widget.destroy = label_destroy;
    label->widget.fontsize = 30;
    label->widget.alignment = alignment;

    if (text)
        label->text = ho_strdup (text);
    else
        label->text = NULL;

    return (otk_widget_t *) label;
}

/*
 * ***********************************************************************************
 * Window-Widget
 * ***********************************************************************************
 * There is always one current window. This current window is set when
 * creating a new window. It can also be set by calling otk_window_set_current.
 *
 * There are two possibilities of how windows are destroyed:
 *
 * 1. The user keeps a pointer to the window in a variable and marks the
 *    window not to be destroyed by calling otk_window_keep. The user can
 *    himself destroy the window by calling otk_window_destroy.
 *
 * 2. The window is automatically destroyed as soon as a new window is created.
 *
 * When calling otk_free all windows will be destroyed by otk.
 * ***********************************************************************************
 */
static void
window_draw (otk_widget_t * this)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    if (window->fill) {
        odk_draw_rect (this->odk, window->widget.x, window->widget.y,
                       window->widget.x + window->widget.w,
                       window->widget.y + window->widget.h,
                       this->otk->textpalette_label +
                       OSD_TEXT_PALETTE_BACKGROUND, true);
    } else if (!window->fill && window->show_border) {
        odk_draw_rect (this->odk, window->widget.x,
                       window->widget.y,
                       window->widget.x + window->widget.w,
                       window->widget.y + window->widget.h,
                       this->otk->textpalette_button[0] +
                       OSD_TEXT_PALETTE_TRANSPARENT, false);
    }

    if (window->show_border) {
        odk_draw_rect (this->odk, window->widget.x, window->widget.y,
                       window->widget.x + window->widget.w - 1,
                       window->widget.y + window->widget.h - 1,
                       this->otk->textpalette_label +
                       OSD_TEXT_PALETTE_FOREGROUND, false);
    }

    otk_widget_t *widget = l_list_first (window->subs);
    while (widget) {
        if (widget->draw && widget->is_visible)
            widget->draw (widget);
        widget = l_list_next (window->subs, widget);
    }
}

static void
window_destroy (otk_widget_t * this)
{
    otk_t *otk = this->otk;
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    mutex_lock (&otk->draw_mutex);

    otk_window_t *current_window = this->otk->window;
    this->otk->window = window;

    // we free all widgets contained in this window
    l_list_free (window->subs, widget_destroy);
    // we remove this window from the list of windows
    l_list_remove (this->otk->windows, window);

    if (current_window == window)
        this->otk->window = NULL;
    else
        this->otk->window = current_window;

    ho_free (window);

    mutex_unlock (&otk->draw_mutex);
}

otk_widget_t *
otk_window_new (otk_t * otk, int x, int y, int w, int h, bool show_border,
                bool fill)
{
    if (otk->window && !otk->window->keep)
        window_destroy ((otk_widget_t *) otk->window);

    otk->window = ho_new (otk_window_t);

    otk->window->widget.type = OTK_WIDGET_WINDOW;
    otk->window->widget.x = x;
    otk->window->widget.y = y;
    otk->window->widget.w = w;
    otk->window->widget.h = h;
    otk->window->widget.otk = otk;
    otk->window->widget.odk = otk->odk;
    otk->window->widget.draw = window_draw;
    otk->window->widget.destroy = window_destroy;
    otk->window->widget.need_update = false;
    otk->window->keep = false;
    otk->window->fill = fill;
    otk->window->show_border = show_border;
    otk->window->subs = l_list_new ();

    l_list_append (otk->windows, otk->window);

    return (otk_widget_t *) otk->window;
}

void
otk_window_keep (otk_widget_t * this, bool keep)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    window->keep = keep;
}

void
otk_window_destroy (otk_widget_t * this)
{
    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    window_destroy (this);
}

void
otk_window_set_current (otk_t * otk, otk_widget_t * this)
{
    otk_window_t *window = (otk_window_t *) this;

    if (!is_correct_widget (this, OTK_WIDGET_WINDOW))
        return;

    otk->window = window;
}

/*
 * ***********************************************************************************
 * Eventhandlers
 * ***********************************************************************************
 */
static void
listentry_select (otk_listentry_t * listentry)
{
    listentry->doubleclicktime = time (NULL);
    int was_selected = listentry->is_selected;

    if (listentry->list->allow_select == OTK_LIST_NO_SELECTION)
        return;

    else if (listentry->list->allow_select == OTK_LIST_SINGLE_SELECTION) {
        otk_listentry_t *current = l_list_first (listentry->list->entries);
        while (current) {
            current->is_selected = false;
            current = l_list_next (listentry->list->entries, current);
        }
        if (was_selected) {
            listentry->list->num_selected = 0;
        } else {
            listentry->is_selected = true;
            listentry->list->num_selected = 1;
        }
    }

    else if (listentry->list->allow_select == OTK_LIST_MULTIPLE_SELECTION) {
        if (was_selected) {
            listentry->is_selected = false;
            listentry->list->num_selected -= 1;
        } else {
            listentry->is_selected = true;
            listentry->list->num_selected += 1;
        }
    }

    otk_draw (listentry->list->widget.otk);

    if (listentry->select_cb)
        listentry->select_cb (listentry->select_cb_data);
}

static void
listentry_activate (otk_listentry_t * listentry)
{
    listentry->doubleclicktime = 0;

    otk_listentry_t *current = l_list_first (listentry->list->entries);
    while (current) {
        current->is_selected = false;
        current = l_list_next (listentry->list->entries, current);
    }
    listentry->list->num_selected = 0;

    if (listentry->activate_cb)
        listentry->activate_cb (listentry->activate_cb_data);
}

static void
listentry_select_activate_handler (otk_listentry_t * listentry)
{
    time_t cur_time = time (NULL);
    if (cur_time - 1 < listentry->doubleclicktime) {
        listentry_activate (listentry);
    } else {
        listentry_select (listentry);
    }
}

static void
motion_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    int x = ev->data.pos.x;
    int y = ev->data.pos.y;

    otk_widget_t *new_focus = widget_find_xy (otk, x, y);
    otk_widget_t *old_focus = otk->window->focus_ptr;

    if (old_focus == new_focus)
        return;

    if (new_focus && (new_focus->is_visible)
        && (new_focus->is_enabled)
        && (new_focus->selectable)) {
        otk_widget_set_focus (new_focus);
        otk_draw (otk);
    }

    else if (old_focus) {
        otk_widget_unset_focus (otk->window->focus_ptr);
        otk_draw (otk);
    }
}

static void
button_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    if (ev->source.button == OXINE_BUTTON_NULL)
        return;

    int x = ev->data.pos.x;
    int y = ev->data.pos.y;

    otk_widget_t *widget = widget_find_xy (otk, x, y);

    if (!widget
        || !widget->is_visible || !widget->is_enabled || !widget->selectable)
        return;

    switch (ev->source.button) {
    case OXINE_BUTTON1:                                       // left button
        if (is_correct_widget (widget, OTK_WIDGET_BUTTON)) {
            otk_button_t *button = (otk_button_t *) widget;

            if (button->cb)
                button->cb (button->cb_data);
        }

        else if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *listentry = (otk_listentry_t *) widget;
            listentry_select_activate_handler (listentry);
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
            otk_slider_t *slider = (otk_slider_t *) widget;

            if (slider->set_value_cb) {
                double p = 0;
                if (slider->direction == OTK_SLIDER_HORIZONTAL) {
                    int b = (widget->h - slider->marker_h) / 2;
                    int w = (widget->w - 2 * b);
                    int x = (widget->x + b);
                    p = ((ev->data.pos.x - x) * 100) / w;
                } else if (slider->direction == OTK_SLIDER_VERTICAL) {
                    int b = (widget->w - slider->marker_w) / 2;
                    int h = (widget->h - 2 * b);
                    int y = (widget->y + b);
                    p = ((ev->data.pos.y - y) * 100) / h;
                    p = 100 - p;
                }
                int max = slider->max_value - slider->min_value;
                int value = (p * max / 100) + slider->min_value;
                if (value > slider->max_value)
                    value = slider->max_value;
                if (value < slider->min_value)
                    value = slider->min_value;
                slider->set_value_cb (slider->set_value_cb_data, value);
            }
        }

        else if (is_correct_widget (widget, OTK_WIDGET_CHECKBOX)) {
            otk_checkbox_t *checkbox = (otk_checkbox_t *) widget;

            checkbox->is_checked = (checkbox->is_checked + 1) % 2;

            if (checkbox->cb) {
                checkbox->cb (checkbox->cb_data);
            }
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
            otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

            if (scrollbar->cb_click) {
                int position = (ev->data.pos.y - widget->y) * 100 / widget->h;
                scrollbar->cb_click (scrollbar->cb_data, position);
            }
        }

        else if (is_correct_widget (widget, OTK_WIDGET_EDITBOX)) {
            otk_editbox_t *editbox = (otk_editbox_t *) widget;

            odk_osd_set_font (editbox->widget.otk->odk, editbox->widget.font,
                              editbox->widget.fontsize);

            if (editbox->carret_x > ev->data.pos.x) {
                while ((editbox->cur_position > 0)
                       && (editbox->carret_x > ev->data.pos.x)) {
                    editbox->cur_position--;
                    editbox_calculate_carret_position (editbox);
                }
            }

            else {
                while ((editbox->cur_position <
                        editbox->first_visible + editbox->num_visible)
                       && (editbox->carret_x < ev->data.pos.x)) {
                    editbox->cur_position++;
                    editbox_calculate_carret_position (editbox);
                }
            }
        }

        break;
    case OXINE_BUTTON4:                                       // scrollwheel down
        if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) widget;
            otk_list_t *list = entry->list;

            list->first_visible--;
            listentries_adapt (list);
            otk_draw (otk);

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
            otk_slider_t *slider = (otk_slider_t *) widget;
            slider_plus_cb (slider);
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
            otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

            if (scrollbar->cb_click) {
                int position = scrollbar->position - (widget->h / 100);
                scrollbar->cb_click (scrollbar->cb_data, position);
            }
        }
        break;

    case OXINE_BUTTON5:                                       // scrollwheel up
        if (is_correct_widget (widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) widget;
            otk_list_t *list = entry->list;

            list->first_visible++;
            listentries_adapt (list);
            otk_draw (otk);

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SLIDER)) {
            otk_slider_t *slider = (otk_slider_t *) widget;
            slider_minus_cb (slider);
        }

        else if (is_correct_widget (widget, OTK_WIDGET_SCROLLBAR)) {
            otk_scrollbar_t *scrollbar = (otk_scrollbar_t *) widget;

            if (scrollbar->cb_click) {
                int position = scrollbar->position + (widget->h / 100);
                scrollbar->cb_click (scrollbar->cb_data, position);
            }
        }
        break;
    default:
        break;
    }
}

static void
key_handler (otk_t * otk, oxine_event_t * ev)
{
    if (!otk->window)
        return;

    if (ev->source.key == OXINE_KEY_NULL)
        return;

    otk_widget_t *cur_widget = otk->window->focus_ptr;
    otk_widget_t *new_widget = NULL;

    if (is_correct_widget (cur_widget, OTK_WIDGET_EDITBOX)
        && (ev->source.key == OXINE_KEY_INPUT)) {
        editbox_key_handler ((otk_editbox_t *) cur_widget, ev);
        ev->source.key = OXINE_KEY_NULL;
        return;
    }

    if (ev->source.key == OXINE_KEY_INPUT) {
        ev->source.key = OXINE_KEY_NULL;
        return;
    }

    if (is_correct_widget (cur_widget, OTK_WIDGET_EDITBOX)
        && (ev->source.key != OXINE_KEY_UP)
        && (ev->source.key != OXINE_KEY_DOWN)
        && (ev->source.key != OXINE_KEY_SPEED)
        && (ev->source.key != OXINE_KEY_PREV_WIDGET)
        && (ev->source.key != OXINE_KEY_NEXT_WIDGET)
        && (ev->source.key != OXINE_KEY_MENU_MAIN)
        && (ev->source.key != OXINE_KEY_MENU_MUSIC)
        && (ev->source.key != OXINE_KEY_MENU_VIDEO)
        && (ev->source.key != OXINE_KEY_MENU_OSD)
        && (ev->source.key != OXINE_KEY_MENU_CURRENT_TITLE)
        && (ev->source.key != OXINE_KEY_PLAY_VDR)
        && (ev->source.key != OXINE_KEY_PLAY_DVB)
        && (ev->source.key != OXINE_KEY_PLAY_V4L)) {
        ev->source.key = OXINE_KEY_NULL;
        return;
    }

    int key = ev->source.key;

    switch (key) {
    case OXINE_KEY_FIRST:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            list->first_visible = 0;
            listentries_adapt (list);
            new_widget = l_list_first (list->entries);

            if (new_widget && !(new_widget->is_visible)
                && !(new_widget->is_enabled)) {
                new_widget = l_list_next (list->entries, new_widget);
            }

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_LAST:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            list->first_visible = list->num_entries;
            listentries_adapt (list);
            new_widget = l_list_last (list->entries);

            if (new_widget && !(new_widget->is_visible)
                && !(new_widget->is_enabled)) {
                new_widget = l_list_prev (list->entries, new_widget);
            }

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_PAGE_UP:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            list->first_visible -= list->num_visible;
            listentries_adapt (list);
            new_widget = cur_widget;
            while (!((otk_listentry_t *) new_widget)->is_first) {
                new_widget = l_list_prev (list->entries, new_widget);
            }

            if (new_widget && !(new_widget->is_visible)
                && !(new_widget->is_enabled)) {
                new_widget = l_list_next (list->entries, new_widget);
            }

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_PAGE_DOWN:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            list->first_visible += list->num_visible;
            listentries_adapt (list);
            new_widget = cur_widget;
            while (!((otk_listentry_t *) new_widget)->is_last) {
                new_widget = l_list_next (list->entries, new_widget);
            }

            if (new_widget && !(new_widget->is_visible)
                && !(new_widget->is_enabled)) {
                new_widget = l_list_prev (list->entries, new_widget);
            }

            if (list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_UP:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            otk_listentry_t *new_entry = entry;
            do {
                if (new_entry->is_first) {
                    list->first_visible--;
                    listentries_adapt (list);
                }

                new_entry = l_list_prev (list->entries, new_entry);
            } while (new_entry && !(new_entry->widget.is_visible)
                     && !(new_entry->widget.is_enabled));
            new_widget = (otk_widget_t *) new_entry;

            if (new_entry && (new_entry != entry) && list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }

        else {
            new_widget = widget_find_neighbour (otk, DIRECTION_UP);
        }
        break;
    case OXINE_KEY_DOWN:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;

            otk_listentry_t *new_entry = entry;
            do {
                if (new_entry->is_last) {
                    list->first_visible++;
                    listentries_adapt (list);
                }

                new_entry = l_list_next (list->entries, new_entry);
            } while (new_entry && !(new_entry->widget.is_visible)
                     && !(new_entry->widget.is_enabled));
            new_widget = (otk_widget_t *) new_entry;

            if (new_entry && (new_entry != entry) && list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }

        else {
            new_widget = widget_find_neighbour (otk, DIRECTION_DOWN);
        }
        break;
    case OXINE_KEY_LEFT:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;
            if (list->num_visible == 1)
                new_widget = widget_get_prev_focus (otk);
            else
                new_widget = widget_find_neighbour (otk, DIRECTION_LEFT);
        }

        else {
            new_widget = widget_find_neighbour (otk, DIRECTION_LEFT);
        }
        break;
    case OXINE_KEY_RIGHT:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            otk_list_t *list = entry->list;
            if (list->num_visible == 1)
                new_widget = widget_get_next_focus (otk);
            else
                new_widget = widget_find_neighbour (otk, DIRECTION_LEFT);
        }

        else {
            new_widget = widget_find_neighbour (otk, DIRECTION_RIGHT);
        }
        break;
    case OXINE_KEY_SELECT:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            listentry_select (entry);
        }
        break;
    case OXINE_KEY_ACTIVATE:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;
            listentry_activate (entry);
            ev->source.key = OXINE_KEY_NULL;
        }

        else if (is_correct_widget (cur_widget, OTK_WIDGET_BUTTON)) {
            otk_button_t *button = (otk_button_t *) cur_widget;
            button->cb (button->cb_data);
            ev->source.key = OXINE_KEY_NULL;
        }

        else if (is_correct_widget (cur_widget, OTK_WIDGET_CHECKBOX)) {
            otk_checkbox_t *checkbox = (otk_checkbox_t *) cur_widget;

            checkbox->is_checked = (checkbox->is_checked + 1) % 2;
            checkbox->widget.need_repaint = true;

            if (checkbox->cb) {
                checkbox->cb (checkbox->cb_data);
            }

            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    case OXINE_KEY_REMOVE:
        if (is_correct_widget (cur_widget, OTK_WIDGET_LISTENTRY)) {
            otk_listentry_t *entry = (otk_listentry_t *) cur_widget;

            if (entry->remove_cb) {
                entry->remove_cb (entry->remove_cb_data);
            }

            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    case OXINE_KEY_PREV_WIDGET:
        {
            new_widget = widget_get_prev_focus (otk);
            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    case OXINE_KEY_NEXT_WIDGET:
        {
            new_widget = widget_get_next_focus (otk);
            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    default:
        break;
    }

    if (new_widget) {
        otk_widget_set_focus (new_widget);
        otk_draw (otk);
    }
}

static void
otk_event_handler (void *this, oxine_event_t * ev)
{
    otk_t *otk = (otk_t *) this;
    mutex_lock (&otk->draw_mutex);

    switch (ev->type) {
    case OXINE_EVENT_KEY:
        key_handler (otk, ev);
        break;
    case OXINE_EVENT_MOTION:
        motion_handler (otk, ev);
        break;
    case OXINE_EVENT_BUTTON:
        button_handler (otk, ev);
        break;
    case OXINE_EVENT_OSD_FORMAT_CHANGED:
        otk_draw (otk);
        break;
    default:
        break;
    }

    mutex_unlock (&otk->draw_mutex);
}


/*
 * ***********************************************************************************
 * Global Functions
 * ***********************************************************************************
 */
void
otk_draw (otk_t * otk)
{
    if (!otk->window)
        return;

    mutex_lock (&otk->draw_mutex);

    /* Clear the OSD drawing area */
    odk_osd_clear (otk->odk);

    /* Draw our widgets */
    otk_widget_t *widget = (otk_widget_t *) otk->window;
    widget->draw (widget);

    /* Show the OSD drawing area */
    odk_osd_show (otk->odk);

    mutex_unlock (&otk->draw_mutex);
}

void
otk_clear (otk_t * otk)
{
    mutex_lock (&otk->draw_mutex);

    if (otk->window && !otk->window->keep)
        window_destroy ((otk_widget_t *) otk->window);
    otk->window = NULL;

    odk_osd_clear (otk->odk);
    odk_osd_hide (otk->odk);

    mutex_unlock (&otk->draw_mutex);
}

void
otk_widget_unset_focus (otk_widget_t * widget)
{
    if (!widget)
        return;
    if (widget->otk->window->focus_ptr != widget)
        return;

    widget->is_focused = false;
    widget->otk->window->focus_ptr = NULL;

    if (widget->focus_leave_cb)
        widget->focus_leave_cb (widget->focus_leave_cb_data);
}

void
otk_widget_set_focus (otk_widget_t * widget)
{
    if (!widget)
        return;
    if (!widget->is_visible)
        return;
    if (!widget->is_enabled)
        return;
    if (widget->selectable == OTK_SELECTABLE_NONE)
        return;

    if (widget->otk->window->focus_ptr)
        otk_widget_unset_focus (widget->otk->window->focus_ptr);

    widget->is_focused = true;
    widget->otk->window->focus_ptr = widget;

    if (widget->focus_enter_cb)
        widget->focus_enter_cb (widget->focus_enter_cb_data);
}

void
otk_widget_set_focus_callbacks (otk_widget_t * widget,
                                otk_cb_t enter_cb,
                                void *enter_cb_data,
                                otk_cb_t leave_cb, void *leave_cb_data)
{
    widget->focus_enter_cb_data = enter_cb_data;
    widget->focus_enter_cb = enter_cb;
    widget->focus_leave_cb_data = leave_cb_data;
    widget->focus_leave_cb = leave_cb;
}

void
otk_widget_set_update (otk_widget_t * this, bool update)
{
    this->need_update = update;
}

static void
otk_update_job (void *otk_p)
{
    otk_t *otk = (otk_t *) otk_p;
    bool need_repaint = false;

    mutex_lock (&otk->draw_mutex);

    if (otk->window) {
        otk_widget_t *widget = l_list_first (otk->window->subs);
        while (widget) {
            if (widget->need_update) {
                switch (widget->type) {
                case OTK_WIDGET_LABEL:
                    {
                        otk_label_t *label = (otk_label_t *) widget;
                        if (label->update_cb) {
                            label->update_cb (label->update_cb_data, widget);
                        }
                        break;
                    }
                case OTK_WIDGET_SLIDER:
                    {
                        otk_slider_t *slider = (otk_slider_t *) widget;
                        if (slider->get_value_cb) {
                            int value = slider->get_value_cb (slider->
                                                              get_value_cb_data);
                            need_repaint |= (slider->last_value != value);
                        }
                        break;
                    }
                }
            }
            need_repaint |= widget->need_repaint;
            widget = l_list_next (otk->window->subs, widget);
        }
    }

    if (need_repaint)
        otk_draw (otk);

    mutex_unlock (&otk->draw_mutex);

    otk->update_job = schedule_job (500, otk_update_job, otk);
}

static int
read_color (otk_t * otk, xml_node_t * node)
{
    uint32_t c[] = {
        0xffffff,
        0xff0000,
        0x00ff00
    };
    uint8_t t[] = {
        0xf, 0xf, 0xf
    };

    xml_node_t *child = node->child;
    while (child) {
        if (strcasecmp (child->name, "color") == 0) {
            char *name = xml_parser_get_property (child, "name");
            if (!name) {
                error (_("Missing property (required) '%s'!"), "name");
                break;
            }

            unsigned int cx;
            char *value = xml_parser_get_property (child, "value");
            if (!value || (sscanf (value, "%06x", &cx) != 1)) {
                error (_("Missing property (required) '%s'!"), "value");
                cx = 0xff0000;
            }

            unsigned int tx;
            char *trans = xml_parser_get_property (child, "transparency");
            if (!trans || (sscanf (trans, "%02x", &tx) != 1)) {
                tx = 0xf;
            }

            if (strcasecmp (name, "foreground") == 0) {
                c[0] = cx;
                t[0] = tx;
            }

            else if (strcasecmp (name, "background") == 0) {
                c[1] = cx;
                t[1] = tx;
            }

            else if (strcasecmp (name, "border") == 0) {
                c[2] = cx;
                t[2] = tx;
            }

            else {
                error (_("Unknown name '%s' for node!"), name);
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), child->name);
        }

        child = child->next;
    }

    return odk_osd_alloc_text_palette (otk->odk,
                                       c[0], t[0], c[1], t[1], c[2], t[2]);
}

static int
otk_load_palette (otk_t * otk, const char *mrl)
{
    debug (_("Loading palette file '%s'..."), mrl);

    int size;
    char *file = read_entire_file (mrl, &size);
    xml_node_t *node;

    if (!file) {
        return 0;
    }

    xml_parser_init (file, strlen (file), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), mrl);
        return 0;
    }

    if (strcasecmp (node->name, "oxine_palette")) {
        error (_("Root node of '%s' must be '%s'!"), mrl, "oxine_palette");
        return 0;
    }

    xml_node_t *child = node->child;
    while (child) {
        if (strcasecmp (child->name, "sub_palette") == 0) {
            char *name = xml_parser_get_property (child, "name");

            if (strcasecmp (name, "button") == 0) {
                otk->textpalette_button[0] = read_color (otk, child);
            }

            else if (strcasecmp (name, "focused_button") == 0) {
                otk->textpalette_button[1] = read_color (otk, child);
            }

            else if (strcasecmp (name, "disabled_button") == 0) {
                otk->textpalette_button[2] = read_color (otk, child);
            }

            else if (strcasecmp (name, "label") == 0) {
                otk->textpalette_label = read_color (otk, child);
            }

            else {
                error (_("Unknown name '%s' for node!"), name);
            }
        }

        else {
            error (_("Unknown name '%s' for node!"), child->name);
        }
        child = child->next;
    }

    xml_parser_free_tree (node);
    ho_free (file);

    return 1;
}

otk_t *
otk_init (odk_t * odk)
{
    otk_t *otk = ho_new (otk_t);

    otk->odk = odk;
    otk->window = NULL;
    otk->windows = l_list_new ();

    /* We try to load a palette from the users home directory and from the
     * current skin directory. If both fail, we fall back to the standard
     * colors. */
    char mrl[1024];
    snprintf (mrl, 1024, "%s/palette.xml", get_dir_oxine ());
    if (!otk_load_palette (otk, mrl)
        && !otk_load_palette (otk, OXINE_SKINDIR "/default/palette.xml")) {
        error (_("Could not load palette, reverting to standard colors."));

        uint32_t c[3];

        /* define the palette for a non-focused button */
        c[0] = 0xffffff;                                       // foreground
        c[1] = 0x2a569d;                                       // background
        c[2] = 0xffffff;                                       // border
        otk->textpalette_button[0] = odk_osd_alloc_text_palette (odk,
                                                                 c[0], 0xf,
                                                                 c[1], 0xf,
                                                                 c[2], 0xf);

        /* define the palette for a focused button */
        c[0] = 0xffffff;                                       // foreground
        c[1] = 0x5175b0;                                       // background
        c[2] = 0xffffff;                                       // border
        otk->textpalette_button[1] = odk_osd_alloc_text_palette (odk,
                                                                 c[0], 0xf,
                                                                 c[1], 0xf,
                                                                 c[2], 0xf);

        /* define the palette for a disabled button */
        c[0] = 0x5175b0;                                       // foreground
        c[1] = 0x2a569d;                                       // background
        c[2] = 0x5175b0;                                       // border
        otk->textpalette_button[2] = odk_osd_alloc_text_palette (odk,
                                                                 c[0], 0xf,
                                                                 c[1], 0xf,
                                                                 c[2], 0xf);

        /* define the palette for a label */
        c[0] = 0xffffff;                                       // foreground
        c[1] = 0x5175b0;                                       // background
        c[2] = 0xffffff;                                       // border
        otk->textpalette_label = odk_osd_alloc_text_palette (odk,
                                                             c[0], 0xf,
                                                             c[1], 0xf,
                                                             c[2], 0xf);
    }

    /* create our drawing mutex */
    pthread_mutexattr_init (&otk->draw_mutex_attr);
    pthread_mutexattr_settype (&otk->draw_mutex_attr,
                               PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&otk->draw_mutex, &otk->draw_mutex_attr);

    /* register the otk event handler */
    odk_add_event_handler (odk, otk_event_handler, otk,
                           EVENT_HANDLER_PRIORITY_HIGH);

    /* start the update job */
    otk->update_job = schedule_job (500, otk_update_job, otk);

    return otk;
}

void
otk_free (otk_t * otk)
{
    /* stop the update job */
    lock_job_mutex ();
    cancel_job (otk->update_job);
    unlock_job_mutex ();

    /* unregister the otk event handler */
    odk_del_event_handler (otk->odk, otk_event_handler);

    otk_clear (otk);
    l_list_free (otk->windows, widget_destroy);
    otk->windows = NULL;
    otk->window = NULL;

    pthread_mutex_destroy (&otk->draw_mutex);
    pthread_mutexattr_destroy (&otk->draw_mutex_attr);

    ho_free (otk);
}
