
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playback_menu.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "codeset.h"
#include "dvd_menu.h"
#include "environment.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "lang.h"
#include "logger.h"
#include "main_menu.h"
#include "meta_info.h"
#include "oxine.h"
#include "playback_menu.h"
#include "playlist_menu.h"
#include "scheduler.h"

extern oxine_t *oxine;

#define B_FOCUS_MODE    1
#define B_FOCUS_PLAY    2
#define B_FOCUS_VOL     3
#define B_FOCUS_PARAM   4
static int playback_menu_focus_buttons = B_FOCUS_PLAY;
static int playback_menu_show_volume = 0;

static void show_playback_menu_gui (void *oxine_p);

/*
 * **************************************************************************
 * Callbacks of the stream parameters menu
 * **************************************************************************
 */
static void
update_param_label (void *oxine_p, otk_widget_t * label,
                    odk_stream_param_t param)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    char *param_string = odk_get_stream_param_as_string (oxine->odk, param);
    otk_label_set_text (label, param_string);
    ho_free (param_string);
}


static int
volume_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME, value);
    return value;
}


static int
volume_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    return odk_get_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME);
}


static int
zoom_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X, value);
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y, value);
    return value;
}


static int
zoom_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y);
}


static int
audio_channel_change (void *oxine_p, int pos)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_AUDIO_CHANNEL, pos - 1);

    return pos;
}


static int
spu_change (void *oxine_p, int pos)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_SPU_CHANNEL, pos - 1);

    return pos;
}


static void
av_offset_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_OFFSET, 9, -900000, 900000);
}


static void
av_offset_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_AUDIO_OFFSET, -9, -900000, 900000);
}


static void
av_offset_reset (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_AUDIO_OFFSET, 0);
}


static void
av_offset_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_AUDIO_OFFSET);
}


static void
spu_offset_plus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_OFFSET, 9, -900000, 900000);
}


static void
spu_offset_minus (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_change_stream_param (oxine->odk,
                             ODK_PARAM_SPU_OFFSET, -9, -900000, 900000);
}


static void
spu_offset_reset (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET, 0);
}


static void
spu_offset_upcall (void *oxine_p, otk_widget_t * label)
{
    update_param_label (oxine_p, label, ODK_PARAM_SPU_OFFSET);
}


static int
deinterlace_change (void *oxine_p, int pos)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE, pos);

    return pos;
}


static int
aspect_ratio_change (void *oxine_p, int pos)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    int value = ODK_VO_ASPECT_AUTO;
    switch (pos) {
    case 1:
        value = ODK_VO_ASPECT_SQUARE;
        break;
    case 2:
        value = ODK_VO_ASPECT_4_3;
        break;
    case 3:
        value = ODK_VO_ASPECT_ANAMORPHIC;
        break;
    case 4:
        value = ODK_VO_ASPECT_DVB;
        break;
    default:
        break;
    }
    odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ASPECT_RATIO, value);

    return pos;
}


static int
saturation_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION, value);
}


static int
saturation_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION);
}


static int
hue_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_HUE, value);
}


static int
hue_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_HUE);
}


static int
contrast_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST, value);
}


static int
contrast_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST);
}


static int
brightness_set (void *oxine_p, int value)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_set_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS, value);
}


static int
brightness_get (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS);
}


/*
 * **************************************************************************
 * GUI of the stream parameter menu
 * **************************************************************************
 */
#define LABEL_X     20
#define LABEL_W     300
#define EDIT_X      (LABEL_X + LABEL_W + 20)
#define EDIT_W      (780 - EDIT_X)

#define EDIT_H      40

static void
show_stream_param (oxine_t * oxine, odk_stream_param_t param,
                   otk_label_uc_t upcall,
                   otk_cb_t minus_cb, otk_cb_t plus_cb,
                   otk_cb_t reset_cb, int y)
{
    otk_widget_t *b;
    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    char *param_string = odk_get_stream_param_as_string (oxine->odk, param);

    b = otk_label_new (oxine->otk, LABEL_X, y + EDIT_H / 2, LABEL_W,
                       alignleft, param_name);
    b = otk_label_new (oxine->otk, EDIT_X, y + EDIT_H / 2, EDIT_W,
                       alignleft, param_string);
    otk_label_set_upcall (b, upcall, oxine);

    ho_free (param_name);
    ho_free (param_string);

    b = otk_bitmap_button_new (oxine->otk, 686, y, 30, EDIT_H,
                               odk_get_bitmap (BITMAP_MINUS),
                               minus_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    b = otk_bitmap_button_new (oxine->otk, 718, y, 30, EDIT_H,
                               odk_get_bitmap (BITMAP_PLUS), plus_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    b = otk_bitmap_button_new (oxine->otk, 750, y, 30, EDIT_H,
                               odk_get_bitmap (BITMAP_HOME), reset_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
}


static void
show_stream_param_slider (oxine_t * oxine, odk_stream_param_t param,
                          otk_int_get_cb_t get_cb,
                          otk_int_set_cb_t set_cb,
                          bool with_reset_button, int default_value,
                          int min_value, int max_value, int min_step, int y)
{
    otk_widget_t *w;
    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    w = otk_label_new (oxine->otk, LABEL_X, y + EDIT_H / 2, LABEL_W,
                       alignleft, param_name);
    ho_free (param_name);

    w = otk_slider_new (oxine->otk, EDIT_X, y, EDIT_W, EDIT_H, 8, 36, true,
                        OTK_SLIDER_HORIZONTAL, "%d%%", 1, with_reset_button,
                        default_value, min_value, max_value, min_step,
                        get_cb, oxine, set_cb, oxine);
    otk_widget_set_update (w, true);
}


static void
show_stream_param_selector (oxine_t * oxine, odk_stream_param_t param,
                            otk_int_set_cb_t cb, int min_value,
                            int max_value, int y)
{
    otk_widget_t *w;
    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;

    char *param_name = odk_get_stream_param_name (oxine->odk, param);
    w = otk_label_new (oxine->otk, LABEL_X, y + EDIT_H / 2, LABEL_W,
                       alignleft, param_name);
    ho_free (param_name);

    w = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W, cb, oxine);
    int i = min_value;
    for (; i <= max_value; i++) {
        char *t = odk_get_stream_param_value_as_string (oxine->odk, param, i);
        otk_listentry_new (w, t, NULL, NULL, NULL, NULL, NULL, NULL);
        ho_free (t);
    }
    otk_list_set_pos (w,
                      odk_get_stream_param (oxine->odk, param) - min_value);
    y += 50;
}


static void
stream_parameter_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (event->type != OXINE_EVENT_KEY)
        return;
    if (event->source.key != OXINE_KEY_BACK)
        return;

    if ((oxine->current_menu == show_playback_settings_p1_cb)
        || (oxine->current_menu == show_playback_settings_p2_cb)) {
        show_playback_menu_gui (oxine);
        event->source.key = OXINE_KEY_NULL;
    }
}


void
show_playback_settings_p2_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (oxine->backto_menu == show_playback_settings_p2_cb)
        return;

    playback_menu_focus_buttons = B_FOCUS_PARAM;

    hide_user_interface (oxine);
    create_new_window (false, true);

    oxine->backto_menu = show_playback_settings_p2_cb;
    oxine->current_menu = show_playback_settings_p2_cb;
    odk_add_event_handler (oxine->odk, stream_parameter_menu_event_handler,
                           oxine, EVENT_HANDLER_PRIORITY_NORMAL);

    int y = 100;

    int ac = odk_get_stream_info (oxine->odk,
                                  ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
    int sc = odk_get_stream_info (oxine->odk,
                                  ODK_STREAM_INFO_MAX_SPU_CHANNEL);

    if (odk_get_stream_info (oxine->odk, ODK_STREAM_INFO_HAS_VIDEO)) {
        // Audio Channel
        if (ac > 1) {
            show_stream_param_selector (oxine, ODK_PARAM_AUDIO_CHANNEL,
                                        audio_channel_change,
                                        AUDIO_CHANNEL_AUTO, ac - 1, y);
            y += 50;
        }
        // Audio-Video Offset
        if (ac > 0) {
            show_stream_param (oxine, ODK_PARAM_AUDIO_OFFSET,
                               av_offset_upcall, av_offset_minus,
                               av_offset_plus, av_offset_reset, y);
            y += 50;
        }
        // Subtitle Channel
        if (sc > 1) {
            show_stream_param_selector (oxine, ODK_PARAM_SPU_CHANNEL,
                                        spu_change,
                                        SPU_CHANNEL_AUTO, sc - 1, y);
            y += 50;
        }
        // Subtitle Offset
        if (sc > 0) {
            show_stream_param (oxine, ODK_PARAM_SPU_OFFSET,
                               spu_offset_upcall, spu_offset_minus,
                               spu_offset_plus, spu_offset_reset, y);
            y += 50;
        }
        // Aspect Ratio
        {
            show_stream_param_selector (oxine, ODK_PARAM_VO_ASPECT_RATIO,
                                        aspect_ratio_change,
                                        ODK_VO_ASPECT_AUTO,
                                        ODK_VO_ASPECT_DVB, y);
            y += 50;
        }
        // Deinterlace
        {
            show_stream_param_selector (oxine, ODK_PARAM_VO_DEINTERLACE,
                                        deinterlace_change, 0, 1, y);
            y += 50;
        }
    }

    otk_widget_t *b;
    b = otk_text_button_new (oxine->otk, 20, 545, 180, 35, _("Next Page"),
                             show_playback_settings_p1_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_focus (b);
    b = otk_text_button_new (oxine->otk, 210, 545, 180, 35, _("Back"),
                             show_playback_menu_gui, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

    show_user_interface (oxine);
}


void
show_playback_settings_p1_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (oxine->backto_menu == show_playback_settings_p1_cb)
        return;

    playback_menu_focus_buttons = B_FOCUS_PARAM;

    hide_user_interface (oxine);
    create_new_window (false, true);

    oxine->backto_menu = show_playback_settings_p1_cb;
    oxine->current_menu = show_playback_settings_p1_cb;
    odk_add_event_handler (oxine->odk, stream_parameter_menu_event_handler,
                           oxine, EVENT_HANDLER_PRIORITY_NORMAL);

    int y = 100;

    // Volume
    otk_widget_t *b;
    show_stream_param_slider (oxine, ODK_PARAM_AUDIO_VOLUME,
                              volume_get, volume_set,
                              true, 100, 0, 100, 5, y);
    y += 50;
    show_stream_param_slider (oxine, ODK_PARAM_VO_BRIGHTNESS,
                              brightness_get, brightness_set,
                              true, 32896, 0, 65535, 656, y);
    y += 50;
    show_stream_param_slider (oxine, ODK_PARAM_VO_CONTRAST,
                              contrast_get, contrast_set,
                              true, 32896, 0, 65535, 656, y);
    y += 50;
    show_stream_param_slider (oxine, ODK_PARAM_VO_SATURATION,
                              saturation_get, saturation_set,
                              true, 32896, 0, 65535, 656, y);
    y += 50;
    show_stream_param_slider (oxine, ODK_PARAM_VO_HUE,
                              hue_get, hue_set,
                              true, 32896, 0, 65535, 656, y);
    y += 50;
    show_stream_param_slider (oxine, ODK_PARAM_VO_ZOOM_X,
                              zoom_get, zoom_set, true, 100, 0, 100, 1, y);
    y += 50;

    if (odk_get_stream_info (oxine->odk, ODK_STREAM_INFO_HAS_VIDEO)) {
        b = otk_text_button_new (oxine->otk, 20, 545, 180, 35, _("Next Page"),
                                 show_playback_settings_p2_cb, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);
        b = otk_text_button_new (oxine->otk, 210, 545, 180, 35, _("Back"),
                                 show_playback_menu_gui, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    } else {
        b = otk_text_button_new (oxine->otk, 20, 545, 180, 35, _("Back"),
                                 show_playback_menu_gui, oxine);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        otk_widget_set_focus (b);
    }

    show_user_interface (oxine);
}

/*
 * **************************************************************************
 * Callbacks of the playing menu
 * **************************************************************************
 */

static void
hide_volume_slider_cb (void *oxine_p)
{
    playback_menu_focus_buttons = B_FOCUS_VOL;
    playback_menu_show_volume = 0;
    show_playback_menu_gui (oxine_p);
}


static void
show_volume_slider_cb (void *oxine_p)
{
    playback_menu_focus_buttons = B_FOCUS_VOL;
    playback_menu_show_volume = 1;
    show_playback_menu_gui (oxine_p);
}


static void
change_playlist_mode_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PLAYMODE;

    playback_menu_focus_buttons = B_FOCUS_MODE;

    otk_send_event (oxine->otk, &event);
}


static void
track_select_cb (void *entry_cb_data)
{
    playlist_play_item (oxine, show_playback_menu_cb,
                        oxine->current_playlist,
                        (playitem_t *) entry_cb_data);

    /* Save current channel to config file. */
    if (odk_current_is_dvb (oxine->odk))
        xine_config_save (oxine->xine, get_file_config ());
}


static void
playlist_prev_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PREV;

    otk_send_event (oxine->otk, &event);
}


static void
playlist_next_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_NEXT;

    otk_send_event (oxine->otk, &event);
}


static void
media_pplay_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_PPLAY;

    otk_send_event (oxine->otk, &event);
}


static void
media_stop_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine_event_t event;
    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_STOP;

    otk_send_event (oxine->otk, &event);
}


static int
set_stream_progress (void *oxine_p, int position)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    odk_set_stream_param (oxine->odk, ODK_PARAM_POSITION, position);

    return position;
}


static int
get_stream_progress (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    return odk_get_stream_param (oxine->odk, ODK_PARAM_POSITION);
}


/* 
 * **************************************************************************
 * GUI-Layout of the playing menu
 * **************************************************************************
 */
static void
set_time_gone_string (void *oxine_p, otk_widget_t * widget)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    char time_gone_str[64];

    int pos_time;
    int length;

    if (odk_get_pos_length (oxine->odk, NULL, &pos_time, &length)) {
        pos_time /= 1000;
        sprintf (time_gone_str, "%d:%02d:%02d",
                 pos_time / 3600, (pos_time % 3600) / 60,
                 ((pos_time % 3600) % 60));
        otk_label_set_text (widget, time_gone_str);
    }
}


static void
set_time_total_string (void *oxine_p, otk_widget_t * widget)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    char time_total_str[64];

    int pos_time;
    int length;

    if (odk_get_pos_length (oxine->odk, NULL, &pos_time, &length)) {
        if (length == 0)
            length = pos_time;
        length /= 1000;
        sprintf (time_total_str, "%d:%02d:%02d",
                 length / 3600, (length % 3600) / 60, ((length % 3600) % 60));
        otk_label_set_text (widget, time_total_str);
    }
}


static void
show_playback_menu_file_info (oxine_t * oxine)
{
    int a0 = OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER;
    int al = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;
    int ar = OTK_ALIGN_RIGHT | OTK_ALIGN_TOP;

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    if (!title) {
        title = create_title (odk_current_get_mrl (oxine->odk));
    }
    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, 220, 117, 560, a0, title);
    otk_widget_set_font (l, "sans", 40);

    int y = 150;
    int x1 = 380;
    int x2 = 400;
    int w1 = 200;
    int w2 = 380;

    if (album && (strlen (album) > 0)) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Album:"));
        otk_label_new (oxine->otk, x2, y, w2, al, album);
        y += 40;
    }

    if (artist && (strlen (artist) > 0)) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Artist:"));
        otk_label_new (oxine->otk, x2, y, w2, al, artist);
        y += 40;
    }

    if (year && (atoi (year) > 0) && (atoi (year) <= current_year)) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Year:"));
        otk_label_new (oxine->otk, x2, y, w2, al, year);
        y += 40;
    }

    if (track && (strlen (track) > 0)) {
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Track:"));
        otk_label_new (oxine->otk, x2, y, w2, al, track);
        y += 40;
    }

    if (genre && (strlen (genre) > 0)) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x1, y, w1, ar, _("Genre:"));
        otk_label_new (oxine->otk, x2, y, w2, al, genre);
        y += 40;
    }

    {
        y += 10;
        l = otk_label_new (oxine->otk, x1, y, w1, ar, _("Length:"));
        l = otk_label_new (oxine->otk, x2, y, w2, al, "0:00:00");
        set_time_total_string (oxine, l);
        otk_label_set_upcall (l, set_time_total_string, oxine);
        otk_widget_set_update (l, true);
        y += 40;
    }

    {
        l = otk_label_new (oxine->otk, x1, y, w1, ar, _("Elapsed:"));
        l = otk_label_new (oxine->otk, x2, y, w2, al, "0:00:00");
        set_time_gone_string (oxine, l);
        otk_label_set_upcall (l, set_time_gone_string, oxine);
        otk_widget_set_update (l, true);
        y += 40;
    }

    ho_free (title);
    ho_free (artist);
    ho_free (album);
    ho_free (year);
    ho_free (genre);
    ho_free (track);
}


/*
 * This menu shows a list of all the files in the playlist.
 */
static void
show_playback_menu_file_list (oxine_t * oxine)
{
    otk_widget_t *l;

    if (odk_current_is_dvb (oxine->odk)) {
        l = otk_list_new (oxine->otk, 220, 100, 560, 480,
                          30, 30, true, true, OTK_LIST_NO_SELECTION, oxine);
    }

    else {
        l = otk_list_new (oxine->otk, 220, 100, 560, 410,
                          30, 30, true, true, OTK_LIST_NO_SELECTION, oxine);
    }

    playitem_t *cur = playlist_first (oxine->current_playlist);
    while (cur) {
        otk_listentry_new (l, cur->title, track_select_cb, cur, NULL, NULL,
                           NULL, NULL);
        cur = playlist_next (oxine->current_playlist, cur);
    }

    int pos = playlist_get_current_pos (oxine->current_playlist);
    otk_list_set_selected (l, pos, 1);
    otk_list_set_pos (l, pos);

    if (odk_current_is_dvb (oxine->odk)) {
        otk_list_set_focus (l, pos);
    }
}


static void
show_controls (oxine_t * oxine, int with_volume)
{
    if (odk_current_is_dvb (oxine->odk))
        return;

    /* If we're playing a playlist and the playlist has more than one entry, a
     * lot of buttons have to be shown. */
    int playlist_with_more_than_one_entry =
        (playlist_length (oxine->current_playlist) > 1);

    int x = 20;
    int y = 530;
    otk_widget_t *b;

    /*
     * ********************************************************
     * Show the play/ pause, stop, next/ prev buttons.
     * ********************************************************
     */
    char *media_pplay_label = NULL;
    if (odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED) == ODK_SPEED_PAUSE) {
        media_pplay_label = ">";
    } else {
        media_pplay_label = "<";
    }

    {
        b = otk_text_button_new (oxine->otk, x, y, 50, 50,
                                 media_pplay_label, media_pplay_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        if (playback_menu_focus_buttons == B_FOCUS_PLAY)
            otk_widget_set_focus (b);
        x += 60;
    }

    y = 535;
    {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "}", media_stop_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    if (playlist_with_more_than_one_entry) {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "[", playlist_prev_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;

        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "]", playlist_next_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    {
        b = otk_text_button_new (oxine->otk, x, y, 40, 40,
                                 "{", eject_cb, oxine);
        otk_widget_set_font (b, "cetus", 20);
        otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
        x += 50;
    }

    /*
     * ********************************************************
     * Show the progress bar.
     * ********************************************************
     */
    int w = 780 - x;
    if (with_volume)
        w = 730 - x;

    {
        b = otk_slider_new (oxine->otk, x, y, w, 40,
                            0, 20, false,
                            OTK_SLIDER_HORIZONTAL, NULL,
                            false, false, 0, 0, 100, 1,
                            get_stream_progress, oxine,
                            set_stream_progress, oxine);
        otk_widget_set_update (b, true);
    }

    /*
     * ********************************************************
     * Show the volume controls. 
     * ********************************************************
     */
    if (!with_volume)
        return;

    otk_cb_t volume_button_cb = NULL;
    if (playback_menu_show_volume) {
        volume_button_cb = hide_volume_slider_cb;
    } else {
        volume_button_cb = show_volume_slider_cb;
    }

    uint8_t *volume_button_bitmap = NULL;
    if (odk_get_stream_param (oxine->odk, ODK_PARAM_AUDIO_MUTE)) {
        volume_button_bitmap = odk_get_bitmap (BITMAP_MUTE);
    } else {
        volume_button_bitmap = odk_get_bitmap (BITMAP_VOLUME);
    }

    {
        b = otk_bitmap_button_new (oxine->otk, 740, 535, 40, 40,
                                   volume_button_bitmap,
                                   volume_button_cb, oxine);
        if (playback_menu_focus_buttons == B_FOCUS_VOL)
            otk_widget_set_focus (b);
    }

    /* Show the volume slider. */
    if (playback_menu_show_volume) {
        b = otk_slider_new (oxine->otk, 740, 255, 40, 270,
                            36, 8, true,
                            OTK_SLIDER_VERTICAL, NULL,
                            true, false, 0, 0, 100, 5,
                            volume_get, oxine, volume_set, oxine);
        otk_widget_set_update (b, true);
    }
}

static void
show_playback_menu_gui (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    // FIXME: this will get removed eventually
    assert (odk_current_get_mrl (oxine->odk));

    show_menu_background (NULL);

    hide_user_interface (oxine);

    oxine->current_menu = show_playback_menu_cb;
    if (odk_current_has_video (oxine->odk))
        oxine->backto_menu = hide_user_interface;
    else
        oxine->backto_menu = show_playback_menu_gui;

    create_new_window (false, true);

    show_controls (oxine, 1);

    int x = 20;
    int y = 100;

    otk_widget_t *b = NULL;
    if (odk_current_is_dvd (oxine->odk)) {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_2, dvd_menu2_cb, oxine);
        y += 40;

        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_5, dvd_menu5_cb, oxine);
        y += 40;

        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 DVD_MENU_TITLE_7, dvd_menu7_cb, oxine);
        y += 40;
    }

    if (!odk_current_is_dvd (oxine->odk)
        && !odk_current_is_vcd (oxine->odk)
        && !odk_current_is_v4l (oxine->odk)
        && !odk_current_is_dvb (oxine->odk)) {
        switch (playlist_get_playmode (oxine->current_playlist)) {
        case PLAYLIST_MODE_REPEAT:
            b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                     _("Loop Mode"),
                                     change_playlist_mode_cb, oxine);
            break;
        case PLAYLIST_MODE_RANDOM:
            b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                     _("Shuffle Mode"),
                                     change_playlist_mode_cb, oxine);
            break;
        default:
            b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                     _("Normal Mode"),
                                     change_playlist_mode_cb, oxine);
            break;
        }
        if (playback_menu_focus_buttons == B_FOCUS_MODE)
            otk_widget_set_focus (b);
        y += 40;
    }

    {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                 _("Settings"),
                                 show_playback_settings_p1_cb, oxine);
        if (playback_menu_focus_buttons == B_FOCUS_PARAM)
            otk_widget_set_focus (b);
        y += 40;
    }

    y += 10;
    if (odk_current_is_dvd (oxine->odk)
        || odk_current_is_vcd (oxine->odk)
        || odk_current_is_v4l (oxine->odk)
        || odk_current_is_dvb (oxine->odk)) {
        b = otk_text_button_new (oxine->otk, x, y, 180, 35, _("Mainmenu"),
                                 show_main_menu_cb, oxine);
        y += 40;
    } else {
        if (oxine->current_playlist == oxine->rw_playlist) {
            b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                     _("Playlist"), show_playlist_menu_cb,
                                     oxine);
            y += 40;
        } else {
            b = otk_text_button_new (oxine->otk, x, y, 180, 35,
                                     _("Media Browser"),
                                     show_filelist_menu_cb, oxine);
            y += 40;
        }
    }

    if (odk_current_is_dvb (oxine->odk)) {
        show_playback_menu_file_list (oxine);
    } else {
        show_playback_menu_file_info (oxine);
    }

    y += 10;
    if (!odk_current_is_dvd (oxine->odk)
        && !odk_current_is_vcd (oxine->odk)
        && !odk_current_is_v4l (oxine->odk)
        && !odk_current_is_dvb (oxine->odk)
        && !odk_current_is_cdda (oxine->odk)
        && odk_current_get_mrl (oxine->odk)) {
        char *image_dir = get_dirname (odk_current_get_mrl (oxine->odk));
        char *image_mrl = get_thumbnail (image_dir);

        int hi = 515 - y;
        int yi = y + (hi / 2);

        if (image_mrl && (access (image_mrl, R_OK) == 0)) {
            odk_osd_show_image (oxine->odk, image_mrl,
                                110, yi, 180, hi,
                                ODK_ALIGN_CENTER | ODK_ALIGN_VCENTER,
                                true, 0x000000);
        }

        ho_free (image_dir);
        ho_free (image_mrl);
    }

    show_user_interface (oxine);
}

void
show_playback_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    show_playback_menu_gui (oxine);
}

void
show_playback_controls_cb (void *oxine_p)
{
    hide_user_interface (oxine);

    create_new_window (false, false);

    playback_menu_focus_buttons = B_FOCUS_PLAY;
    show_controls (oxine, false);

    oxine->playback_controls_are_visible = true;
    oxine->current_menu = show_playback_controls_cb;

    show_user_interface (oxine);

    cancel_job (oxine->otk_clear_job);
    oxine->otk_clear_job = schedule_job (6000, (void (*)(void *))
                                         hide_user_interface, oxine);
}


/*
 * **************************************************************************
 * GUI-Layout of the info menu
 * **************************************************************************
 */
void
show_playback_info_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    hide_user_interface (oxine);

    int alignleft = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;
    int alignright = OTK_ALIGN_RIGHT | OTK_ALIGN_TOP;
    int aligncenter = OTK_ALIGN_CENTER | OTK_ALIGN_TOP;

    create_new_window (false, false);

    static int current_year = 0;
    if (!current_year) {
        time_t current = time (NULL);
        struct tm *brokentime = localtime (&current);
        current_year = brokentime->tm_year + 1900;
    }

    char *title = odk_get_meta_info (oxine->odk, META_INFO_TITLE);
    char *artist = odk_get_meta_info (oxine->odk, META_INFO_ARTIST);
    char *album = odk_get_meta_info (oxine->odk, META_INFO_ALBUM);
    char *year = odk_get_meta_info (oxine->odk, META_INFO_YEAR);
    char *genre = odk_get_meta_info (oxine->odk, META_INFO_GENRE);
    char *track = odk_get_meta_info (oxine->odk, XINE_META_INFO_TRACK_NUMBER);

    if (!title)
        title = create_title (odk_current_get_mrl (oxine->odk));

    int x = 400;
    int y = 180;
    int w1 = x - 40;
    int w2 = 780 - x;

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, x, y, 760, aligncenter, title);
    otk_widget_set_font (l, "sans", 40);
    y += 50;

    if (album && strlen (album) > 0) {
        otk_label_new (oxine->otk, x - 20, y, w1, alignright, _("Album:"));
        otk_label_new (oxine->otk, x, y, w2, alignleft, album);
        y += 40;
    }

    if (artist && strlen (artist) > 0) {
        otk_label_new (oxine->otk, x - 20, y, w1, alignright, _("Artist:"));
        otk_label_new (oxine->otk, x, y, w2, alignleft, artist);
        y += 40;
    }

    if (year && atoi (year) > 0 && atoi (year) <= current_year) {
        otk_label_new (oxine->otk, x - 20, y, w1, alignright, _("Year:"));
        otk_label_new (oxine->otk, x, y, w2, alignleft, year);
        y += 40;
    }

    if (track && strlen (track) > 0) {
        otk_label_new (oxine->otk, x - 20, y, w1, alignright, _("Track:"));
        otk_label_new (oxine->otk, x, y, w2, alignleft, track);
        y += 40;
    }

    if (genre && strlen (genre) > 0) {
        genre[0] = toupper (genre[0]);
        otk_label_new (oxine->otk, x - 20, y, w1, alignright, _("Genre:"));
        otk_label_new (oxine->otk, x, y, w2, alignleft, genre);
        y += 40;
    }

    ho_free (title);
    ho_free (artist);
    ho_free (album);
    ho_free (year);
    ho_free (genre);
    ho_free (track);

    oxine->current_menu = show_playback_info_cb;
    otk_draw (oxine->otk);

    cancel_job (oxine->otk_clear_job);
    oxine->otk_clear_job = schedule_job (10000, (void (*)(void *))
                                         hide_user_interface, oxine);

}
