
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist_m3u.c 1250 2006-10-06 07:52:11Z mschwerin $
 *
 * For information on playlist formats have a look at
 * http://gonze.com/playlists/playlist-format-survey.html
 *
 */
#include "config.h"

#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "playlist.h"
#include "playlist_m3u.h"

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>


/* 
 * ***************************************************************************
 * Name:            trim_whitespace
 * Access:          public
 *
 * Description:     Trims whitespace off the start and end of the string.
 * ***************************************************************************
 */
char *
trim_whitespace (char *str)
{
    int p = strlen (str) - 1;
    while (p >= 0
           && (str[p] == '\n' || str[p] == '\r' || str[p] == '\t'
               || str[p] == ' '))
        str[p--] = 0;

    p = 0;
    while (p <= strlen (str)
           && (str[p] == '\n' || str[p] == '\r' || str[p] == '\t'
               || str[p] == ' ')) {
        p++;
    }

    return (char *) (str + p);
}


bool
playlist_m3u_read (filelist_t * filelist)
{
    int size = 0;
    char *buf = read_entire_file (filelist->mrl, &size);

    if (!buf)
        return false;

    char *line = strtok (buf, "\n");
    while (line) {
        line = trim_whitespace (line);

        // if its not a comment or empty, add it
        if (strlen (line) > 0 && strstr (line, "#") != line) {
            char *mrl = line;
            char *title = create_title (mrl);
            filelist_add (filelist, title, mrl, FILE_TYPE_REGULAR);
            ho_free (title);
        }

        line = strtok (NULL, "\n");
    }

    ho_free (buf);

    return true;
}


bool
playlist_m3u_save (playlist_t * playlist, const char *mrl)
{
    FILE *f = fopen (mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return false;
    }

    playitem_t *playitem = playlist_first (playlist);
    while (playitem) {
        fprintf (f, "%s\n", playitem->mrl);
        playitem = playlist_next (playlist, playitem);
    }
    fclose (f);

    return true;
}


bool
playlist_m3u_load (playlist_t * playlist, const char *mrl)
{
    int size = 0;
    char *buf = read_entire_file (mrl, &size);
    if (!buf)
        return false;

    char *saveptr;
    char *line = strtok_r (buf, "\n", &saveptr);
    while (line) {
        line = trim_whitespace (line);

        // if its not a comment or empty, add it
        if (strlen (line) > 0 && strstr (line, "#") != line) {
            char *mrl = line;
            char *title = create_title (mrl);
            playlist_add (playlist, title, mrl);
            ho_free (title);
        }

        line = strtok_r (NULL, "\n", &saveptr);
    }

    ho_free (buf);

    return true;
}
